/*
 This file is part of GNU Taler
 (C) 2022 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import { Amounts, ExchangeTosStatus } from "@gnu-taler/taler-util";
import { Fragment, h, VNode } from "preact";
import { useState } from "preact/hooks";
import { Amount } from "../../components/Amount.js";
import { Part } from "../../components/Part.js";
import { QR } from "../../components/QR.js";
import { SelectList } from "../../components/SelectList.js";
import { Input, LinkSuccess, SvgIcon } from "../../components/styled/index.js";
import { TermsOfService } from "../../components/TermsOfService/index.js";
import { useTranslationContext } from "../../context/translation.js";
import { Button } from "../../mui/Button.js";
import editIcon from "../../svg/edit_24px.svg";
import {
  ExchangeDetails,
  getAmountWithFee,
  WithdrawDetails,
} from "../../wallet/Transaction.js";
import { State } from "./index.js";

export function SuccessView(state: State.Success): VNode {
  const { i18n } = useTranslationContext();
  const currentTosVersionIsAccepted =
    state.currentExchange.tosStatus === ExchangeTosStatus.Accepted;
  return (
    <Fragment>
      <section style={{ textAlign: "left" }}>
        <Part
          title={
            <div
              style={{
                display: "flex",
                alignItems: "center",
              }}
            >
              <i18n.Translate>Exchange</i18n.Translate>
              <Button onClick={state.doSelectExchange.onClick} variant="text">
                <SvgIcon
                  title="Edit"
                  dangerouslySetInnerHTML={{ __html: editIcon }}
                  color="black"
                />
              </Button>
            </div>
          }
          text={
            <ExchangeDetails exchange={state.currentExchange.exchangeBaseUrl} />
          }
          kind="neutral"
          big
        />
        <Part
          title={i18n.str`Details`}
          text={
            <WithdrawDetails
              amount={getAmountWithFee(
                state.toBeReceived,
                state.chosenAmount,
                "credit",
              )}
            />
          }
        />
        {state.ageRestriction && (
          <Input>
            <SelectList
              label={i18n.str`Age restriction`}
              list={state.ageRestriction.list}
              name="age"
              value={state.ageRestriction.value}
              onChange={state.ageRestriction.onChange}
            />
          </Input>
        )}
      </section>

      <section>
        {currentTosVersionIsAccepted ? (
          <Button
            variant="contained"
            color="success"
            disabled={!state.doWithdrawal.onClick}
            onClick={state.doWithdrawal.onClick}
          >
            <i18n.Translate>
              Withdraw &nbsp; <Amount value={state.toBeReceived} />
            </i18n.Translate>
          </Button>
        ) : (
          <TermsOfService
            exchangeUrl={state.currentExchange.exchangeBaseUrl}
            onChange={state.onTosUpdate}
          />
        )}
      </section>
      {state.talerWithdrawUri ? (
        <WithdrawWithMobile talerWithdrawUri={state.talerWithdrawUri} />
      ) : undefined}
    </Fragment>
  );
}

function WithdrawWithMobile({
  talerWithdrawUri,
}: {
  talerWithdrawUri: string;
}): VNode {
  const { i18n } = useTranslationContext();
  const [showQR, setShowQR] = useState<boolean>(false);

  return (
    <section>
      <LinkSuccess upperCased onClick={() => setShowQR((qr) => !qr)}>
        {!showQR ? i18n.str`Withdraw to a mobile phone` : i18n.str`Hide QR`}
      </LinkSuccess>
      {showQR && (
        <div>
          <QR text={talerWithdrawUri} />
          <i18n.Translate>
            Scan the QR code or &nbsp;
            <a href={talerWithdrawUri}>
              <i18n.Translate>click here</i18n.Translate>
            </a>
          </i18n.Translate>
        </div>
      )}
    </section>
  );
}
