/*
 This file is part of GNU Taler
 (C) 2022 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import { parseRecoveryUri } from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { useAlertContext } from "../../context/alert.js";
import { useBackendContext } from "../../context/backend.js";
import { useTranslationContext } from "../../context/translation.js";
import { Props, State } from "./index.js";

export function useComponentState({
  talerRecoveryUri,
  onCancel,
  onSuccess,
}: Props): State {
  const api = useBackendContext();
  const { pushAlertOnError } = useAlertContext();
  const { i18n } = useTranslationContext();
  if (!talerRecoveryUri) {
    return {
      status: "error",
      error: {
        type: "error",
        message: i18n.str`Missing URI`,
        description: i18n.str``,
        cause: new Error("something"),
        context: {},
      },
    };
  }
  const info = parseRecoveryUri(talerRecoveryUri);

  if (!info) {
    return {
      status: "error",
      error: {
        type: "error",
        message: i18n.str`Could not parse the recovery URI`,
        description: i18n.str``,
        cause: new Error("something"),
        context: {},
      },
    };
  }
  const recovery = info;

  async function recoverBackup(): Promise<void> {
    await api.wallet.call(WalletApiOperation.ImportBackupRecovery, {
      recovery,
    });
    onSuccess();
  }

  return {
    status: "ready",

    accept: {
      onClick: pushAlertOnError(recoverBackup),
    },
    cancel: {
      onClick: pushAlertOnError(onCancel),
    },
    error: undefined,
  };
}
