/*
 This file is part of GNU Taler
 (C) 2022 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import { ComponentChildren, Fragment, h, VNode } from "preact";
import { useState } from "preact/hooks";
import { useTranslationContext } from "../../../web-util/src/index.browser.js";
import {
  Alert as AlertNotification,
  useAlertContext,
} from "../context/alert.js";
import { Alert } from "../mui/Alert.js";

/**
 *
 * @author sebasjm
 */

function AlertContext({
  context,
  cause,
}: {
  cause: unknown;
  context: undefined | object;
}): VNode {
  const [more, setMore] = useState(false);
  const [wrap, setWrap] = useState(false);
  const { i18n } = useTranslationContext();
  if (!more) {
    return (
      <div style={{ display: "flex", justifyContent: "right" }}>
        <a
          onClick={() => setMore(true)}
          style={{ cursor: "pointer", textDecoration: "underline" }}
        >
          <i18n.Translate>more info</i18n.Translate>
        </a>
      </div>
    );
  }
  const errorInfo = JSON.stringify(
    context === undefined ? { cause } : { context, cause },
    undefined,
    2,
  );
  return (
    <Fragment>
      <div style={{ display: "flex", justifyContent: "right" }}>
        <a
          onClick={() => setWrap(!wrap)}
          style={{ cursor: "pointer", textDecoration: "underline" }}
        >
          <i18n.Translate>wrap text</i18n.Translate>
        </a>
        &nbsp;&nbsp;
        <a
          onClick={() => navigator.clipboard.writeText(errorInfo)}
          style={{ cursor: "pointer", textDecoration: "underline" }}
        >
          <i18n.Translate>copy content</i18n.Translate>
        </a>
        &nbsp;&nbsp;
        <a
          onClick={() => setMore(false)}
          style={{ cursor: "pointer", textDecoration: "underline" }}
        >
          <i18n.Translate>less info</i18n.Translate>
        </a>
      </div>
      <pre
        style={
          wrap
            ? {
                whiteSpace: "pre-wrap",
                overflowWrap: "anywhere",
              }
            : {
                overflow: "overlay",
              }
        }
      >
        {errorInfo}
      </pre>
    </Fragment>
  );
}

export function ErrorAlertView({
  error,
  onClose,
}: {
  error: AlertNotification;
  onClose?: () => Promise<void>;
}): VNode {
  return (
    <Wrapper>
      <AlertView alert={error} onClose={onClose} />
    </Wrapper>
  );
}

export function AlertView({
  alert,
  onClose,
}: {
  alert: AlertNotification;
  onClose?: () => Promise<void>;
}): VNode {
  return (
    <Alert title={alert.message} severity={alert.type} onClose={onClose}>
      <div style={{ display: "flex", flexDirection: "column" }}>
        <div>{alert.description}</div>
        {alert.type === "error" ? (
          <AlertContext context={alert.context} cause={alert.cause} />
        ) : undefined}
      </div>
    </Alert>
  );
}

export function CurrentAlerts(): VNode {
  const { alerts, removeAlert } = useAlertContext();
  if (alerts.length === 0) return <Fragment />;
  return (
    <Wrapper>
      {alerts.map((n, i) => (
        <AlertView key={i} alert={n} onClose={async () => removeAlert(n)} />
      ))}
    </Wrapper>
  );
}

function Wrapper({ children }: { children: ComponentChildren }): VNode {
  return <div style={{ margin: "1em" }}>{children}</div>;
}
