/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  GlobalTestState,
  WalletCli,
  setupDb,
  ExchangeService,
  BankService,
} from "../harness/harness.js";
import {
  BankAccessApi,
  BankApi,
  WalletApiOperation,
} from "@gnu-taler/taler-wallet-core";
import { CoinConfig } from "../harness/denomStructures.js";
import { j2s, URL } from "@gnu-taler/taler-util";
import { withdrawViaBank } from "../harness/helpers.js";

const coinRsaCommon = {
  cipher: "RSA" as const,
  durationLegal: "3 years",
  durationSpend: "2 years",
  durationWithdraw: "7 days",
  rsaKeySize: 1024,
};

const coin_u1 = (curr: string): CoinConfig => ({
  ...coinRsaCommon,
  name: `${curr}_u1`,
  value: `${curr}:1`,
  feeDeposit: `${curr}:0`,
  feeRefresh: `${curr}:0`,
  feeRefund: `${curr}:0`,
  feeWithdraw: `${curr}:1`,
});

const coin_u5 = (curr: string): CoinConfig => ({
  ...coinRsaCommon,
  name: `${curr}_u5`,
  value: `${curr}:5`,
  feeDeposit: `${curr}:0`,
  feeRefresh: `${curr}:0`,
  feeRefund: `${curr}:0`,
  feeWithdraw: `${curr}:1`,
});

export const weirdCoinConfig = [coin_u1, coin_u5];

/**
 * Test withdrawal with a weird denomination structure to
 * make sure fees are computed as expected.
 */
export async function runWithdrawalFeesTest(t: GlobalTestState) {
  // Set up test environment

  const db = await setupDb(t);

  const bank = await BankService.create(t, {
    allowRegistrations: true,
    currency: "TESTKUDOS",
    database: db.connStr,
    httpPort: 8082,
  });

  const exchange = ExchangeService.create(t, {
    name: "testexchange-1",
    currency: "TESTKUDOS",
    httpPort: 8081,
    database: db.connStr,
  });

  const exchangeBankAccount = await bank.createExchangeAccount(
    "myexchange",
    "x",
  );
  await exchange.addBankAccount("1", exchangeBankAccount);

  await bank.start();

  await bank.pingUntilAvailable();

  const coinConfig: CoinConfig[] = weirdCoinConfig.map((x) => x("TESTKUDOS"));
  exchange.addCoinConfigList(coinConfig);

  await exchange.start();
  await exchange.pingUntilAvailable();

  console.log("setup done!");

  const wallet = new WalletCli(t);

  await wallet.client.call(WalletApiOperation.AddExchange, {
    exchangeBaseUrl: exchange.baseUrl,
  });

  const amount = "TESTKUDOS:7.5";

  const user = await BankApi.createRandomBankUser(bank);
  const wop = await BankAccessApi.createWithdrawalOperation(bank, user, amount);

  // Hand it to the wallet

  const details = await wallet.client.call(
    WalletApiOperation.GetWithdrawalDetailsForUri,
    {
      talerWithdrawUri: wop.taler_withdraw_uri,
    },
  );

  console.log(j2s(details));

  const amountDetails = await wallet.client.call(
    WalletApiOperation.GetWithdrawalDetailsForAmount,
    {
      amount: details.amount,
      exchangeBaseUrl: details.possibleExchanges[0].exchangeBaseUrl,
    },
  );

  console.log(j2s(amountDetails));

  t.assertAmountEquals(amountDetails.amountEffective, "TESTKUDOS:5");
  t.assertAmountEquals(amountDetails.amountRaw, "TESTKUDOS:7.5");

  await wallet.runPending();

  // Withdraw (AKA select)

  await wallet.client.call(WalletApiOperation.AcceptBankIntegratedWithdrawal, {
    exchangeBaseUrl: exchange.baseUrl,
    talerWithdrawUri: wop.taler_withdraw_uri,
  });

  // Confirm it

  await BankApi.confirmWithdrawalOperation(bank, user, wop);

  await exchange.runWirewatchOnce();

  await wallet.runUntilDone();

  // Check balance

  const balResp = await wallet.client.call(WalletApiOperation.GetBalances, {});
  console.log(j2s(balResp));

  t.assertAmountEquals(balResp.balances[0].available, "TESTKUDOS:5");

  const txns = await wallet.client.call(WalletApiOperation.GetTransactions, {});
  console.log(j2s(txns));
}

runWithdrawalFeesTest.suites = ["wallet"];
