/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import { Duration } from "@gnu-taler/taler-util";
import { CoinConfig, defaultCoinConfig } from "../harness/denomStructures.js";
import {
  BankService,
  ExchangeService,
  getPayto,
  GlobalTestState,
  MerchantService,
  setupDb,
  WalletCli,
} from "../harness/harness.js";
import {
  withdrawViaBank,
  makeTestPayment,
  EnvOptions,
  SimpleTestEnvironment,
} from "../harness/helpers.js";

export async function createKycTestkudosEnvironment(
  t: GlobalTestState,
  coinConfig: CoinConfig[] = defaultCoinConfig.map((x) => x("TESTKUDOS")),
  opts: EnvOptions = {},
): Promise<SimpleTestEnvironment> {
  const db = await setupDb(t);

  const bank = await BankService.create(t, {
    allowRegistrations: true,
    currency: "TESTKUDOS",
    database: db.connStr,
    httpPort: 8082,
  });

  const exchange = ExchangeService.create(t, {
    name: "testexchange-1",
    currency: "TESTKUDOS",
    httpPort: 8081,
    database: db.connStr,
  });

  const merchant = await MerchantService.create(t, {
    name: "testmerchant-1",
    currency: "TESTKUDOS",
    httpPort: 8083,
    database: db.connStr,
  });

  const exchangeBankAccount = await bank.createExchangeAccount(
    "myexchange",
    "x",
  );
  exchange.addBankAccount("1", exchangeBankAccount);

  bank.setSuggestedExchange(exchange, exchangeBankAccount.accountPaytoUri);

  await bank.start();

  await bank.pingUntilAvailable();

  const ageMaskSpec = opts.ageMaskSpec;

  if (ageMaskSpec) {
    exchange.enableAgeRestrictions(ageMaskSpec);
    // Enable age restriction for all coins.
    exchange.addCoinConfigList(
      coinConfig.map((x) => ({
        ...x,
        name: `${x.name}-age`,
        ageRestricted: true,
      })),
    );
    // For mixed age restrictions, we also offer coins without age restrictions
    if (opts.mixedAgeRestriction) {
      exchange.addCoinConfigList(
        coinConfig.map((x) => ({ ...x, ageRestricted: false })),
      );
    }
  } else {
    exchange.addCoinConfigList(coinConfig);
  }

  await exchange.modifyConfig(async (config) => {
    const myprov = "kyc-provider-myprov";
    config.setString(myprov, "cost", "0");
    config.setString(myprov, "logic", "oauth2");
    config.setString(myprov, "provided_checks", "dummy1");
    config.setString(myprov, "user_type", "individual");
    config.setString(myprov, "kyc_oauth2_validity", "forever");
    config.setString(
      myprov,
      "kyc_oauth2_auth_url",
      "http://localhost:6666/oauth/v2/token",
    );
    config.setString(
      myprov,
      "kyc_oauth2_login_url",
      "http://localhost:6666/oauth/v2/login",
    );
    config.setString(
      myprov,
      "kyc_oauth2_info_url",
      "http://localhost:6666/oauth/v2/login",
    );
    config.setString(myprov, "kyc_oauth2_client_id", "taler-exchange");
    config.setString(myprov, "kyc_oauth2_client_secret", "exchange-secret");
    config.setString(myprov, "kyc_oauth2_post_url", "https://taler.com");

    config.setString(
      "kyc-legitimization-withdraw1",
      "operation_type",
      "withdraw",
    );
    config.setString(
      "kyc-legitimization-withdraw1",
      "required_checks",
      "dummy1",
    );
    config.setString("kyc-legitimization-withdraw1", "timeframe", "1d");
    config.setString(
      "kyc-legitimization-withdraw1",
      "threshold",
      "TESTKUDOS:5",
    );
  });

  await exchange.start();
  await exchange.pingUntilAvailable();

  merchant.addExchange(exchange);

  await merchant.start();
  await merchant.pingUntilAvailable();

  await merchant.addInstance({
    id: "default",
    name: "Default Instance",
    paytoUris: [getPayto("merchant-default")],
    defaultWireTransferDelay: Duration.toTalerProtocolDuration(
      Duration.fromSpec({ minutes: 1 }),
    ),
  });

  await merchant.addInstance({
    id: "minst1",
    name: "minst1",
    paytoUris: [getPayto("minst1")],
    defaultWireTransferDelay: Duration.toTalerProtocolDuration(
      Duration.fromSpec({ minutes: 1 }),
    ),
  });

  console.log("setup done!");

  const wallet = new WalletCli(t);

  return {
    commonDb: db,
    exchange,
    merchant,
    wallet,
    bank,
    exchangeBankAccount,
  };
}

export async function runKycTest(t: GlobalTestState) {
  // Set up test environment

  const { wallet, bank, exchange, merchant } =
    await createKycTestkudosEnvironment(t);

  // Withdraw digital cash into the wallet.

  await withdrawViaBank(t, { wallet, bank, exchange, amount: "TESTKUDOS:20" });

  const order = {
    summary: "Buy me!",
    amount: "TESTKUDOS:5",
    fulfillment_url: "taler://fulfillment-success/thx",
  };

  await makeTestPayment(t, { wallet, merchant, order });
  await wallet.runUntilDone();
}

runKycTest.suites = ["wallet"];
// See bugs.taler.net/n/7599
runKycTest.experimental = true;
