/*
 This file is part of GNU Taler
 (C) 2021-2023 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 *
 * @author Sebastian Javier Marchano (sebasjm)
 */

import { useTranslationContext } from "@gnu-taler/web-util/lib/index.browser";
import { Fragment, h, VNode } from "preact";
import { useState } from "preact/hooks";
import { Loading } from "../../../../components/exception/loading.js";
import { NotificationCard } from "../../../../components/menu/index.js";
import { MerchantBackend } from "../../../../declaration.js";
import { HttpError } from "../../../../utils/request.js";
import {
  useInstanceTemplates,
  useTemplateAPI,
} from "../../../../hooks/templates.js";
import { Notification } from "../../../../utils/types.js";
import { ListPage } from "./ListPage.js";

interface Props {
  onUnauthorized: () => VNode;
  onLoadError: (error: HttpError) => VNode;
  onNotFound: () => VNode;
  onCreate: () => void;
  onSelect: (id: string) => void;
}

export default function ListTemplates({
  onUnauthorized,
  onLoadError,
  onCreate,
  onSelect,
  onNotFound,
}: Props): VNode {
  const [position, setPosition] = useState<string | undefined>(undefined);
  const { i18n } = useTranslationContext();
  const [notif, setNotif] = useState<Notification | undefined>(undefined);
  const { deleteTemplate } = useTemplateAPI();
  const result = useInstanceTemplates({ position }, (id) => setPosition(id));

  if (result.clientError && result.isUnauthorized) return onUnauthorized();
  if (result.clientError && result.isNotfound) return onNotFound();
  if (result.loading) return <Loading />;
  if (!result.ok) return onLoadError(result);

  return (
    <Fragment>
      <NotificationCard notification={notif} />

      <ListPage
        templates={result.data.templates}
        onLoadMoreBefore={
          result.isReachingStart ? result.loadMorePrev : undefined
        }
        onLoadMoreAfter={result.isReachingEnd ? result.loadMore : undefined}
        onCreate={onCreate}
        onSelect={(e) => {
          onSelect(e.template_id);
        }}
        onDelete={(e: MerchantBackend.Template.TemplateEntry) =>
          deleteTemplate(e.template_id)
            .then(() =>
              setNotif({
                message: i18n.str`template delete successfully`,
                type: "SUCCESS",
              }),
            )
            .catch((error) =>
              setNotif({
                message: i18n.str`could not delete the template`,
                type: "ERROR",
                description: error.message,
              }),
            )
        }
      />
    </Fragment>
  );
}
