/*
 This file is part of GNU Taler
 (C) 2021-2023 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 *
 * @author Sebastian Javier Marchano (sebasjm)
 */

import { tests } from "@gnu-taler/web-util/lib/index.browser";
import { expect } from "chai";
import { MerchantBackend } from "../declaration.js";
import { API_INFORM_TRANSFERS, API_LIST_TRANSFERS } from "./urls.js";
import { ApiMockEnvironment } from "./testing.js";
import { useInstanceTransfers, useTransferAPI } from "./transfer.js";

describe("transfer api interaction with listing", () => {
  it("should evict cache when informing a transfer", async () => {
    const env = new ApiMockEnvironment();

    env.addRequestExpectation(API_LIST_TRANSFERS, {
      qparam: { limit: 0 },
      response: {
        transfers: [{ wtid: "2" } as MerchantBackend.Transfers.TransferDetails],
      },
    });
    // FIXME: is this query really needed? if the hook is rendered without
    // position argument then then backend is returning the newest and no need
    // to this second query
    env.addRequestExpectation(API_LIST_TRANSFERS, {
      qparam: { limit: -20 },
      response: {
        transfers: [],
      },
    });

    const moveCursor = (d: string) => {
      console.log("new position", d);
    };

    const hookBehavior = await tests.hookBehaveLikeThis(
      () => {
        const query = useInstanceTransfers({}, moveCursor);
        const api = useTransferAPI();
        return { query, api };
      },
      {},
      [
        ({ query, api }) => {
          expect(query.loading).true;
        },

        ({ query, api }) => {
          expect(env.assertJustExpectedRequestWereMade()).deep.eq({
            result: "ok",
          });
          expect(query.loading).undefined;
          expect(query.ok).true;
          if (!query.ok) return;
          expect(query.data).deep.equals({
            transfers: [{ wtid: "2" }],
          });

          env.addRequestExpectation(API_INFORM_TRANSFERS, {
            request: {
              wtid: "3",
              credit_amount: "EUR:1",
              exchange_url: "exchange.url",
              payto_uri: "payto://",
            },
            response: { total: "" } as any,
          });

          env.addRequestExpectation(API_LIST_TRANSFERS, {
            qparam: { limit: 0 },
            response: {
              transfers: [{ wtid: "2" } as any, { wtid: "3" } as any],
            },
          });

          env.addRequestExpectation(API_LIST_TRANSFERS, {
            qparam: { limit: -20 },
            response: {
              transfers: [],
            },
          });

          api.informTransfer({
            wtid: "3",
            credit_amount: "EUR:1",
            exchange_url: "exchange.url",
            payto_uri: "payto://",
          });
        },
        ({ query, api }) => {
          expect(env.assertJustExpectedRequestWereMade()).deep.eq({
            result: "ok",
          });
          expect(query.loading).undefined;
          expect(query.ok).true;
          if (!query.ok) return;

          expect(query.data).deep.equals({
            transfers: [{ wtid: "3" }, { wtid: "2" }],
          });
        },
      ],
      env.buildTestingContext(),
    );

    expect(hookBehavior).deep.eq({ result: "ok" });
    expect(env.assertJustExpectedRequestWereMade()).deep.eq({ result: "ok" });
  });
});

describe("transfer listing pagination", () => {
  it("should not load more if has reach the end", async () => {
    const env = new ApiMockEnvironment();

    env.addRequestExpectation(API_LIST_TRANSFERS, {
      qparam: { limit: 0, payto_uri: "payto://" },
      response: {
        transfers: [{ wtid: "2" } as any],
      },
    });

    env.addRequestExpectation(API_LIST_TRANSFERS, {
      qparam: { limit: -20, payto_uri: "payto://" },
      response: {
        transfers: [{ wtid: "1" } as any],
      },
    });

    const moveCursor = (d: string) => {
      console.log("new position", d);
    };
    const hookBehavior = await tests.hookBehaveLikeThis(
      () => {
        return useInstanceTransfers({ payto_uri: "payto://" }, moveCursor);
      },
      {},
      [
        (query) => {
          expect(env.assertJustExpectedRequestWereMade()).deep.eq({
            result: "ok",
          });
          expect(query.loading).true;
        },
        (query) => {
          expect(query.loading).undefined;
          expect(query.ok).true;
          if (!query.ok) return;
          expect(query.data).deep.equals({
            transfers: [{ wtid: "2" }, { wtid: "1" }],
          });
          expect(query.isReachingEnd).true;
          expect(query.isReachingStart).true;

          //check that this button won't trigger more updates since
          //has reach end and start
          query.loadMore();
          query.loadMorePrev();
        },
      ],
      env.buildTestingContext(),
    );

    expect(env.assertJustExpectedRequestWereMade()).deep.eq({ result: "ok" });
    expect(hookBehavior).deep.eq({ result: "ok" });
  });

  it("should load more if result brings more that PAGE_SIZE", async () => {
    const env = new ApiMockEnvironment();

    const transfersFrom0to20 = Array.from({ length: 20 }).map((e, i) => ({
      wtid: String(i),
    }));
    const transfersFrom20to40 = Array.from({ length: 20 }).map((e, i) => ({
      wtid: String(i + 20),
    }));
    const transfersFrom20to0 = [...transfersFrom0to20].reverse();

    env.addRequestExpectation(API_LIST_TRANSFERS, {
      qparam: { limit: 20, payto_uri: "payto://", offset: "1" },
      response: {
        transfers: transfersFrom0to20,
      },
    });

    env.addRequestExpectation(API_LIST_TRANSFERS, {
      qparam: { limit: -20, payto_uri: "payto://", offset: "1" },
      response: {
        transfers: transfersFrom20to40,
      },
    });

    const moveCursor = (d: string) => {
      console.log("new position", d);
    };

    const hookBehavior = await tests.hookBehaveLikeThis(
      () => {
        return useInstanceTransfers(
          { payto_uri: "payto://", position: "1" },
          moveCursor,
        );
      },
      {},
      [
        (result) => {
          expect(env.assertJustExpectedRequestWereMade()).deep.eq({
            result: "ok",
          });
          expect(result.loading).true;
        },
        (result) => {
          expect(result.loading).undefined;
          expect(result.ok).true;
          if (!result.ok) return;
          expect(result.data).deep.equals({
            transfers: [...transfersFrom20to0, ...transfersFrom20to40],
          });
          expect(result.isReachingEnd).false;
          expect(result.isReachingStart).false;

          //query more
          env.addRequestExpectation(API_LIST_TRANSFERS, {
            qparam: { limit: -40, payto_uri: "payto://", offset: "1" },
            response: {
              transfers: [...transfersFrom20to40, { wtid: "41" }],
            },
          });
          result.loadMore();
        },
        (result) => {
          expect(env.assertJustExpectedRequestWereMade()).deep.eq({
            result: "ok",
          });
          expect(result.loading).true;
        },
        (result) => {
          expect(env.assertJustExpectedRequestWereMade()).deep.eq({
            result: "ok",
          });
          expect(result.loading).undefined;
          expect(result.ok).true;
          if (!result.ok) return;
          expect(result.data).deep.equals({
            transfers: [
              ...transfersFrom20to0,
              ...transfersFrom20to40,
              { wtid: "41" },
            ],
          });
          expect(result.isReachingEnd).true;
          expect(result.isReachingStart).false;
        },
      ],
      env.buildTestingContext(),
    );

    expect(env.assertJustExpectedRequestWereMade()).deep.eq({ result: "ok" });
    expect(hookBehavior).deep.eq({ result: "ok" });
  });
});
