/*
 This file is part of GNU Taler
 (C) 2021-2023 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 *
 * @author Sebastian Javier Marchano (sebasjm)
 */

import { expect } from "chai";
import { MerchantBackend } from "../declaration.js";
import {
  useInstanceReserves,
  useReserveDetails,
  useReservesAPI,
  useTipDetails,
} from "./reserves.js";
import { ApiMockEnvironment } from "./testing.js";
import {
  API_AUTHORIZE_TIP,
  API_AUTHORIZE_TIP_FOR_RESERVE,
  API_CREATE_RESERVE,
  API_DELETE_RESERVE,
  API_GET_RESERVE_BY_ID,
  API_GET_TIP_BY_ID,
  API_LIST_RESERVES,
} from "./urls.js";
import { tests } from "@gnu-taler/web-util/lib/index.browser";

describe("reserve api interaction with listing", () => {
  it("should evict cache when creating a reserve", async () => {
    const env = new ApiMockEnvironment();

    env.addRequestExpectation(API_LIST_RESERVES, {
      response: {
        reserves: [
          {
            reserve_pub: "11",
          } as MerchantBackend.Tips.ReserveStatusEntry,
        ],
      },
    });

    const hookBehavior = await tests.hookBehaveLikeThis(
      () => {
        const api = useReservesAPI();
        const query = useInstanceReserves();
        return { query, api };
      },
      {},
      [
        ({ query, api }) => {
          expect(query.loading).true;
        },
        ({ query, api }) => {
          expect(query.loading).false;
          expect(query.ok).true;
          if (!query.ok) return;
          expect(query.data).deep.equals({
            reserves: [{ reserve_pub: "11" }],
          });

          env.addRequestExpectation(API_CREATE_RESERVE, {
            request: {
              initial_balance: "ARS:3333",
              exchange_url: "http://url",
              wire_method: "iban",
            },
            response: {
              reserve_pub: "22",
              payto_uri: "payto",
            },
          });

          env.addRequestExpectation(API_LIST_RESERVES, {
            response: {
              reserves: [
                {
                  reserve_pub: "11",
                } as MerchantBackend.Tips.ReserveStatusEntry,
                {
                  reserve_pub: "22",
                } as MerchantBackend.Tips.ReserveStatusEntry,
              ],
            },
          });

          api.createReserve({
            initial_balance: "ARS:3333",
            exchange_url: "http://url",
            wire_method: "iban",
          });
        },
        ({ query, api }) => {
          expect(env.assertJustExpectedRequestWereMade()).deep.eq({
            result: "ok",
          });
          expect(query.loading).false;
          expect(query.ok).true;
          if (!query.ok) return;

          expect(query.data).deep.equals({
            reserves: [
              {
                reserve_pub: "11",
              } as MerchantBackend.Tips.ReserveStatusEntry,
              {
                reserve_pub: "22",
              } as MerchantBackend.Tips.ReserveStatusEntry,
            ],
          });
        },
      ],
      env.buildTestingContext(),
    );

    expect(hookBehavior).deep.eq({ result: "ok" });
    expect(env.assertJustExpectedRequestWereMade()).deep.eq({ result: "ok" });
  });

  it("should evict cache when deleting a reserve", async () => {
    const env = new ApiMockEnvironment();

    env.addRequestExpectation(API_LIST_RESERVES, {
      response: {
        reserves: [
          {
            reserve_pub: "11",
          } as MerchantBackend.Tips.ReserveStatusEntry,
          {
            reserve_pub: "22",
          } as MerchantBackend.Tips.ReserveStatusEntry,
          {
            reserve_pub: "33",
          } as MerchantBackend.Tips.ReserveStatusEntry,
        ],
      },
    });

    const hookBehavior = await tests.hookBehaveLikeThis(
      () => {
        const api = useReservesAPI();
        const query = useInstanceReserves();
        return { query, api };
      },
      {},
      [
        ({ query, api }) => {
          expect(query.loading).true;
        },
        ({ query, api }) => {
          expect(env.assertJustExpectedRequestWereMade()).deep.eq({
            result: "ok",
          });

          expect(query.loading).false;
          expect(query.ok).true;
          if (!query.ok) return;
          expect(query.data).deep.equals({
            reserves: [
              { reserve_pub: "11" },
              { reserve_pub: "22" },
              { reserve_pub: "33" },
            ],
          });

          env.addRequestExpectation(API_DELETE_RESERVE("11"), {});
          env.addRequestExpectation(API_LIST_RESERVES, {
            response: {
              reserves: [
                {
                  reserve_pub: "22",
                } as MerchantBackend.Tips.ReserveStatusEntry,
                {
                  reserve_pub: "33",
                } as MerchantBackend.Tips.ReserveStatusEntry,
              ],
            },
          });

          api.deleteReserve("11");
        },
        ({ query, api }) => {
          expect(env.assertJustExpectedRequestWereMade()).deep.eq({
            result: "ok",
          });
          expect(query.loading).false;
          expect(query.ok).true;
          if (!query.ok) return;
          expect(query.data).deep.equals({
            reserves: [{ reserve_pub: "22" }, { reserve_pub: "33" }],
          });
        },
      ],
      env.buildTestingContext(),
    );

    expect(hookBehavior).deep.eq({ result: "ok" });
    expect(env.assertJustExpectedRequestWereMade()).deep.eq({ result: "ok" });
  });
});

describe("reserve api interaction with details", () => {
  it("should evict cache when adding a tip for a specific reserve", async () => {
    const env = new ApiMockEnvironment();

    env.addRequestExpectation(API_GET_RESERVE_BY_ID("11"), {
      response: {
        payto_uri: "payto://here",
        tips: [{ reason: "why?", tip_id: "id1", total_amount: "USD:10" }],
      } as MerchantBackend.Tips.ReserveDetail,
      qparam: {
        tips: "yes",
      },
    });

    const hookBehavior = await tests.hookBehaveLikeThis(
      () => {
        const api = useReservesAPI();
        const query = useReserveDetails("11");
        return { query, api };
      },
      {},
      [
        ({ query, api }) => {
          expect(query.loading).true;
        },
        ({ query, api }) => {
          expect(env.assertJustExpectedRequestWereMade()).deep.eq({
            result: "ok",
          });
          expect(query.loading).false;
          expect(query.ok).true;
          if (!query.ok) return;
          expect(query.data).deep.equals({
            payto_uri: "payto://here",
            tips: [{ reason: "why?", tip_id: "id1", total_amount: "USD:10" }],
          });

          env.addRequestExpectation(API_AUTHORIZE_TIP_FOR_RESERVE("11"), {
            request: {
              amount: "USD:12",
              justification: "not",
              next_url: "http://taler.net",
            },
            response: {
              tip_id: "id2",
              taler_tip_uri: "uri",
              tip_expiration: { t_s: 1 },
              tip_status_url: "url",
            },
          });

          env.addRequestExpectation(API_GET_RESERVE_BY_ID("11"), {
            response: {
              payto_uri: "payto://here",
              tips: [
                { reason: "why?", tip_id: "id1", total_amount: "USD:10" },
                { reason: "not", tip_id: "id2", total_amount: "USD:12" },
              ],
            } as MerchantBackend.Tips.ReserveDetail,
            qparam: {
              tips: "yes",
            },
          });

          api.authorizeTipReserve("11", {
            amount: "USD:12",
            justification: "not",
            next_url: "http://taler.net",
          });
        },
        ({ query, api }) => {
          expect(env.assertJustExpectedRequestWereMade()).deep.eq({
            result: "ok",
          });
          expect(query.loading).false;

          expect(query.loading).false;
          expect(query.ok).true;
          if (!query.ok) return;

          expect(query.data).deep.equals({
            payto_uri: "payto://here",
            tips: [
              { reason: "why?", tip_id: "id1", total_amount: "USD:10" },
              { reason: "not", tip_id: "id2", total_amount: "USD:12" },
            ],
          });
        },
      ],
      env.buildTestingContext(),
    );

    expect(hookBehavior).deep.eq({ result: "ok" });
    expect(env.assertJustExpectedRequestWereMade()).deep.eq({ result: "ok" });
  });

  it("should evict cache when adding a tip for a random reserve", async () => {
    const env = new ApiMockEnvironment();

    env.addRequestExpectation(API_GET_RESERVE_BY_ID("11"), {
      response: {
        payto_uri: "payto://here",
        tips: [{ reason: "why?", tip_id: "id1", total_amount: "USD:10" }],
      } as MerchantBackend.Tips.ReserveDetail,
      qparam: {
        tips: "yes",
      },
    });

    const hookBehavior = await tests.hookBehaveLikeThis(
      () => {
        const api = useReservesAPI();
        const query = useReserveDetails("11");
        return { query, api };
      },
      {},
      [
        ({ query, api }) => {
          expect(query.loading).true;
        },
        ({ query, api }) => {
          expect(env.assertJustExpectedRequestWereMade()).deep.eq({
            result: "ok",
          });
          expect(query.loading).false;
          expect(query.ok).true;
          if (!query.ok) return;
          expect(query.data).deep.equals({
            payto_uri: "payto://here",
            tips: [{ reason: "why?", tip_id: "id1", total_amount: "USD:10" }],
          });

          env.addRequestExpectation(API_AUTHORIZE_TIP, {
            request: {
              amount: "USD:12",
              justification: "not",
              next_url: "http://taler.net",
            },
            response: {
              tip_id: "id2",
              taler_tip_uri: "uri",
              tip_expiration: { t_s: 1 },
              tip_status_url: "url",
            },
          });

          env.addRequestExpectation(API_GET_RESERVE_BY_ID("11"), {
            response: {
              payto_uri: "payto://here",
              tips: [
                { reason: "why?", tip_id: "id1", total_amount: "USD:10" },
                { reason: "not", tip_id: "id2", total_amount: "USD:12" },
              ],
            } as MerchantBackend.Tips.ReserveDetail,
            qparam: {
              tips: "yes",
            },
          });

          api.authorizeTip({
            amount: "USD:12",
            justification: "not",
            next_url: "http://taler.net",
          });
        },
        ({ query, api }) => {
          expect(env.assertJustExpectedRequestWereMade()).deep.eq({
            result: "ok",
          });
          expect(query.loading).false;
          expect(query.ok).true;
          if (!query.ok) return;

          expect(query.data).deep.equals({
            payto_uri: "payto://here",
            tips: [
              { reason: "why?", tip_id: "id1", total_amount: "USD:10" },
              { reason: "not", tip_id: "id2", total_amount: "USD:12" },
            ],
          });
        },
      ],
      env.buildTestingContext(),
    );

    expect(hookBehavior).deep.eq({ result: "ok" });
    expect(env.assertJustExpectedRequestWereMade()).deep.eq({ result: "ok" });
  });
});

describe("reserve api interaction with tip details", () => {
  it("should list tips", async () => {
    const env = new ApiMockEnvironment();

    env.addRequestExpectation(API_GET_TIP_BY_ID("11"), {
      response: {
        total_picked_up: "USD:12",
        reason: "not",
      } as MerchantBackend.Tips.TipDetails,
      qparam: {
        pickups: "yes",
      },
    });

    const hookBehavior = await tests.hookBehaveLikeThis(
      () => {
        const query = useTipDetails("11");
        return { query };
      },
      {},
      [
        ({ query }) => {
          expect(env.assertJustExpectedRequestWereMade()).deep.eq({
            result: "ok",
          });
          expect(query.loading).true;
        },
        ({ query }) => {
          expect(query.loading).false;
          expect(query.ok).true;
          if (!query.ok) return;
          expect(query.data).deep.equals({
            total_picked_up: "USD:12",
            reason: "not",
          });
        },
      ],
      env.buildTestingContext(),
    );

    expect(hookBehavior).deep.eq({ result: "ok" });
    expect(env.assertJustExpectedRequestWereMade()).deep.eq({ result: "ok" });
  });
});
