/*
 This file is part of GNU Taler
 (C) 2021-2023 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 *
 * @author Sebastian Javier Marchano (sebasjm)
 */

import { tests } from "@gnu-taler/web-util/lib/index.browser";
import { expect } from "chai";
import { MerchantBackend } from "../declaration.js";
import { useInstanceOrders, useOrderAPI, useOrderDetails } from "./order.js";
import { ApiMockEnvironment } from "./testing.js";
import {
  API_CREATE_ORDER,
  API_DELETE_ORDER,
  API_FORGET_ORDER_BY_ID,
  API_GET_ORDER_BY_ID,
  API_LIST_ORDERS,
  API_REFUND_ORDER_BY_ID,
} from "./urls.js";

describe("order api interaction with listing", () => {
  it("should evict cache when creating an order", async () => {
    const env = new ApiMockEnvironment();

    env.addRequestExpectation(API_LIST_ORDERS, {
      qparam: { delta: 0, paid: "yes" },
      response: {
        orders: [{ order_id: "1" } as MerchantBackend.Orders.OrderHistoryEntry],
      },
    });

    env.addRequestExpectation(API_LIST_ORDERS, {
      qparam: { delta: -20, paid: "yes" },
      response: {
        orders: [{ order_id: "2" } as MerchantBackend.Orders.OrderHistoryEntry],
      },
    });

    const newDate = (d: Date) => {
      //console.log("new date", d);
    };

    const hookBehavior = await tests.hookBehaveLikeThis(
      () => {
        const query = useInstanceOrders({ paid: "yes" }, newDate);
        const api = useOrderAPI();
        return { query, api };
      },
      {},
      [
        ({ query, api }) => {
          expect(query.loading).true;
        },
        ({ query, api }) => {
          expect(env.assertJustExpectedRequestWereMade()).deep.eq({
            result: "ok",
          });
          expect(query.loading).undefined;
          expect(query.ok).true;
          if (!query.ok) return;
          expect(query.data).deep.equals({
            orders: [{ order_id: "1" }, { order_id: "2" }],
          });

          env.addRequestExpectation(API_CREATE_ORDER, {
            request: {
              order: { amount: "ARS:12", summary: "pay me" },
            },
            response: { order_id: "3" },
          });

          env.addRequestExpectation(API_LIST_ORDERS, {
            qparam: { delta: 0, paid: "yes" },
            response: {
              orders: [{ order_id: "1" } as any],
            },
          });

          env.addRequestExpectation(API_LIST_ORDERS, {
            qparam: { delta: -20, paid: "yes" },
            response: {
              orders: [{ order_id: "2" } as any, { order_id: "3" } as any],
            },
          });

          api.createOrder({
            order: { amount: "ARS:12", summary: "pay me" },
          } as any);
        },
        ({ query, api }) => {
          expect(env.assertJustExpectedRequestWereMade()).deep.eq({
            result: "ok",
          });

          expect(query.loading).undefined;
          expect(query.ok).true;
          if (!query.ok) return;
          expect(query.data).deep.equals({
            orders: [{ order_id: "1" }, { order_id: "2" }, { order_id: "3" }],
          });
        },
      ],
      env.buildTestingContext(),
    );
    expect(hookBehavior).deep.eq({ result: "ok" });
    expect(env.assertJustExpectedRequestWereMade()).deep.eq({ result: "ok" });
  });

  it("should evict cache when doing a refund", async () => {
    const env = new ApiMockEnvironment();

    env.addRequestExpectation(API_LIST_ORDERS, {
      qparam: { delta: 0, paid: "yes" },
      response: {
        orders: [
          {
            order_id: "1",
            amount: "EUR:12",
            refundable: true,
          } as MerchantBackend.Orders.OrderHistoryEntry,
        ],
      },
    });

    env.addRequestExpectation(API_LIST_ORDERS, {
      qparam: { delta: -20, paid: "yes" },
      response: { orders: [] },
    });

    const newDate = (d: Date) => {
      //console.log("new date", d);
    };

    const hookBehavior = await tests.hookBehaveLikeThis(
      () => {
        const query = useInstanceOrders({ paid: "yes" }, newDate);
        const api = useOrderAPI();
        return { query, api };
      },
      {},
      [
        ({ query, api }) => {
          expect(query.loading).true;
        },
        ({ query, api }) => {
          expect(env.assertJustExpectedRequestWereMade()).deep.eq({
            result: "ok",
          });

          expect(query.loading).undefined;
          expect(query.ok).true;
          if (!query.ok) return;
          expect(query.data).deep.equals({
            orders: [
              {
                order_id: "1",
                amount: "EUR:12",
                refundable: true,
              },
            ],
          });
          env.addRequestExpectation(API_REFUND_ORDER_BY_ID("1"), {
            request: {
              reason: "double pay",
              refund: "EUR:1",
            },
          });

          env.addRequestExpectation(API_LIST_ORDERS, {
            qparam: { delta: 0, paid: "yes" },
            response: {
              orders: [
                { order_id: "1", amount: "EUR:12", refundable: false } as any,
              ],
            },
          });

          env.addRequestExpectation(API_LIST_ORDERS, {
            qparam: { delta: -20, paid: "yes" },
            response: { orders: [] },
          });

          api.refundOrder("1", {
            reason: "double pay",
            refund: "EUR:1",
          });
        },
        ({ query, api }) => {
          expect(env.assertJustExpectedRequestWereMade()).deep.eq({
            result: "ok",
          });

          expect(query.loading).undefined;
          expect(query.ok).true;
          if (!query.ok) return;
          expect(query.data).deep.equals({
            orders: [
              {
                order_id: "1",
                amount: "EUR:12",
                refundable: false,
              },
            ],
          });
        },
      ],
      env.buildTestingContext(),
    );

    expect(hookBehavior).deep.eq({ result: "ok" });
    expect(env.assertJustExpectedRequestWereMade()).deep.eq({ result: "ok" });
  });

  it("should evict cache when deleting an order", async () => {
    const env = new ApiMockEnvironment();

    env.addRequestExpectation(API_LIST_ORDERS, {
      qparam: { delta: 0, paid: "yes" },
      response: {
        orders: [{ order_id: "1" } as MerchantBackend.Orders.OrderHistoryEntry],
      },
    });

    env.addRequestExpectation(API_LIST_ORDERS, {
      qparam: { delta: -20, paid: "yes" },
      response: {
        orders: [{ order_id: "2" } as MerchantBackend.Orders.OrderHistoryEntry],
      },
    });

    const newDate = (d: Date) => {
      //console.log("new date", d);
    };

    const hookBehavior = await tests.hookBehaveLikeThis(
      () => {
        const query = useInstanceOrders({ paid: "yes" }, newDate);
        const api = useOrderAPI();
        return { query, api };
      },
      {},
      [
        ({ query, api }) => {
          expect(query.loading).true;
        },
        ({ query, api }) => {
          expect(env.assertJustExpectedRequestWereMade()).deep.eq({
            result: "ok",
          });
          expect(query.loading).undefined;
          expect(query.ok).true;
          if (!query.ok) return;
          expect(query.data).deep.equals({
            orders: [{ order_id: "1" }, { order_id: "2" }],
          });

          env.addRequestExpectation(API_DELETE_ORDER("1"), {});

          env.addRequestExpectation(API_LIST_ORDERS, {
            qparam: { delta: 0, paid: "yes" },
            response: {
              orders: [],
            },
          });

          env.addRequestExpectation(API_LIST_ORDERS, {
            qparam: { delta: -20, paid: "yes" },
            response: {
              orders: [{ order_id: "2" } as any],
            },
          });

          api.deleteOrder("1");
        },
        ({ query, api }) => {
          expect(env.assertJustExpectedRequestWereMade()).deep.eq({
            result: "ok",
          });
          expect(query.loading).undefined;
          expect(query.ok).true;
          if (!query.ok) return;
          expect(query.data).deep.equals({
            orders: [{ order_id: "2" }],
          });
        },
      ],
      env.buildTestingContext(),
    );
    expect(hookBehavior).deep.eq({ result: "ok" });
    expect(env.assertJustExpectedRequestWereMade()).deep.eq({ result: "ok" });
  });
});

describe("order api interaction with details", () => {
  it("should evict cache when doing a refund", async () => {
    const env = new ApiMockEnvironment();

    env.addRequestExpectation(API_GET_ORDER_BY_ID("1"), {
      // qparam: { delta: 0, paid: "yes" },
      response: {
        summary: "description",
        refund_amount: "EUR:0",
      } as unknown as MerchantBackend.Orders.CheckPaymentPaidResponse,
    });

    const newDate = (d: Date) => {
      //console.log("new date", d);
    };

    const hookBehavior = await tests.hookBehaveLikeThis(
      () => {
        const query = useOrderDetails("1");
        const api = useOrderAPI();
        return { query, api };
      },
      {},
      [
        ({ query, api }) => {
          expect(query.loading).true;
        },
        ({ query, api }) => {
          expect(env.assertJustExpectedRequestWereMade()).deep.eq({
            result: "ok",
          });
          expect(query.loading).false;
          expect(query.ok).true;
          if (!query.ok) return;
          expect(query.data).deep.equals({
            summary: "description",
            refund_amount: "EUR:0",
          });
          env.addRequestExpectation(API_REFUND_ORDER_BY_ID("1"), {
            request: {
              reason: "double pay",
              refund: "EUR:1",
            },
          });

          env.addRequestExpectation(API_GET_ORDER_BY_ID("1"), {
            response: {
              summary: "description",
              refund_amount: "EUR:1",
            } as unknown as MerchantBackend.Orders.CheckPaymentPaidResponse,
          });

          api.refundOrder("1", {
            reason: "double pay",
            refund: "EUR:1",
          });
        },
        ({ query, api }) => {
          expect(env.assertJustExpectedRequestWereMade()).deep.eq({
            result: "ok",
          });
          expect(query.loading).false;
          expect(query.ok).true;
          if (!query.ok) return;
          expect(query.data).deep.equals({
            summary: "description",
            refund_amount: "EUR:1",
          });
        },
      ],
      env.buildTestingContext(),
    );

    expect(hookBehavior).deep.eq({ result: "ok" });
    expect(env.assertJustExpectedRequestWereMade()).deep.eq({ result: "ok" });
  });

  it("should evict cache when doing a forget", async () => {
    const env = new ApiMockEnvironment();

    env.addRequestExpectation(API_GET_ORDER_BY_ID("1"), {
      // qparam: { delta: 0, paid: "yes" },
      response: {
        summary: "description",
        refund_amount: "EUR:0",
      } as unknown as MerchantBackend.Orders.CheckPaymentPaidResponse,
    });

    const newDate = (d: Date) => {
      //console.log("new date", d);
    };

    const hookBehavior = await tests.hookBehaveLikeThis(
      () => {
        const query = useOrderDetails("1");
        const api = useOrderAPI();
        return { query, api };
      },
      {},
      [
        ({ query, api }) => {
          expect(query.loading).true;
        },
        ({ query, api }) => {
          expect(env.assertJustExpectedRequestWereMade()).deep.eq({
            result: "ok",
          });
          expect(query.loading).false;
          expect(query.ok).true;
          if (!query.ok) return;
          expect(query.data).deep.equals({
            summary: "description",
            refund_amount: "EUR:0",
          });
          env.addRequestExpectation(API_FORGET_ORDER_BY_ID("1"), {
            request: {
              fields: ["$.summary"],
            },
          });

          env.addRequestExpectation(API_GET_ORDER_BY_ID("1"), {
            response: {
              summary: undefined,
            } as unknown as MerchantBackend.Orders.CheckPaymentPaidResponse,
          });

          api.forgetOrder("1", {
            fields: ["$.summary"],
          });
        },
        ({ query, api }) => {
          expect(env.assertJustExpectedRequestWereMade()).deep.eq({
            result: "ok",
          });
          expect(query.loading).false;
          expect(query.ok).true;
          if (!query.ok) return;
          expect(query.data).deep.equals({
            summary: undefined,
          });
        },
      ],
      env.buildTestingContext(),
    );
    expect(hookBehavior).deep.eq({ result: "ok" });
    expect(env.assertJustExpectedRequestWereMade()).deep.eq({ result: "ok" });
  });
});

describe("order listing pagination", () => {
  it("should not load more if has reach the end", async () => {
    const env = new ApiMockEnvironment();
    env.addRequestExpectation(API_LIST_ORDERS, {
      qparam: { delta: 20, wired: "yes", date_ms: 12 },
      response: {
        orders: [{ order_id: "1" } as any],
      },
    });

    env.addRequestExpectation(API_LIST_ORDERS, {
      qparam: { delta: -20, wired: "yes", date_ms: 13 },
      response: {
        orders: [{ order_id: "2" } as any],
      },
    });

    const newDate = (d: Date) => {
      //console.log("new date", d);
    };

    const hookBehavior = await tests.hookBehaveLikeThis(
      () => {
        const date = new Date(12);
        const query = useInstanceOrders({ wired: "yes", date }, newDate);
        const api = useOrderAPI();
        return { query, api };
      },
      {},
      [
        ({ query, api }) => {
          expect(query.loading).true;
        },
        ({ query, api }) => {
          expect(env.assertJustExpectedRequestWereMade()).deep.eq({
            result: "ok",
          });
          expect(query.loading).undefined;
          expect(query.ok).true;
          if (!query.ok) return;
          expect(query.data).deep.equals({
            orders: [{ order_id: "1" }, { order_id: "2" }],
          });
          expect(query.isReachingEnd).true;
          expect(query.isReachingStart).true;

          // should not trigger new state update or query
          query.loadMore();
          query.loadMorePrev();
        },
      ],
      env.buildTestingContext(),
    );

    expect(hookBehavior).deep.eq({ result: "ok" });
    expect(env.assertJustExpectedRequestWereMade()).deep.eq({ result: "ok" });
  });

  it("should load more if result brings more that PAGE_SIZE", async () => {
    const env = new ApiMockEnvironment();

    const ordersFrom0to20 = Array.from({ length: 20 }).map((e, i) => ({
      order_id: String(i),
    }));
    const ordersFrom20to40 = Array.from({ length: 20 }).map((e, i) => ({
      order_id: String(i + 20),
    }));
    const ordersFrom20to0 = [...ordersFrom0to20].reverse();

    env.addRequestExpectation(API_LIST_ORDERS, {
      qparam: { delta: 20, wired: "yes", date_ms: 12 },
      response: {
        orders: ordersFrom0to20,
      },
    });

    env.addRequestExpectation(API_LIST_ORDERS, {
      qparam: { delta: -20, wired: "yes", date_ms: 13 },
      response: {
        orders: ordersFrom20to40,
      },
    });

    const newDate = (d: Date) => {
      //console.log("new date", d);
    };

    const hookBehavior = await tests.hookBehaveLikeThis(
      () => {
        const date = new Date(12);
        const query = useInstanceOrders({ wired: "yes", date }, newDate);
        const api = useOrderAPI();
        return { query, api };
      },
      {},
      [
        ({ query, api }) => {
          expect(query.loading).true;
        },
        ({ query, api }) => {
          expect(env.assertJustExpectedRequestWereMade()).deep.eq({
            result: "ok",
          });
          expect(query.loading).undefined;
          expect(query.ok).true;
          if (!query.ok) return;
          expect(query.data).deep.equals({
            orders: [...ordersFrom20to0, ...ordersFrom20to40],
          });
          expect(query.isReachingEnd).false;
          expect(query.isReachingStart).false;

          env.addRequestExpectation(API_LIST_ORDERS, {
            qparam: { delta: -40, wired: "yes", date_ms: 13 },
            response: {
              orders: [...ordersFrom20to40, { order_id: "41" }],
            },
          });

          query.loadMore();
        },
        ({ query, api }) => {
          expect(env.assertJustExpectedRequestWereMade()).deep.eq({
            result: "ok",
          });
          expect(query.loading).true;
        },
        ({ query, api }) => {
          expect(query.loading).undefined;
          expect(query.ok).true;
          if (!query.ok) return;
          expect(query.data).deep.equals({
            orders: [
              ...ordersFrom20to0,
              ...ordersFrom20to40,
              { order_id: "41" },
            ],
          });

          env.addRequestExpectation(API_LIST_ORDERS, {
            qparam: { delta: 40, wired: "yes", date_ms: 12 },
            response: {
              orders: [...ordersFrom0to20, { order_id: "-1" }],
            },
          });

          query.loadMorePrev();
        },
        ({ query, api }) => {
          expect(env.assertJustExpectedRequestWereMade()).deep.eq({
            result: "ok",
          });
          expect(query.loading).true;
        },
        ({ query, api }) => {
          expect(query.loading).undefined;
          expect(query.ok).true;
          if (!query.ok) return;
          expect(query.data).deep.equals({
            orders: [
              { order_id: "-1" },
              ...ordersFrom20to0,
              ...ordersFrom20to40,
              { order_id: "41" },
            ],
          });
        },
      ],
      env.buildTestingContext(),
    );
    expect(hookBehavior).deep.eq({ result: "ok" });
    expect(env.assertJustExpectedRequestWereMade()).deep.eq({ result: "ok" });
  });
});
