/*
 This file is part of GNU Taler
 (C) 2021-2023 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 *
 * @author Sebastian Javier Marchano (sebasjm)
 */

import { tests } from "@gnu-taler/web-util/lib/index.browser";
import { ComponentChildren, h, VNode } from "preact";
import { MerchantBackend } from "../declaration.js";
import {
  useAdminAPI,
  useInstanceAPI,
  useManagementAPI,
} from "../hooks/instance.js";
import { expect } from "chai";
import { ApiMockEnvironment } from "../hooks/testing.js";
import {
  API_CREATE_INSTANCE,
  API_UPDATE_CURRENT_INSTANCE_AUTH,
  API_UPDATE_INSTANCE_AUTH_BY_ID,
} from "../hooks/urls.js";

interface TestingContextProps {
  children?: ComponentChildren;
}

describe("backend context api ", () => {
  it("should use new token after updating the instance token in the settings as user", async () => {
    const env = new ApiMockEnvironment();

    const hookBehavior = await tests.hookBehaveLikeThis(
      () => {
        const instance = useInstanceAPI();
        const management = useManagementAPI("default");
        const admin = useAdminAPI();

        return { instance, management, admin };
      },
      {},
      [
        ({ instance, management, admin }) => {
          env.addRequestExpectation(API_UPDATE_INSTANCE_AUTH_BY_ID("default"), {
            request: {
              method: "token",
              token: "another_token",
            },
            response: {
              name: "instance_name",
            } as MerchantBackend.Instances.QueryInstancesResponse,
          });

          management.setNewToken("another_token");
        },
        ({ instance, management, admin }) => {
          expect(env.assertJustExpectedRequestWereMade()).deep.eq({
            result: "ok",
          });

          env.addRequestExpectation(API_CREATE_INSTANCE, {
            auth: "another_token",
            request: {
              id: "new_instance_id",
            } as MerchantBackend.Instances.InstanceConfigurationMessage,
          });

          admin.createInstance({
            id: "new_instance_id",
          } as MerchantBackend.Instances.InstanceConfigurationMessage);
        },
      ],
      env.buildTestingContext(),
    );

    expect(hookBehavior).deep.eq({ result: "ok" });
    expect(env.assertJustExpectedRequestWereMade()).deep.eq({ result: "ok" });
  });

  it("should use new token after updating the instance token in the settings as admin", async () => {
    const env = new ApiMockEnvironment();

    const hookBehavior = await tests.hookBehaveLikeThis(
      () => {
        const instance = useInstanceAPI();
        const management = useManagementAPI("default");
        const admin = useAdminAPI();

        return { instance, management, admin };
      },
      {},
      [
        ({ instance, management, admin }) => {
          env.addRequestExpectation(API_UPDATE_CURRENT_INSTANCE_AUTH, {
            request: {
              method: "token",
              token: "another_token",
            },
            response: {
              name: "instance_name",
            } as MerchantBackend.Instances.QueryInstancesResponse,
          });
          instance.setNewToken("another_token");
        },
        ({ instance, management, admin }) => {
          expect(env.assertJustExpectedRequestWereMade()).deep.eq({
            result: "ok",
          });

          env.addRequestExpectation(API_CREATE_INSTANCE, {
            auth: "another_token",
            request: {
              id: "new_instance_id",
            } as MerchantBackend.Instances.InstanceConfigurationMessage,
          });

          admin.createInstance({
            id: "new_instance_id",
          } as MerchantBackend.Instances.InstanceConfigurationMessage);
        },
      ],
      env.buildTestingContext(),
    );

    expect(hookBehavior).deep.eq({ result: "ok" });
    expect(env.assertJustExpectedRequestWereMade()).deep.eq({ result: "ok" });
  });
});
