/*
 This file is part of GNU Taler
 (C) 2022 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import {
  AbsoluteTime,
  Amounts,
  TalerProtocolTimestamp,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { useTranslationContext } from "@gnu-taler/web-util/browser";
import { alertFromError, useAlertContext } from "../../context/alert.js";
import { useBackendContext } from "../../context/backend.js";
import { useAsyncAsHook } from "../../hooks/useAsyncAsHook.js";
import { Props, State } from "./index.js";

export function useComponentState({
  talerPayPushUri,
  onClose,
  onSuccess,
}: Props): State {
  const api = useBackendContext();
  const { pushAlertOnError } = useAlertContext();
  const { i18n } = useTranslationContext();
  const hook = useAsyncAsHook(async () => {
    return await api.wallet.call(WalletApiOperation.PreparePeerPushCredit, {
      talerUri: talerPayPushUri,
    });
  }, []);

  if (!hook) {
    return {
      status: "loading",
      error: undefined,
    };
  }
  if (hook.hasError) {
    return {
      status: "error",
      retry: {
        onClick: pushAlertOnError(async () => {
          hook.retry();
        }),
      },
      error: alertFromError(
        i18n,
        i18n.str`Could not load the invoice payment status`,
        hook,
      ),
    };
  }

  const {
    contractTerms,
    transactionId,
    amountEffective,
    amountRaw,
    exchangeBaseUrl,
  } = hook.response;

  const effective = Amounts.parseOrThrow(amountEffective);
  const raw = Amounts.parseOrThrow(amountRaw);
  const summary: string = contractTerms.summary;
  const expiration: TalerProtocolTimestamp = contractTerms.purse_expiration;

  async function accept(): Promise<void> {
    const resp = await api.wallet.call(
      WalletApiOperation.ConfirmPeerPushCredit,
      {
        transactionId,
      },
    );
    onSuccess(resp.transactionId);
  }
  return {
    status: "ready",
    effective,
    exchangeBaseUrl,
    raw,
    error: undefined,
    accept: {
      onClick: pushAlertOnError(accept),
    },
    summary,
    expiration: expiration
      ? AbsoluteTime.fromProtocolTimestamp(expiration)
      : undefined,
    cancel: {
      onClick: pushAlertOnError(onClose),
    },
  };
}
