/*
 This file is part of GNU Taler
 (C) 2022-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
import {
  HttpStatusCode,
  TalerError,
  assertUnreachable,
} from "@gnu-taler/taler-util";
import {
  Attention,
  Loading,
  useTranslationContext,
} from "@gnu-taler/web-util/browser";
import { ComponentChildren, Fragment, VNode, h } from "preact";
import { useChallengeSession } from "../hooks/challenge.js";
import { SessionId, useSessionState } from "../hooks/session.js";

interface Props {
  nonce: string;
  children: ComponentChildren;
  sessionId?: SessionId;
  onCompleted?: () => void;
  onChangeLeft?: () => void;
  onNoMoreChanges?: () => void;
  onNoInfo: () => void;
}
export function CheckChallengeIsUpToDate({
  sessionId: sessionFromParam,
  nonce,
  children,
  onCompleted,
  onChangeLeft,
  onNoMoreChanges,
  onNoInfo,
}: Props): VNode {
  const { state, updateStatus } = useSessionState();
  const { i18n } = useTranslationContext();

  const sessionId = sessionFromParam
    ? sessionFromParam
    : !state
      ? undefined
      : {
          clientId: state.clientId,
          redirectURL: state.redirectURL,
          state: state.state,
        };

  const result = useChallengeSession(nonce, sessionId);
  console.log("asd");
  if (!sessionId) {
    onNoInfo();
    return <Loading />;
  }

  if (!result) {
    return <Loading />;
  }
  if (result instanceof TalerError) {
    return <pre>{JSON.stringify(result, undefined, 2)}</pre>;
  }

  if (result.type === "fail") {
    switch (result.case) {
      case HttpStatusCode.BadRequest: {
        return (
          <Attention type="danger" title={i18n.str`Bad request`}>
            <i18n.Translate>
              Could not start the challenge, check configuration.
            </i18n.Translate>
          </Attention>
        );
      }
      case HttpStatusCode.NotFound: {
        return (
          <Attention type="danger" title={i18n.str`Not found`}>
            <i18n.Translate>Nonce not found</i18n.Translate>
          </Attention>
        );
      }
      case HttpStatusCode.NotAcceptable: {
        return (
          <Attention type="danger" title={i18n.str`Not acceptable`}>
            <i18n.Translate>
              Server has wrong template configuration
            </i18n.Translate>
          </Attention>
        );
      }
      case HttpStatusCode.InternalServerError: {
        return (
          <Attention type="danger" title={i18n.str`Internal error`}>
            <i18n.Translate>Check logs</i18n.Translate>
          </Attention>
        );
      }
      default:
        assertUnreachable(result);
    }
  }

  updateStatus(result.body);

  if (onCompleted && "redirectURL" in result.body) {
    onCompleted();
    return <Loading />;
  }

  if (onNoMoreChanges && !result.body.changes_left) {
    onNoMoreChanges();
    return <Loading />;
  }

  if (onChangeLeft && !result.body.changes_left) {
    onChangeLeft();
    return <Loading />;
  }

  return <Fragment>{children}</Fragment>;
}
