/*
 This file is part of GNU Taler
 (C) 2021-2023 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 *
 * @author Sebastian Javier Marchano (sebasjm)
 */

import {
  ErrorType,
  HttpError,
  useTranslationContext,
} from "@gnu-taler/web-util/browser";
import { Fragment, h, VNode } from "preact";
import { useState } from "preact/hooks";
import { Loading } from "../../../../components/exception/loading.js";
import { NotificationCard } from "../../../../components/menu/index.js";
import { MerchantBackend } from "../../../../declaration.js";
import {
  useTemplateAPI,
  useTemplateDetails,
} from "../../../../hooks/templates.js";
import { Notification } from "../../../../utils/types.js";
import { QrPage } from "./QrPage.js";
import { HttpStatusCode } from "@gnu-taler/taler-util";

export type Entity = MerchantBackend.Transfers.TransferInformation;
interface Props {
  onBack?: () => void;
  onUnauthorized: () => VNode;
  onNotFound: () => VNode;
  onLoadError: (e: HttpError<MerchantBackend.ErrorDetail>) => VNode;
  tid: string;
}

export default function TemplateQrPage({
  tid,
  onBack,
  onLoadError,
  onNotFound,
  onUnauthorized,
}: Props): VNode {
  const result = useTemplateDetails(tid);
  const [notif, setNotif] = useState<Notification | undefined>(undefined);

  if (result.loading) return <Loading />;
  if (!result.ok) {
    if (
      result.type === ErrorType.CLIENT &&
      result.status === HttpStatusCode.Unauthorized
    )
      return onUnauthorized();
    if (
      result.type === ErrorType.CLIENT &&
      result.status === HttpStatusCode.NotFound
    )
      return onNotFound();
    return onLoadError(result);
  }

  return (
    <>
      <NotificationCard notification={notif} />
      <QrPage contract={result.data.template_contract} id={tid} onBack={onBack} />
    </>
  );
}
