/*
 This file is part of GNU Taler
 (C) 2021-2023 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 *
 * @author Sebastian Javier Marchano (sebasjm)
 */

import { useTranslationContext } from "@gnu-taler/web-util/browser";
import { format } from "date-fns";
import { h, VNode, Fragment } from "preact";
import { useState } from "preact/hooks";
import { DatePicker } from "../../../../components/picker/DatePicker.js";
import { MerchantBackend, WithId } from "../../../../declaration.js";
import { CardTable } from "./Table.js";
import { dateFormatForSettings, useSettings } from "../../../../hooks/useSettings.js";

export interface ListPageProps {
  onShowAll: () => void;
  onShowNotPaid: () => void;
  onShowPaid: () => void;
  onShowRefunded: () => void;
  onShowNotWired: () => void;
  onShowWired: () => void;
  onCopyURL: (id: string) => void;
  isAllActive: string;
  isPaidActive: string;
  isNotPaidActive: string;
  isRefundedActive: string;
  isNotWiredActive: string;
  isWiredActive: string;

  jumpToDate?: Date;
  onSelectDate: (date?: Date) => void;

  orders: (MerchantBackend.Orders.OrderHistoryEntry & WithId)[];
  onLoadMoreBefore?: () => void;
  hasMoreBefore?: boolean;
  hasMoreAfter?: boolean;
  onLoadMoreAfter?: () => void;

  onSelectOrder: (o: MerchantBackend.Orders.OrderHistoryEntry & WithId) => void;
  onRefundOrder: (o: MerchantBackend.Orders.OrderHistoryEntry & WithId) => void;
  onCreate: () => void;
}

export function ListPage({
  hasMoreAfter,
  hasMoreBefore,
  onLoadMoreAfter,
  onLoadMoreBefore,
  orders,
  isAllActive,
  onSelectOrder,
  onRefundOrder,
  jumpToDate,
  onCopyURL,
  onShowAll,
  onShowPaid,
  onShowNotPaid,
  onShowRefunded,
  onShowNotWired,
  onShowWired,
  onSelectDate,
  isPaidActive,
  isRefundedActive,
  isNotWiredActive,
  onCreate,
  isNotPaidActive,
  isWiredActive,
}: ListPageProps): VNode {
  const { i18n } = useTranslationContext();
  const dateTooltip = i18n.str`select date to show nearby orders`;
  const [pickDate, setPickDate] = useState(false);
  const [settings] = useSettings();

  return (
    <Fragment>
      <div class="columns">
        <div class="column is-two-thirds">
          <div class="tabs" style={{ overflow: "inherit" }}>
            <ul>
              <li class={isNotPaidActive}>
                <div
                  class="has-tooltip-right"
                  data-tooltip={i18n.str`only show paid orders`}
                >
                  <a onClick={onShowNotPaid}>
                    <i18n.Translate>New</i18n.Translate>
                  </a>
                </div>
              </li>
              <li class={isPaidActive}>
                <div
                  class="has-tooltip-right"
                  data-tooltip={i18n.str`only show paid orders`}
                >
                  <a onClick={onShowPaid}>
                    <i18n.Translate>Paid</i18n.Translate>
                  </a>
                </div>
              </li>
              <li class={isRefundedActive}>
                <div
                  class="has-tooltip-right"
                  data-tooltip={i18n.str`only show orders with refunds`}
                >
                  <a onClick={onShowRefunded}>
                    <i18n.Translate>Refunded</i18n.Translate>
                  </a>
                </div>
              </li>
              <li class={isNotWiredActive}>
                <div
                  class="has-tooltip-left"
                  data-tooltip={i18n.str`only show orders where customers paid, but wire payments from payment provider are still pending`}
                >
                  <a onClick={onShowNotWired}>
                    <i18n.Translate>Not wired</i18n.Translate>
                  </a>
                </div>
              </li>
              <li class={isWiredActive}>
                <div
                  class="has-tooltip-left"
                  data-tooltip={i18n.str`only show orders where customers paid, but wire payments from payment provider are still pending`}
                >
                  <a onClick={onShowWired}>
                    <i18n.Translate>Completed</i18n.Translate>
                  </a>
                </div>
              </li>
              <li class={isAllActive}>
                <div
                  class="has-tooltip-right"
                  data-tooltip={i18n.str`remove all filters`}
                >
                  <a onClick={onShowAll}>
                    <i18n.Translate>All</i18n.Translate>
                  </a>
                </div>
              </li>
            </ul>
          </div>
        </div>
        <div class="column ">
          <div class="buttons is-right">
            <div class="field has-addons">
              {jumpToDate && (
                <div class="control">
                  <a class="button is-fullwidth" onClick={() => onSelectDate(undefined)}>
                    <span
                      class="icon"
                      data-tooltip={i18n.str`clear date filter`}
                    >
                      <i class="mdi mdi-close" />
                    </span>
                  </a>
                </div>
              )}
              <div class="control">
                <span class="has-tooltip-top" data-tooltip={dateTooltip}>
                  <input
                    class="input"
                    type="text"
                    readonly
                    value={!jumpToDate ? "" : format(jumpToDate, dateFormatForSettings(settings))}
                    placeholder={i18n.str`date (${dateFormatForSettings(settings)})`}
                    onClick={() => {
                      setPickDate(true);
                    }}
                  />
                </span>
              </div>
              <div class="control">
                <span class="has-tooltip-left" data-tooltip={dateTooltip}>
                  <a
                    class="button  is-fullwidth"
                    onClick={() => {
                      setPickDate(true);
                    }}
                  >
                    <span class="icon">
                      <i class="mdi mdi-calendar" />
                    </span>
                  </a>
                </span>
              </div>
            </div>
          </div>
        </div>
      </div>

      <DatePicker
        opened={pickDate}
        closeFunction={() => setPickDate(false)}
        dateReceiver={onSelectDate}
      />

      <CardTable
        orders={orders}
        onCreate={onCreate}
        onCopyURL={onCopyURL}
        onSelect={onSelectOrder}
        onRefund={onRefundOrder}
        hasMoreAfter={hasMoreAfter}
        hasMoreBefore={hasMoreBefore}
        onLoadMoreAfter={onLoadMoreAfter}
        onLoadMoreBefore={onLoadMoreBefore}
      />
    </Fragment>
  );
}
