/*
 This file is part of GNU Taler
 (C) 2021-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
import {
  HttpResponse,
  HttpResponseOk,
  RequestError,
} from "@gnu-taler/web-util/browser";
import { AuditorBackend, MerchantBackend, WithId } from "../declaration.js";
import { useBackendInstanceRequest, useMatchMutate } from "./backend.js";

// FIX default import https://github.com/microsoft/TypeScript/issues/49189
import _useSWR, { SWRHook, useSWRConfig } from "swr";
const useSWR = _useSWR as unknown as SWRHook;

export interface DepositConfirmationAPI {
  getDepositConfirmation: (
    id: string,
  ) => Promise<void>;
  createDepositConfirmation: (
    data: MerchantBackend.Products.ProductAddDetail,
  ) => Promise<void>;
  updateDepositConfirmation: (
    id: string,
    data: MerchantBackend.Products.ProductPatchDetail,
  ) => Promise<void>;
  deleteDepositConfirmation: (id: string) => Promise<void>;
}

export function useDepositConfirmationAPI(): DepositConfirmationAPI {
  const mutateAll = useMatchMutate();
  const { mutate } = useSWRConfig();

  const { request } = useBackendInstanceRequest();

  const createDepositConfirmation = async (
    data: MerchantBackend.Products.ProductAddDetail,
  ): Promise<void> => {
    const res = await request(`/private/products`, {
      method: "POST",
      data,
    });

    return await mutateAll(/.*\/private\/products.*/);
  };

  const updateDepositConfirmation = async (
    productId: string,
    data: MerchantBackend.Products.ProductPatchDetail,
  ): Promise<void> => {
    const r = await request(`/private/products/${productId}`, {
      method: "PATCH",
      data,
    });

    return await mutateAll(/.*\/private\/products.*/);
  };

  const deleteDepositConfirmation = async (productId: string): Promise<void> => {
    await request(`/private/products/${productId}`, {
      method: "DELETE",
    });
    await mutate([`/private/products`]);
  };

  const getDepositConfirmation = async (
    serialId: string,
  ): Promise<void> => {
    await request(`/deposit-confirmation/${serialId}`, {
      method: "GET",
    });

    return
  };

  return {createDepositConfirmation, updateDepositConfirmation, deleteDepositConfirmation, getDepositConfirmation};
}

export function useDepositConfirmation(): HttpResponse<
  (AuditorBackend.DepositConfirmation.DepositConfirmationDetail & WithId)[],
  AuditorBackend.ErrorDetail
> {
  const { fetcher, multiFetcher } = useBackendInstanceRequest();

  const { data: list, error: listError } = useSWR<
    HttpResponseOk<AuditorBackend.DepositConfirmation.DepositConfirmationList>,
    RequestError<AuditorBackend.ErrorDetail>
  >([`/deposit-confirmation`], fetcher, {
    refreshInterval: 0,
    refreshWhenHidden: false,
    revalidateOnFocus: false,
    revalidateOnReconnect: false,
    refreshWhenOffline: false,
  });

  const paths = (list?.data.depositConfirmations || []).map(
    (p) => `/deposit-confirmation/${p.serial_id}`,
  );
  const { data: depositConfirmations, error: depositConfirmationError } = useSWR<
    HttpResponseOk<AuditorBackend.DepositConfirmation.DepositConfirmationDetail>[],
    RequestError<AuditorBackend.ErrorDetail>
  >([paths], multiFetcher, {
    refreshInterval: 0,
    refreshWhenHidden: false,
    revalidateOnFocus: false,
    revalidateOnReconnect: false,
    refreshWhenOffline: false,
  });

  if (listError) return listError.cause;
  if (depositConfirmationError) return depositConfirmationError.cause;

  if (depositConfirmations) {
    const dataWithId = depositConfirmations.map((d) => {
      //take the id from the queried url
      return {
        ...d.data,
        id: d.info?.url.replace(/.*\/deposit-confirmation\//, "") || "",
      };
    });
    return { ok: true, data: dataWithId };
  }
  return { loading: true };
}

export function useDepositConfirmationDetails(
  serialId: string,
): HttpResponse<
  AuditorBackend.DepositConfirmation.DepositConfirmationDetail,
  AuditorBackend.ErrorDetail
> {
  const { fetcher } = useBackendInstanceRequest();

  const { data, error, isValidating } = useSWR<
    HttpResponseOk<AuditorBackend.DepositConfirmation.DepositConfirmationDetail>,
    RequestError<AuditorBackend.ErrorDetail>
  >([`/deposit-confirmation/${serialId}`], fetcher, {
    refreshInterval: 0,
    refreshWhenHidden: false,
    revalidateOnFocus: false,
    revalidateOnReconnect: false,
    refreshWhenOffline: false,
  });

  if (isValidating) return { loading: true, data: data?.data };
  if (data) return data;
  if (error) return error.cause;
  return { loading: true };
}
