/*
 This file is part of GNU Taler
 (C) 2019 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  AcceptTipResponse,
  Logger,
  PrepareTipResult,
  TransactionAction,
  TransactionIdStr,
  TransactionMajorState,
  TransactionMinorState,
  TransactionState,
  TransactionType,
  assertUnreachable,
} from "@gnu-taler/taler-util";
import {
  PendingTaskType,
  TaskIdStr,
  TaskRunResult,
  TombstoneTag,
  TransactionContext,
  constructTaskIdentifier,
} from "./common.js";
import { RewardRecord, RewardRecordStatus } from "./db.js";
import {
  constructTransactionIdentifier,
} from "./transactions.js";
import { InternalWalletState, WalletExecutionContext } from "./wallet.js";

export class RewardTransactionContext implements TransactionContext {
  public transactionId: TransactionIdStr;
  public taskId: TaskIdStr;

  constructor(
    public wex: WalletExecutionContext,
    public walletRewardId: string,
  ) {
    this.transactionId = constructTransactionIdentifier({
      tag: TransactionType.Reward,
      walletRewardId,
    });
    this.taskId = constructTaskIdentifier({
      tag: PendingTaskType.RewardPickup,
      walletRewardId,
    });
  }

  async deleteTransaction(): Promise<void> {
    throw Error("unsupported operation");
  }

  async suspendTransaction(): Promise<void> {
    throw Error("unsupported operation");
  }

  async abortTransaction(): Promise<void> {
    throw Error("unsupported operation");
  }

  async resumeTransaction(): Promise<void> {
    throw Error("unsupported operation");
  }

  async failTransaction(): Promise<void> {
    throw Error("unsupported operation");
  }
}

/**
 * Get the (DD37-style) transaction status based on the
 * database record of a reward.
 */
export function computeRewardTransactionStatus(
  tipRecord: RewardRecord,
): TransactionState {
  switch (tipRecord.status) {
    case RewardRecordStatus.Done:
      return {
        major: TransactionMajorState.Done,
      };
    case RewardRecordStatus.Aborted:
      return {
        major: TransactionMajorState.Aborted,
      };
    case RewardRecordStatus.PendingPickup:
      return {
        major: TransactionMajorState.Pending,
        minor: TransactionMinorState.Pickup,
      };
    case RewardRecordStatus.DialogAccept:
      return {
        major: TransactionMajorState.Dialog,
        minor: TransactionMinorState.Proposed,
      };
    case RewardRecordStatus.SuspendedPickup:
      return {
        major: TransactionMajorState.Pending,
        minor: TransactionMinorState.Pickup,
      };
    case RewardRecordStatus.Failed:
      return {
        major: TransactionMajorState.Failed,
      };
    default:
      assertUnreachable(tipRecord.status);
  }
}

export function computeTipTransactionActions(
  tipRecord: RewardRecord,
): TransactionAction[] {
  switch (tipRecord.status) {
    case RewardRecordStatus.Done:
      return [TransactionAction.Delete];
    case RewardRecordStatus.Failed:
      return [TransactionAction.Delete];
    case RewardRecordStatus.Aborted:
      return [TransactionAction.Delete];
    case RewardRecordStatus.PendingPickup:
      return [TransactionAction.Suspend, TransactionAction.Fail];
    case RewardRecordStatus.SuspendedPickup:
      return [TransactionAction.Resume, TransactionAction.Fail];
    case RewardRecordStatus.DialogAccept:
      return [TransactionAction.Abort];
    default:
      assertUnreachable(tipRecord.status);
  }
}

export async function prepareReward(
  ws: InternalWalletState,
  talerTipUri: string,
): Promise<PrepareTipResult> {
  throw Error("the rewards feature is not supported anymore");
}

export async function processTip(
  ws: InternalWalletState,
  walletTipId: string,
): Promise<TaskRunResult> {
  return TaskRunResult.finished();
}

export async function acceptTip(
  ws: InternalWalletState,
  transactionId: TransactionIdStr,
): Promise<AcceptTipResponse> {
  throw Error("the rewards feature is not supported anymore");
}
