/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  ExchangeEntryStatus,
  NotificationType,
  TalerError,
  TalerErrorCode,
  WalletNotification,
  j2s,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { CoinConfig, defaultCoinConfig } from "../harness/denomStructures.js";
import {
  ExchangeService,
  FakebankService,
  GlobalTestState,
  WalletClient,
  WalletService,
  setupDb,
} from "../harness/harness.js";
import { withdrawViaBankV2 } from "../harness/helpers.js";

/**
 * Test for DD48 notifications.
 */
export async function runWalletDd48Test(t: GlobalTestState) {
  // Set up test environment

  const db = await setupDb(t);

  const bank = await FakebankService.create(t, {
    allowRegistrations: true,
    currency: "TESTKUDOS",
    database: db.connStr,
    httpPort: 8082,
  });

  const exchange = ExchangeService.create(t, {
    name: "testexchange-1",
    currency: "TESTKUDOS",
    httpPort: 8081,
    database: db.connStr,
  });

  const exchangeBankAccount = await bank.createExchangeAccount(
    "myexchange",
    "x",
  );
  exchange.addBankAccount("1", exchangeBankAccount);

  bank.setSuggestedExchange(exchange, exchangeBankAccount.accountPaytoUri);

  await bank.start();

  await bank.pingUntilAvailable();

  const coinConfig: CoinConfig[] = defaultCoinConfig.map((x) => x("TESTKUDOS"));
  exchange.addCoinConfigList(coinConfig);

  await exchange.start();
  await exchange.pingUntilAvailable();

  const walletService = new WalletService(t, {
    name: "wallet",
    useInMemoryDb: true,
  });
  await walletService.start();
  await walletService.pingUntilAvailable();

  const allNotifications: WalletNotification[] = [];

  const walletClient = new WalletClient({
    name: "wallet",
    unixPath: walletService.socketPath,
    onNotification(n) {
      console.log("got notification", n);
      allNotifications.push(n);
    },
  });
  await walletClient.connect();
  await walletClient.client.call(WalletApiOperation.InitWallet, {
    config: {
      testing: {
        skipDefaults: true,
      },
    },
  });

  await walletClient.call(WalletApiOperation.AddExchange, {
    exchangeBaseUrl: exchange.baseUrl,
  });

  {
    const exchangeEntry = await walletClient.call(
      WalletApiOperation.GetExchangeEntryByUrl,
      {
        exchangeBaseUrl: exchange.baseUrl,
      },
    );

    t.assertDeepEqual(
      exchangeEntry.exchangeEntryStatus,
      ExchangeEntryStatus.Ephemeral,
    );

    const resources = await walletClient.call(
      WalletApiOperation.GetExchangeResources,
      {
        exchangeBaseUrl: exchange.baseUrl,
      },
    );
    t.assertDeepEqual(resources.hasResources, false);
  }

  const wres = await withdrawViaBankV2(t, {
    walletClient,
    amount: "TESTKUDOS:20",
    bank,
    exchange,
  });

  await wres.withdrawalFinishedCond;

  const exchangeEntry = await walletClient.call(
    WalletApiOperation.GetExchangeEntryByUrl,
    {
      exchangeBaseUrl: exchange.baseUrl,
    },
  );

  t.assertDeepEqual(
    exchangeEntry.exchangeEntryStatus,
    ExchangeEntryStatus.Used,
  );

  t.assertTrue(
    !!allNotifications.find(
      (x) =>
        x.type === NotificationType.ExchangeStateTransition &&
        x.oldExchangeState == null &&
        x.newExchangeState.exchangeEntryStatus ===
          ExchangeEntryStatus.Ephemeral,
    ),
  );

  console.log(j2s(allNotifications));

  const delErr = await t.assertThrowsAsync(async () => {
    await walletClient.call(WalletApiOperation.DeleteExchange, {
      exchangeBaseUrl: exchange.baseUrl,
    });
  });

  t.assertTrue(delErr instanceof TalerError);
  t.assertDeepEqual(
    delErr.errorDetail.code,
    TalerErrorCode.WALLET_EXCHANGE_ENTRY_USED,
  );

  await walletClient.call(WalletApiOperation.DeleteExchange, {
    exchangeBaseUrl: exchange.baseUrl,
    purge: true,
  });
}

runWalletDd48Test.suites = ["wallet"];
