/*
 This file is part of GNU Taler
 (C) 2020-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  Duration,
  j2s,
  MerchantApiClient,
  NotificationType,
  TransactionMajorState,
  TransactionType,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { GlobalTestState } from "../harness/harness.js";
import {
  createSimpleTestkudosEnvironmentV2,
  withdrawViaBankV2,
} from "../harness/helpers.js";

export async function runRefundTest(t: GlobalTestState) {
  // Set up test environment

  const {
    walletClient: wallet,
    bank,
    exchange,
    merchant,
  } = await createSimpleTestkudosEnvironmentV2(t);

  const merchantClient = new MerchantApiClient(merchant.makeInstanceBaseUrl());

  // Withdraw digital cash into the wallet.

  const withdrawalRes = await withdrawViaBankV2(t, {
    walletClient: wallet,
    bank,
    exchange,
    amount: "TESTKUDOS:20",
  });

  await withdrawalRes.withdrawalFinishedCond;

  // Set up order.
  const orderResp = await merchantClient.createOrder({
    order: {
      summary: "Buy me!",
      amount: "TESTKUDOS:5",
      fulfillment_url: "taler://fulfillment-success/thx",
    },
    refund_delay: Duration.toTalerProtocolDuration(
      Duration.fromSpec({ minutes: 5 }),
    ),
  });

  let orderStatus = await merchantClient.queryPrivateOrderStatus({
    orderId: orderResp.order_id,
  });

  t.assertTrue(orderStatus.order_status === "unpaid");

  // Make wallet pay for the order

  const r1 = await wallet.client.call(WalletApiOperation.PreparePayForUri, {
    talerPayUri: orderStatus.taler_pay_uri,
  });

  await wallet.client.call(WalletApiOperation.ConfirmPay, {
    transactionId: r1.transactionId,
  });

  // Check if payment was successful.

  orderStatus = await merchantClient.queryPrivateOrderStatus({
    orderId: orderResp.order_id,
  });

  t.assertTrue(orderStatus.order_status === "paid");

  {
    const tx = await wallet.client.call(WalletApiOperation.GetTransactionById, {
      transactionId: r1.transactionId,
    });

    t.assertTrue(
      tx.type === TransactionType.Payment && tx.refundPending === undefined,
    );
  }

  const ref = await merchantClient.giveRefund({
    amount: "TESTKUDOS:5",
    instance: "default",
    justification: "foo",
    orderId: orderResp.order_id,
  });

  console.log(ref);

  {
    const refundFinishedCond = wallet.waitForNotificationCond(
      (x) =>
        x.type === NotificationType.TransactionStateTransition &&
        x.transactionId === r1.transactionId &&
        x.newTxState.major === TransactionMajorState.Done,
    );
    await wallet.client.call(WalletApiOperation.StartRefundQuery, {
      transactionId: r1.transactionId,
    });
    await refundFinishedCond;
  }

  {
    const r2 = await wallet.client.call(WalletApiOperation.GetBalances, {});
    console.log(JSON.stringify(r2, undefined, 2));
  }

  {
    const r2 = await wallet.client.call(WalletApiOperation.GetTransactions, {});
    console.log(JSON.stringify(r2, undefined, 2));
  }

  {
    const tx = await wallet.client.call(WalletApiOperation.GetTransactionById, {
      transactionId: r1.transactionId,
    });

    console.log(j2s(tx));

    t.assertTrue(
      tx.type === TransactionType.Payment && tx.refundPending === undefined,
    );
  }
}

runRefundTest.suites = ["wallet"];
