/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  AbsoluteTime,
  Duration,
  MerchantApiClient,
  TransactionMajorState,
  TransactionType,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { GlobalTestState } from "../harness/harness.js";
import {
  applyTimeTravelV2,
  createSimpleTestkudosEnvironmentV2,
  withdrawViaBankV2,
} from "../harness/helpers.js";

/**
 * Test wallet behavior when a refund expires before the wallet
 * can claim it.
 */
export async function runRefundGoneTest(t: GlobalTestState) {
  // Set up test environment

  const { walletClient, bank, exchange, merchant } =
    await createSimpleTestkudosEnvironmentV2(t);

  const merchantClient = new MerchantApiClient(merchant.makeInstanceBaseUrl());

  // Withdraw digital cash into the wallet.

  const wres = await withdrawViaBankV2(t, {
    walletClient,
    bank,
    exchange,
    amount: "TESTKUDOS:20",
  });

  await wres.withdrawalFinishedCond;

  // Set up order.

  const orderResp = await merchantClient.createOrder({
    order: {
      summary: "Buy me!",
      amount: "TESTKUDOS:5",
      fulfillment_url: "taler://fulfillment-success/thx",
      pay_deadline: AbsoluteTime.toProtocolTimestamp(
        AbsoluteTime.addDuration(
          AbsoluteTime.now(),
          Duration.fromSpec({
            minutes: 10,
          }),
        ),
      ),
    },
    refund_delay: Duration.toTalerProtocolDuration(
      Duration.fromSpec({ minutes: 1 }),
    ),
  });

  let orderStatus = await merchantClient.queryPrivateOrderStatus({
    orderId: orderResp.order_id,
  });

  t.assertTrue(orderStatus.order_status === "unpaid");

  // Make wallet pay for the order

  const r1 = await walletClient.call(WalletApiOperation.PreparePayForUri, {
    talerPayUri: orderStatus.taler_pay_uri,
  });

  const r2 = await walletClient.call(WalletApiOperation.ConfirmPay, {
    transactionId: r1.transactionId,
  });

  // Check if payment was successful.

  orderStatus = await merchantClient.queryPrivateOrderStatus({
    orderId: orderResp.order_id,
  });

  t.assertTrue(orderStatus.order_status === "paid");

  console.log(orderStatus);

  await applyTimeTravelV2(
    Duration.toMilliseconds(Duration.fromSpec({ hours: 1 })),
    { exchange, merchant, walletClient: walletClient },
  );
  await exchange.stopAggregator();
  await exchange.runAggregatorOnce();

  const ref = await merchantClient.giveRefund({
    amount: "TESTKUDOS:5",
    instance: "default",
    justification: "foo",
    orderId: orderResp.order_id,
  });

  console.log(ref);

  await walletClient.call(WalletApiOperation.StartRefundQuery, {
    transactionId: r1.transactionId,
  });

  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});

  let r = await walletClient.call(WalletApiOperation.GetBalances, {});
  console.log(JSON.stringify(r, undefined, 2));

  const r3 = await walletClient.call(WalletApiOperation.GetTransactions, {});
  console.log(JSON.stringify(r3, undefined, 2));

  const refundTx = r3.transactions[2];

  t.assertDeepEqual(refundTx.type, TransactionType.Refund);
  t.assertDeepEqual(refundTx.txState.major, TransactionMajorState.Failed);
}

runRefundGoneTest.suites = ["wallet"];
