/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  ConfirmPayResultType,
  MerchantApiClient,
  PreparePayResultType,
  j2s,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { GlobalTestState } from "../harness/harness.js";
import {
  createSimpleTestkudosEnvironmentV2,
  withdrawViaBankV2,
} from "../harness/helpers.js";

/**
 * Test behavior when an order is deleted while the wallet is paying for it.
 */
export async function runPaymentDeletedTest(t: GlobalTestState) {
  // Set up test environment

  const { walletClient, bank, exchange, merchant } =
    await createSimpleTestkudosEnvironmentV2(t);

  // First, make a "free" payment when we don't even have
  // any money in the

  // Withdraw digital cash into the wallet.
  await withdrawViaBankV2(t, {
    walletClient,
    bank,
    exchange,
    amount: "TESTKUDOS:20",
  });

  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});

  const merchantClient = new MerchantApiClient(merchant.makeInstanceBaseUrl());

  const orderResp = await merchantClient.createOrder({
    order: {
      summary: "Hello",
      amount: "TESTKUDOS:2",
    },
  });

  let orderStatus = await merchantClient.queryPrivateOrderStatus({
    orderId: orderResp.order_id,
  });

  t.assertTrue(orderStatus.order_status === "unpaid");

  // Make wallet pay for the order

  const preparePayResult = await walletClient.call(
    WalletApiOperation.PreparePayForUri,
    {
      talerPayUri: orderStatus.taler_pay_uri,
    },
  );

  t.assertTrue(
    preparePayResult.status === PreparePayResultType.PaymentPossible,
  );

  await merchantClient.deleteOrder({
    orderId: orderResp.order_id,
    force: true,
  });

  const r2 = await walletClient.call(WalletApiOperation.ConfirmPay, {
    transactionId: preparePayResult.transactionId,
  });

  t.assertTrue(r2.type === ConfirmPayResultType.Pending);

  await walletClient.call(WalletApiOperation.AbortTransaction, {
    transactionId: preparePayResult.transactionId,
  });

  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});

  const bal = await walletClient.call(WalletApiOperation.GetBalances, {});
  console.log(j2s(bal));
}

runPaymentDeletedTest.suites = ["wallet"];
