/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  ConfirmPayResultType,
  MerchantApiClient,
  PreparePayResultType,
  URL,
  codecForMerchantOrderStatusUnpaid,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { FaultInjectionRequestContext } from "../harness/faultInjection.js";
import { GlobalTestState, harnessHttpLib } from "../harness/harness.js";
import {
  createFaultInjectedMerchantTestkudosEnvironment,
  withdrawViaBankV2,
} from "../harness/helpers.js";

/**
 * Run test for the wallets repurchase detection mechanism
 * based on the fulfillment URL.
 *
 * FIXME: This test is now almost the same as test-paywall-flow,
 * since we can't initiate payment via a "claimed" private order status
 * response.
 */
export async function runPayPaidTest(t: GlobalTestState) {
  // Set up test environment

  const { walletClient, bank, faultyExchange, faultyMerchant } =
    await createFaultInjectedMerchantTestkudosEnvironment(t);

  // Withdraw digital cash into the wallet.

  const wres = await withdrawViaBankV2(t, {
    walletClient,
    bank,
    exchange: faultyExchange,
    amount: "TESTKUDOS:20",
  });

  await wres.withdrawalFinishedCond;

  /**
   * =========================================================================
   * Create an order and let the wallet pay under a session ID
   *
   * We check along the way that the JSON response to /orders/{order_id}
   * returns the right thing.
   * =========================================================================
   */

  const merchant = faultyMerchant;

  const merchantClient = new MerchantApiClient(merchant.makeInstanceBaseUrl());

  let orderResp = await merchantClient.createOrder({
    order: {
      summary: "Buy me!",
      amount: "TESTKUDOS:5",
      fulfillment_url: "https://example.com/article42",
      public_reorder_url: "https://example.com/article42-share",
    },
  });

  let orderStatus = await merchantClient.queryPrivateOrderStatus({
    orderId: orderResp.order_id,
    sessionId: "mysession-one",
  });

  t.assertTrue(orderStatus.order_status === "unpaid");

  t.assertTrue(orderStatus.already_paid_order_id === undefined);
  let publicOrderStatusUrl = orderStatus.order_status_url;

  let publicOrderStatusResp = await harnessHttpLib.fetch(publicOrderStatusUrl);

  if (publicOrderStatusResp.status != 402) {
    throw Error(
      `expected status 402 (before claiming), but got ${publicOrderStatusResp.status}`,
    );
  }

  let pubUnpaidStatus = codecForMerchantOrderStatusUnpaid().decode(
    publicOrderStatusResp.json(),
  );

  console.log(pubUnpaidStatus);

  let preparePayResp = await walletClient.call(
    WalletApiOperation.PreparePayForUri,
    {
      talerPayUri: pubUnpaidStatus.taler_pay_uri,
    },
  );

  t.assertTrue(preparePayResp.status === PreparePayResultType.PaymentPossible);

  const proposalId = preparePayResp.proposalId;

  publicOrderStatusResp = await harnessHttpLib.fetch(publicOrderStatusUrl);

  if (publicOrderStatusResp.status != 402) {
    throw Error(
      `expected status 402 (after claiming), but got ${publicOrderStatusResp.status}`,
    );
  }

  pubUnpaidStatus = codecForMerchantOrderStatusUnpaid().decode(
    publicOrderStatusResp.json(),
  );

  const confirmPayRes = await walletClient.call(WalletApiOperation.ConfirmPay, {
    proposalId: proposalId,
  });

  t.assertTrue(confirmPayRes.type === ConfirmPayResultType.Done);

  publicOrderStatusResp = await harnessHttpLib.fetch(publicOrderStatusUrl);

  console.log(publicOrderStatusResp.json());

  if (publicOrderStatusResp.status != 200) {
    console.log(publicOrderStatusResp.json());
    throw Error(
      `expected status 200 (after paying), but got ${publicOrderStatusResp.status}`,
    );
  }

  /**
   * =========================================================================
   * Now change up the session ID and do payment re-play!
   * =========================================================================
   */

  orderStatus = await merchantClient.queryPrivateOrderStatus({
    orderId: orderResp.order_id,
    sessionId: "mysession-two",
  });

  console.log(
    "order status under mysession-two:",
    JSON.stringify(orderStatus, undefined, 2),
  );

  // Should be claimed (not paid!) because of a new session ID
  t.assertTrue(orderStatus.order_status === "claimed");

  let numPayRequested = 0;
  let numPaidRequested = 0;

  faultyMerchant.faultProxy.addFault({
    async modifyRequest(ctx: FaultInjectionRequestContext) {
      const url = new URL(ctx.requestUrl);
      if (url.pathname.endsWith("/pay")) {
        numPayRequested++;
      } else if (url.pathname.endsWith("/paid")) {
        numPaidRequested++;
      }
    },
  });

  let orderRespTwo = await merchantClient.createOrder({
    order: {
      summary: "Buy me!",
      amount: "TESTKUDOS:5",
      fulfillment_url: "https://example.com/article42",
      public_reorder_url: "https://example.com/article42-share",
    },
  });

  let orderStatusTwo = await merchantClient.queryPrivateOrderStatus({
    orderId: orderRespTwo.order_id,
    sessionId: "mysession-two",
  });

  t.assertTrue(orderStatusTwo.order_status === "unpaid");

  // Pay with new taler://pay URI, which should
  // have the new session ID!
  // Wallet should now automatically re-play payment.
  preparePayResp = await walletClient.call(
    WalletApiOperation.PreparePayForUri,
    {
      talerPayUri: orderStatusTwo.taler_pay_uri,
    },
  );

  t.assertTrue(preparePayResp.status === PreparePayResultType.AlreadyConfirmed);
  t.assertTrue(preparePayResp.paid);

  // Make sure the wallet is actually doing the replay properly.
  t.assertTrue(numPaidRequested == 1);
  t.assertTrue(numPayRequested == 0);
}

runPayPaidTest.suites = ["wallet"];
