/*
 This file is part of GNU Taler
 (C) 2021 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  MerchantApiClient,
  PreparePayResultType,
  TalerErrorCode,
  TransactionType,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { GlobalTestState } from "../harness/harness.js";
import {
  createSimpleTestkudosEnvironmentV2,
  withdrawViaBankV2,
} from "../harness/helpers.js";

export async function runDenomUnofferedTest(t: GlobalTestState) {
  // Set up test environment

  const { walletClient, bank, exchange, merchant } =
    await createSimpleTestkudosEnvironmentV2(t);

  // Withdraw digital cash into the wallet.

  const wres = await withdrawViaBankV2(t, {
    walletClient,
    bank,
    exchange,
    amount: "TESTKUDOS:20",
  });

  await wres.withdrawalFinishedCond;

  // Make the exchange forget the denomination.
  // Effectively we completely reset the exchange,
  // but keep the exchange master public key.

  await merchant.stop();

  await exchange.stop();
  await exchange.purgeDatabase();
  await exchange.purgeSecmodKeys();
  await exchange.start();
  await exchange.pingUntilAvailable();

  await merchant.start();
  await merchant.pingUntilAvailable();

  const order = {
    summary: "Buy me!",
    amount: "TESTKUDOS:5",
    fulfillment_url: "taler://fulfillment-success/thx",
  };

  const merchantClient = new MerchantApiClient(merchant.makeInstanceBaseUrl());

  const orderResp = await merchantClient.createOrder({
    order: order,
  });

  let orderStatus = await merchantClient.queryPrivateOrderStatus({
    orderId: orderResp.order_id,
  });

  t.assertTrue(orderStatus.order_status === "unpaid");

  // Make wallet pay for the order

  const preparePayResult = await walletClient.call(
    WalletApiOperation.PreparePayForUri,
    {
      talerPayUri: orderStatus.taler_pay_uri,
    },
  );

  t.assertTrue(
    preparePayResult.status === PreparePayResultType.PaymentPossible,
  );

  const confirmResp = await walletClient.call(WalletApiOperation.ConfirmPay, {
    transactionId: preparePayResult.transactionId,
  });

  const tx = await walletClient.call(WalletApiOperation.GetTransactionById, {
    transactionId: confirmResp.transactionId,
  });

  t.assertTrue(tx.error != null);
  t.assertTrue(
    tx.error.code === TalerErrorCode.WALLET_PAY_MERCHANT_SERVER_ERROR,
  );

  const merchantErrorCode = (tx.error as any).requestError.errorResponse.code;

  t.assertDeepEqual(
    merchantErrorCode,
    TalerErrorCode.MERCHANT_GENERIC_EXCHANGE_UNEXPECTED_STATUS,
  );

  const exchangeErrorCode = (tx.error as any).requestError.errorResponse
    .exchange_ec;

  t.assertDeepEqual(
    exchangeErrorCode,
    TalerErrorCode.EXCHANGE_GENERIC_DENOMINATION_KEY_UNKNOWN,
  );

  // Depending on whether the merchant has seen the new denominations or not,
  // the error code might be different here.
  // t.assertDeepEqual(
  //   merchantErrorCode,
  //   TalerErrorCode.MERCHANT_POST_ORDERS_ID_PAY_DENOMINATION_KEY_NOT_FOUND,
  // );

  // Force updating the exchange entry so that the wallet knows about the new denominations.
  await walletClient.call(WalletApiOperation.UpdateExchangeEntry, {
    exchangeBaseUrl: exchange.baseUrl,
    force: true,
  });

  await walletClient.call(WalletApiOperation.DeleteTransaction, {
    transactionId: confirmResp.transactionId,
  });

  // Now withdrawal should work again.
  await withdrawViaBankV2(t, {
    walletClient,
    bank,
    exchange,
    amount: "TESTKUDOS:20",
  });

  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});

  const txs = await walletClient.call(WalletApiOperation.GetTransactions, {
    sort: "stable-ascending",
    includeRefreshes: true,
  });
  console.log(JSON.stringify(txs, undefined, 2));

  t.assertDeepEqual(txs.transactions[0].type, TransactionType.Withdrawal);
  t.assertDeepEqual(txs.transactions[1].type, TransactionType.Refresh);
  t.assertDeepEqual(txs.transactions[2].type, TransactionType.DenomLoss);
  t.assertDeepEqual(txs.transactions[3].type, TransactionType.Withdrawal);
}

runDenomUnofferedTest.suites = ["wallet"];
