/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  TalerCorebankApiClient,
  CreditDebitIndicator,
  WireGatewayApiClient,
  createEddsaKeyPair,
  encodeCrock,
} from "@gnu-taler/taler-util";
import { defaultCoinConfig } from "../harness/denomStructures.js";
import {
  BankService,
  ExchangeService,
  GlobalTestState,
  MerchantService,
  generateRandomPayto,
  setupDb,
} from "../harness/harness.js";

/**
 * Run test for basic, bank-integrated withdrawal.
 */
export async function runBankApiTest(t: GlobalTestState) {
  // Set up test environment

  const db = await setupDb(t);

  const bank = await BankService.create(t, {
    currency: "TESTKUDOS",
    httpPort: 8082,
    database: db.connStr,
    allowRegistrations: true,
  });

  const exchange = ExchangeService.create(t, {
    name: "testexchange-1",
    currency: "TESTKUDOS",
    httpPort: 8081,
    database: db.connStr,
  });

  const merchant = await MerchantService.create(t, {
    name: "testmerchant-1",
    currency: "TESTKUDOS",
    httpPort: 8083,
    database: db.connStr,
  });

  const exchangeBankAccount = await bank.createExchangeAccount(
    "myexchange",
    "x",
  );
  exchange.addBankAccount("1", exchangeBankAccount);

  bank.setSuggestedExchange(exchange, exchangeBankAccount.accountPaytoUri);

  await bank.start();

  await bank.pingUntilAvailable();

  exchange.addOfferedCoins(defaultCoinConfig);

  await exchange.start();
  await exchange.pingUntilAvailable();

  merchant.addExchange(exchange);

  await merchant.start();
  await merchant.pingUntilAvailable();

  await merchant.addInstanceWithWireAccount({
    id: "default",
    name: "Default Instance",
    paytoUris: [generateRandomPayto("merchant-default")],
  });

  await merchant.addInstanceWithWireAccount({
    id: "minst1",
    name: "minst1",
    paytoUris: [generateRandomPayto("minst1")],
  });

  console.log("setup done!");

  const bankClient = new TalerCorebankApiClient(bank.corebankApiBaseUrl);

  const bankUser = await bankClient.registerAccount("user1", "pw1");

  // Make sure that registering twice results in a 409 Conflict
  {
    const e = await t.assertThrowsTalerErrorAsync(async () => {
      await bankClient.registerAccount("user1", "pw2");
    });
    t.assertTrue(e.errorDetail.httpStatusCode === 409);
  }

  let balResp = await bankClient.getAccountBalance(bankUser.username);

  console.log(balResp);

  // Check that we got the sign-up bonus.
  t.assertAmountEquals(balResp.balance.amount, "TESTKUDOS:100");
  t.assertTrue(
    balResp.balance.credit_debit_indicator === CreditDebitIndicator.Credit,
  );

  const res = createEddsaKeyPair();

  const wireGatewayApiClient = new WireGatewayApiClient(
    exchangeBankAccount.wireGatewayApiBaseUrl,
    {
      auth: {
        username: exchangeBankAccount.accountName,
        password: exchangeBankAccount.accountPassword,
      },
    },
  );

  await wireGatewayApiClient.adminAddIncoming({
    amount: "TESTKUDOS:115",
    debitAccountPayto: bankUser.accountPaytoUri,
    reservePub: encodeCrock(res.eddsaPub),
  });

  balResp = await bankClient.getAccountBalance(bankUser.username);
  t.assertAmountEquals(balResp.balance.amount, "TESTKUDOS:15");
  t.assertTrue(
    balResp.balance.credit_debit_indicator === CreditDebitIndicator.Debit,
  );
}

runBankApiTest.suites = ["fakebank"]