/*
 This file is part of GNU Taler
 (C) 2022-2023 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import { AmountString, MerchantApiClient } from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { defaultCoinConfig } from "../harness/denomStructures.js";
import { GlobalTestState } from "../harness/harness.js";
import {
  createSimpleTestkudosEnvironmentV2,
  createWalletDaemonWithClient,
  makeTestPaymentV2,
  withdrawViaBankV2,
} from "../harness/helpers.js";

/**
 * Run test for basic, bank-integrated withdrawal and payment.
 */
export async function runAgeRestrictionsMerchantTest(t: GlobalTestState) {
  // Set up test environment

  const {
    walletClient: walletClientOne,
    bank,
    exchange,
    merchant,
    exchangeBankAccount,
  } = await createSimpleTestkudosEnvironmentV2(
    t,
    defaultCoinConfig.map((x) => x("TESTKUDOS")),
    {
      ageMaskSpec: "8:10:12:14:16:18:21",
    },
  );

  const { walletClient: walletClientTwo } = await createWalletDaemonWithClient(
    t,
    {
      name: "w2",
    },
  );

  const { walletClient: walletClientThree } =
    await createWalletDaemonWithClient(t, {
      name: "w3",
    });

  {
    const { walletClient: walletClientZero } =
      await createWalletDaemonWithClient(t, {
        name: "w0",
      });

    const wres = await withdrawViaBankV2(t, {
      walletClient: walletClientZero,
      bank,
      exchange,
      amount: "TESTKUDOS:20" as AmountString,
      restrictAge: 13,
    });
    await wres.withdrawalFinishedCond;

    const order = {
      summary: "Buy me!",
      amount: "TESTKUDOS:5",
      fulfillment_url: "taler://fulfillment-success/thx",
      minimum_age: 9,
    };

    await makeTestPaymentV2(t, {
      walletClient: walletClientZero,
      merchant,
      order,
    });
    await walletClientZero.call(
      WalletApiOperation.TestingWaitTransactionsFinal,
      {},
    );
  }

  {
    const walletClient = walletClientOne;

    const wres = await withdrawViaBankV2(t, {
      walletClient,
      bank,
      exchange,
      amount: "TESTKUDOS:20" as AmountString,
      restrictAge: 13,
    });
    await wres.withdrawalFinishedCond;

    const order = {
      summary: "Buy me!",
      amount: "TESTKUDOS:5",
      fulfillment_url: "taler://fulfillment-success/thx",
      minimum_age: 9,
    };

    await makeTestPaymentV2(t, { walletClient, merchant, order });
    await walletClient.call(
      WalletApiOperation.TestingWaitTransactionsFinal,
      {},
    );
  }

  {
    const walletClient = walletClientTwo;

    const wres = await withdrawViaBankV2(t, {
      walletClient,
      bank,
      exchange,
      amount: "TESTKUDOS:20" as AmountString,
      restrictAge: 13,
    });
    await wres.withdrawalFinishedCond;

    const order = {
      summary: "Buy me!",
      amount: "TESTKUDOS:5",
      fulfillment_url: "taler://fulfillment-success/thx",
    };

    await makeTestPaymentV2(t, { walletClient, merchant, order });
    await walletClient.call(
      WalletApiOperation.TestingWaitTransactionsFinal,
      {},
    );
  }

  {
    const walletClient = walletClientThree;

    const wres = await withdrawViaBankV2(t, {
      walletClient,
      bank,
      exchange,
      amount: "TESTKUDOS:20" as AmountString,
    });
    await wres.withdrawalFinishedCond;

    const order = {
      summary: "Buy me!",
      amount: "TESTKUDOS:5",
      fulfillment_url: "taler://fulfillment-success/thx",
      minimum_age: 9,
    };

    await makeTestPaymentV2(t, { walletClient, merchant, order });
    await walletClient.call(
      WalletApiOperation.TestingWaitTransactionsFinal,
      {},
    );
  }
}

runAgeRestrictionsMerchantTest.suites = ["wallet"];
runAgeRestrictionsMerchantTest.timeoutMs = 120 * 1000;
