/*
 This file is part of GNU Taler
 (C) 2021-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 *
 * @author Sebastian Javier Marchano (sebasjm)
 */

import {
  FacadeCredentials,
  HttpStatusCode,
  OperationFail,
  OperationOk,
  TalerError,
  TalerMerchantApi,
  TalerRevenueHttpClient,
  assertUnreachable,
  opFixedSuccess,
} from "@gnu-taler/taler-util";
import {
  BrowserFetchHttpLib,
  useMerchantApiContext,
  useTranslationContext,
} from "@gnu-taler/web-util/browser";
import { Fragment, h, VNode } from "preact";
import { useState } from "preact/hooks";
import { NotificationCard } from "../../../../components/menu/index.js";
import { Notification } from "../../../../utils/types.js";
import { CreatePage } from "./CreatePage.js";
import { useSessionContext } from "../../../../context/session.js";

export type Entity = TalerMerchantApi.AccountAddDetails;
interface Props {
  onBack?: () => void;
  onConfirm: () => void;
}

export default function CreateValidator({ onConfirm, onBack }: Props): VNode {
  const { lib: api } = useMerchantApiContext();
  const { state } = useSessionContext();
  const [notif, setNotif] = useState<Notification | undefined>(undefined);
  const { i18n } = useTranslationContext();

  return (
    <>
      <NotificationCard notification={notif} />
      <CreatePage
        onBack={onBack}
        onCreate={async (request: Entity) => {
          const revenueAPI = !request.credit_facade_url
            ? undefined
            : new URL("./", request.credit_facade_url);

          if (revenueAPI) {
            const resp = await testRevenueAPI(
              revenueAPI,
              request.credit_facade_credentials,
            );
            if (resp.type === "fail") {
              switch (resp.case) {
                case TestRevenueErrorType.NO_CONFIG: {
                  setNotif({
                    message: i18n.str`Could not create account`,
                    type: "ERROR",
                    description: i18n.str`The endpoint doesn't seems to be a Taler Revenue API`,
                  });
                  return;
                }
                case TestRevenueErrorType.CLIENT_BAD_REQUEST: {
                  setNotif({
                    message: i18n.str`Could not create account`,
                    type: "ERROR",
                    description: i18n.str`Server replied with "bad request".`,
                  });
                  return;
                }
                case TestRevenueErrorType.UNAUTHORIZED: {
                  setNotif({
                    message: i18n.str`Could not create account`,
                    type: "ERROR",
                    description: i18n.str`Unauthorized, try with another credentials.`,
                  });
                  return;
                }
                case TestRevenueErrorType.NOT_FOUND: {
                  setNotif({
                    message: i18n.str`Could not create account`,
                    type: "ERROR",
                    description: i18n.str`Check facade URL, server replied with "not found".`,
                  });
                  return;
                }
                case TestRevenueErrorType.GENERIC_ERROR: {
                  setNotif({
                    message: i18n.str`Could not create account`,
                    type: "ERROR",
                    description: resp.detail.hint,
                  });
                  return;
                }
                default: {
                  assertUnreachable(resp.case);
                }
              }
            }
          }

          return api.instance
            .addBankAccount(state.token, request)
            .then(() => {
              onConfirm();
            })
            .catch((error) => {
              setNotif({
                message: i18n.str`could not create account`,
                type: "ERROR",
                description: error.message,
              });
            });
        }}
      />
    </>
  );
}

export enum TestRevenueErrorType {
  NO_CONFIG,
  CLIENT_BAD_REQUEST,
  UNAUTHORIZED,
  NOT_FOUND,
  GENERIC_ERROR,
}

export async function testRevenueAPI(
  revenueAPI: URL,
  creds: FacadeCredentials | undefined,
): Promise<OperationOk<void> | OperationFail<TestRevenueErrorType>> {
  const api = new TalerRevenueHttpClient(
    revenueAPI.href,
    new BrowserFetchHttpLib(),
  );
  const auth =
    creds === undefined
      ? undefined
      : creds.type === "none"
        ? undefined
        : creds.type === "basic"
          ? {
              username: creds.username,
              password: creds.password,
            }
          : undefined;

  try {
    const config = await api.getConfig(auth);

    if (config.type === "fail") {
      switch (config.case) {
        case HttpStatusCode.Unauthorized: {
          return {
            type: "fail",
            case: TestRevenueErrorType.UNAUTHORIZED,
            detail: {
              code: 1,
            },
          };
        }
        case HttpStatusCode.NotFound: {
          return {
            type: "fail",
            case: TestRevenueErrorType.NO_CONFIG,
            detail: {
              code: 1,
            },
          };
        }
      }
    }

    const history = await api.getHistory(auth);

    if (history.type === "fail") {
      switch (history.case) {
        case HttpStatusCode.BadRequest: {
          return {
            type: "fail",
            case: TestRevenueErrorType.CLIENT_BAD_REQUEST,
            detail: {
              code: 1,
            },
          };
        }
        case HttpStatusCode.Unauthorized: {
          return {
            type: "fail",
            case: TestRevenueErrorType.UNAUTHORIZED,
            detail: {
              code: 1,
            },
          };
        }
        case HttpStatusCode.NotFound: {
          return {
            type: "fail",
            case: TestRevenueErrorType.NOT_FOUND,
            detail: {
              code: 1,
            },
          };
        }
      }
    }
  } catch (err) {
    if (err instanceof TalerError) {
      return {
        type: "fail",
        case: TestRevenueErrorType.GENERIC_ERROR,
        detail: err.errorDetail,
      };
    }
  }

  return opFixedSuccess(undefined);
}
