/*
 This file is part of GNU Taler
 (C) 2021-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
import {
  useMerchantApiContext
} from "@gnu-taler/web-util/browser";
import { PAGINATED_LIST_REQUEST } from "../utils/constants.js";

// FIX default import https://github.com/microsoft/TypeScript/issues/49189
import { AccessToken, TalerHttpError, TalerMerchantManagementResultByMethod } from "@gnu-taler/taler-util";
import _useSWR, { SWRHook, mutate } from "swr";
import { useSessionContext } from "../context/session.js";
import { buildPaginatedResult } from "./webhooks.js";
const useSWR = _useSWR as unknown as SWRHook;

export interface InstanceTransferFilter {
  payto_uri?: string;
  verified?: boolean;
  position?: string;
}

export function revalidateInstanceTransfers() {
  return mutate(
    (key) => Array.isArray(key) && key[key.length - 1] === "listWireTransfers",
    undefined,
    { revalidate: true },
  );
}
export function useInstanceTransfers(
  args?: InstanceTransferFilter,
  updatePosition: (id: string | undefined) => void = (() => { }),
) {
  const { state: session } = useSessionContext();
  const { lib: { instance } } = useMerchantApiContext();

  // const [offset, setOffset] = useState<string | undefined>(args?.position);

  async function fetcher([token, o, p, v]: [AccessToken, string, string, boolean]) {
    return await instance.listWireTransfers(token, {
      paytoURI: p,
      verified: v,
      limit: PAGINATED_LIST_REQUEST,
      offset: o,
      order: "dec",
    });
  }

  const { data, error } = useSWR<
    TalerMerchantManagementResultByMethod<"listWireTransfers">,
    TalerHttpError
  >([session.token, args?.position, args?.payto_uri, args?.verified, "listWireTransfers"], fetcher);

  if (error) return error;
  if (data === undefined) return undefined;
  if (data.type !== "ok") return data;

  return buildPaginatedResult(data.body.transfers, args?.position, updatePosition, (d) => String(d.transfer_serial_id))

}
