import { TranslatedString } from "@gnu-taler/taler-util";
import {
  createNewForm,
  notifyError,
  useTranslationContext,
} from "@gnu-taler/web-util/browser";
import { VNode, h } from "preact";
import { useSettings } from "../hooks/useSettings.js";

export function CreateAccount({
  onNewAccount,
}: {
  onNewAccount: (password: string) => void;
}): VNode {
  const { i18n } = useTranslationContext();
  const Form = createNewForm<{
    password: string;
    repeat: string;
  }>();
  const [settings] = useSettings()

  return (
    <div class="flex min-h-full flex-col ">
      <div class="sm:mx-auto sm:w-full sm:max-w-md">
        <h2 class="mt-6 text-center text-2xl font-bold leading-9 tracking-tight text-gray-900">
          <i18n.Translate>Create account</i18n.Translate>
        </h2>
      </div>

      <div class="mt-10 sm:mx-auto sm:w-full sm:max-w-[480px] ">
        <div class="bg-gray-100 px-6 py-6 shadow sm:rounded-lg sm:px-12">
          <Form.Provider
            computeFormState={(v) => {
              return {
                password: {
                  error: !v.password
                    ? i18n.str`required`
                    : settings.allowInsecurePassword
                      ? undefined
                      : v.password.length < 8
                        ? i18n.str`should have at least 8 characters`
                        : !v.password.match(/[a-z]/) && v.password.match(/[A-Z]/)
                          ? i18n.str`should have lowercase and uppercase characters`
                          : !v.password.match(/\d/)
                            ? i18n.str`should have numbers`
                            : !v.password.match(/[^a-zA-Z\d]/)
                              ? i18n.str`should have at least one character which is not a number or letter`
                              : undefined,
                },
                repeat: {
                  error: !v.repeat
                    ? i18n.str`required`
                    : v.repeat !== v.password
                      ? i18n.str`doesn't match`
                      : undefined,
                },
              };
            }}
            onSubmit={async (v, s) => {
              const error = s?.password?.error ?? s?.repeat?.error;
              if (error) {
                notifyError(
                  i18n.str`Can't create account`,
                  error as TranslatedString,
                );
              } else {
                onNewAccount(v.password!);
              }
            }}
          >
            <div class="mb-4">
              <Form.InputLine
                label={i18n.str`Password`}
                name="password"
                type="password"
                help={
                  settings.allowInsecurePassword
                    ? i18n.str`short password are insecure, turn off insecure password in settings`
                    : i18n.str`lower and upper case letters, number and special character`
                }
                required
              />
            </div>
            <div class="mb-4">
              <Form.InputLine
                label={i18n.str`Repeat password`}
                name="repeat"
                type="password"
                required
              />
            </div>

            <div class="mt-8">
              <button
                type="submit"
                class="flex w-full justify-center rounded-md bg-indigo-600 px-3 py-1.5 text-sm font-semibold leading-6 text-white shadow-sm hover:bg-indigo-500 focus-visible:outline focus-visible:outline-2 focus-visible:outline-offset-2 focus-visible:outline-indigo-600"
              >
                <i18n.Translate>Create</i18n.Translate>
              </button>
            </div>
          </Form.Provider>
        </div>
      </div>
    </div>
  );
}
