import { useState } from "preact/hooks";

// FIX default import https://github.com/microsoft/TypeScript/issues/49189
import { OfficerAccount, OfficerId, OperationOk, TalerExchangeResultByMethod, TalerHttpError, decodeCrock, encodeCrock } from "@gnu-taler/taler-util";
import _useSWR, { SWRHook } from "swr";
import { useExchangeApiContext } from "../context/config.js";
import { AmlExchangeBackend } from "../utils/types.js";
import { useOfficer } from "./useOfficer.js";
const useSWR = _useSWR as unknown as SWRHook;

export const PAGINATED_LIST_SIZE = 10;
// when doing paginated request, ask for one more
// and use it to know if there are more to request
export const PAGINATED_LIST_REQUEST = PAGINATED_LIST_SIZE + 1;

/**
 * FIXME: mutate result when balance change (transaction )
 * @param account
 * @param args
 * @returns
 */
export function useCases(state: AmlExchangeBackend.AmlState) {
  const officer = useOfficer();
  const session = officer.state === "ready" ? officer.account : undefined;
  const { api } = useExchangeApiContext();

  const [offset, setOffset] = useState<string>();

  async function fetcher([officer, state, offset]: [OfficerAccount, AmlExchangeBackend.AmlState, string | undefined]) {
    return await api.getDecisionsByState(officer, state, {
      order: "asc", offset, limit: PAGINATED_LIST_REQUEST
    })
  }

  const { data, error } = useSWR<TalerExchangeResultByMethod<"getDecisionsByState">, TalerHttpError>(
    !session ? undefined : [session, state, offset, "getDecisionsByState"],
    fetcher,
  );

  if (error) return error;
  if (data === undefined) return undefined;
  if (data.type !== "ok") return data;

  return buildPaginatedResult(data.body.records, offset, setOffset, (d) => String(d.rowid));
}

type PaginatedResult<T> = OperationOk<T> & {
  isLastPage: boolean;
  isFirstPage: boolean;
  loadNext(): void;
  loadFirst(): void;
}

//TODO: consider sending this to web-util
export function buildPaginatedResult<R, OffId>(data: R[], offset: OffId | undefined, setOffset: (o: OffId | undefined) => void, getId: (r: R) => OffId): PaginatedResult<R[]> {

  const isLastPage = data.length < PAGINATED_LIST_REQUEST;
  const isFirstPage = offset === undefined;

  const result = structuredClone(data);
  if (result.length == PAGINATED_LIST_REQUEST) {
    result.pop();
  }
  return {
    type: "ok",
    body: result,
    isLastPage,
    isFirstPage,
    loadNext: () => {
      if (!result.length) return;
      const id = getId(result[result.length - 1])
      setOffset(id);
    },
    loadFirst: () => {
      setOffset(undefined);
    },
  };
}


function removeLastElement<T>(list: Array<T>): Array<T> {
  if (list.length === 0) {
    return list;
  }
  return list.slice(0, -1)
}