/*
 This file is part of GNU Taler
 (C) 2021-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 *
 * @author Sebastian Javier Marchano (sebasjm)
 */

import { HttpStatusCode, TalerMerchantApi } from "@gnu-taler/taler-util";
import {
  ButtonBetterBulma,
  LocalNotificationBannerBulma,
  SafeHandlerTemplate,
  useLocalNotificationBetter,
  useTranslationContext,
} from "@gnu-taler/web-util/browser";
import { Fragment, h, VNode } from "preact";
import { StateUpdater, useState } from "preact/hooks";
import { useSessionContext } from "../../../../context/session.js";

const TALER_SCREEN_ID = 38;

type Entity = TalerMerchantApi.CategoryListEntry;

interface Props {
  devices: Entity[];
  onSelect: (e: Entity) => void;
  onCreate: () => void;
  onLoadMoreBefore?: () => void;
  onLoadMoreAfter?: () => void;
}

export function CardTable({
  devices,
  onCreate,
  onSelect,
  onLoadMoreAfter,
  onLoadMoreBefore,
}: Props): VNode {
  const [rowSelection, rowSelectionHandler] = useState<string[]>([]);

  const { i18n } = useTranslationContext();
  const { state: session, lib } = useSessionContext();
  const [notification, safeFunctionHandler] = useLocalNotificationBetter();

  const remove = safeFunctionHandler(
    lib.instance.deleteCategory.bind(lib.instance),
  ).lambda((id: string) => (!session.token ? undefined! : [session.token, id]));
  remove.onSuccess = () => i18n.str`Category deleted`;
  remove.onFail = (fail) => {
    switch (fail.case) {
      case HttpStatusCode.Unauthorized:
        return i18n.str`Unauthorized`;
      case HttpStatusCode.NotFound:
        return i18n.str`Not found`;
    }
  };
  return (
    <Fragment>
      <LocalNotificationBannerBulma notification={notification} />

      <div class="card has-table">
        <header class="card-header">
          <p class="card-header-title">
            <span class="icon">
              <i class="mdi mdi-label" />
            </span>
            <i18n.Translate>Categories</i18n.Translate>
          </p>
          <div class="card-header-icon" aria-label="more options">
            <span
              class="has-tooltip-left"
              data-tooltip={i18n.str`Add new devices`}
            >
              <button class="button is-info" accessKey="+" type="button" onClick={onCreate}>
                <span class="icon is-small">
                  <i class="mdi mdi-plus mdi-36px" />
                </span>
              </button>
            </span>
          </div>
        </header>
        <div class="card-content">
          <div class="b-table has-pagination">
            <div class="table-wrapper has-mobile-cards">
              {devices.length > 0 ? (
                <Table
                  instances={devices}
                  onDelete={remove}
                  onSelect={onSelect}
                  rowSelection={rowSelection}
                  rowSelectionHandler={rowSelectionHandler}
                  onLoadMoreAfter={onLoadMoreAfter}
                  onLoadMoreBefore={onLoadMoreBefore}
                />
              ) : (
                <EmptyTable />
              )}
            </div>
          </div>
        </div>
      </div>
    </Fragment>
  );
}
interface TableProps {
  rowSelection: string[];
  instances: Entity[];
  onDelete: SafeHandlerTemplate<[id: string], unknown>;
  onSelect: (e: Entity) => void;
  rowSelectionHandler: StateUpdater<string[]>;
  onLoadMoreBefore?: () => void;
  onLoadMoreAfter?: () => void;
}

function Table({
  instances,
  onLoadMoreAfter,
  onDelete,
  onSelect,
  onLoadMoreBefore,
}: TableProps): VNode {
  const { i18n } = useTranslationContext();
  return (
    <div class="table-container">
      {onLoadMoreBefore && (
        <button type="button" 
          class="button is-fullwidth"
          data-tooltip={i18n.str`Load more devices before the first one`}
          onClick={onLoadMoreBefore}
        >
          <i18n.Translate>Load first page</i18n.Translate>
        </button>
      )}
      <table class="table is-fullwidth is-striped is-hoverable is-fullwidth">
        <thead>
          <tr>
            <th>
              <i18n.Translate>ID</i18n.Translate>
            </th>
            <th>
              <i18n.Translate>Name</i18n.Translate>
            </th>
            <th>
              <i18n.Translate>Total products</i18n.Translate>
            </th>
            <th />
          </tr>
        </thead>
        <tbody>
          {instances.map((i) => {
            return (
              <tr key={i.category_id}>
                <td
                  onClick={(): void => onSelect(i)}
                  style={{ cursor: "pointer" }}
                >
                  {i.category_id}
                </td>
                <td
                  onClick={(): void => onSelect(i)}
                  style={{ cursor: "pointer" }}
                >
                  {i.name}
                </td>
                <td
                  onClick={(): void => onSelect(i)}
                  style={{ cursor: "pointer" }}
                >
                  {i.product_count}
                </td>
                <td class="is-actions-cell right-sticky">
                  <div class="buttons is-right">
                    <ButtonBetterBulma
                      class="button is-danger is-small has-tooltip-left"
                      data-tooltip={i18n.str`Delete selected category from the database`}
                      onClick={onDelete.withArgs(String(i.category_id))}
                    >
                      <i18n.Translate>Delete</i18n.Translate>
                    </ButtonBetterBulma>
                  </div>
                </td>
              </tr>
            );
          })}
        </tbody>
      </table>
      {onLoadMoreAfter && (
        <button type="button" 
          class="button is-fullwidth"
          data-tooltip={i18n.str`Load more devices after the last one`}
          onClick={onLoadMoreAfter}
        >
          <i18n.Translate>Load next page</i18n.Translate>
        </button>
      )}
    </div>
  );
}

function EmptyTable(): VNode {
  const { i18n } = useTranslationContext();
  return (
    <div class="content has-text-grey has-text-centered">
      <p>
        <span class="icon is-large">
          <i class="mdi mdi-magnify mdi-48px" />
        </span>
      </p>
      <p>
        <i18n.Translate>
          There are no categories yet, add more pressing the + sign
        </i18n.Translate>
      </p>
    </div>
  );
}
