/*
   This file is part of TALER
   Copyright (C) 2022 Taler Systems SA

   TALER is free software; you can redistribute it and/or modify it under the
   terms of the GNU General Public License as published by the Free Software
   Foundation; either version 3, or (at your option) any later version.

   TALER is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
   A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along with
   TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
/**
 * @file backenddb/pg_lookup_product.c
 * @brief Implementation of the lookup_product function for Postgres
 * @author Iván Ávalos
 */
#include "platform.h"
#include <taler/taler_error_codes.h>
#include <taler/taler_dbevents.h>
#include <taler/taler_pq_lib.h>
#include "pg_lookup_product.h"
#include "pg_helper.h"

enum GNUNET_DB_QueryStatus
TMH_PG_lookup_product (void *cls,
                       const char *instance_id,
                       const char *product_id,
                       struct TALER_MERCHANTDB_ProductDetails *pd)
{
  struct PostgresClosure *pg = cls;
  struct GNUNET_PQ_QueryParam params[] = {
    GNUNET_PQ_query_param_string (instance_id),
    GNUNET_PQ_query_param_string (product_id),
    GNUNET_PQ_query_param_end
  };

  if (NULL == pd)
  {
    struct GNUNET_PQ_ResultSpec rs_null[] = {
      GNUNET_PQ_result_spec_end
    };

    check_connection (pg);
    return GNUNET_PQ_eval_prepared_singleton_select (pg->conn,
                                                     "lookup_product",
                                                     params,
                                                     rs_null);
  }
  else
  {
    struct GNUNET_PQ_ResultSpec rs[] = {
      GNUNET_PQ_result_spec_string ("description",
                                    &pd->description),
      TALER_PQ_result_spec_json ("description_i18n",
                                 &pd->description_i18n),
      GNUNET_PQ_result_spec_string ("unit",
                                    &pd->unit),
      TALER_PQ_result_spec_amount_with_currency ("price",
                                   &pd->price),
      TALER_PQ_result_spec_json ("taxes",
                                 &pd->taxes),
      GNUNET_PQ_result_spec_uint64 ("total_stock",
                                    &pd->total_stock),
      GNUNET_PQ_result_spec_uint64 ("total_sold",
                                    &pd->total_sold),
      GNUNET_PQ_result_spec_uint64 ("total_lost",
                                    &pd->total_lost),
      GNUNET_PQ_result_spec_string ("image",
                                    &pd->image),
      TALER_PQ_result_spec_json ("address",
                                 &pd->address),
      GNUNET_PQ_result_spec_timestamp ("next_restock",
                                       &pd->next_restock),
      GNUNET_PQ_result_spec_uint32 ("minimum_age",
                                    &pd->minimum_age),
      GNUNET_PQ_result_spec_end
    };

    check_connection (pg);
    PREPARE (pg,
             "lookup_product",
             "SELECT"
             " description"
             ",description_i18n"
             ",unit"
             ",price"
             ",taxes"
             ",total_stock"
             ",total_sold"
             ",total_lost"
             ",image"
             ",merchant_inventory.address"
             ",next_restock"
             ",minimum_age"
             " FROM merchant_inventory"
             " JOIN merchant_instances"
             "   USING (merchant_serial)"
             " WHERE merchant_instances.merchant_id=$1"
             "   AND merchant_inventory.product_id=$2");
    return GNUNET_PQ_eval_prepared_singleton_select (pg->conn,
                                                     "lookup_product",
                                                     params,
                                                     rs);
  }
}
