/*
  This file is part of TALER
  Copyright (C) 2014-2020 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as
  published by the Free Software Foundation; either version 3, or
  (at your option) any later version.

  TALER is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public
  License along with TALER; see the file COPYING.  If not, see
  <http://www.gnu.org/licenses/>
*/
/**
 * @file testing_api_cmd_wallet_get_reward.c
 * @brief command to test the rewardping.
 * @author Marcello Stanisci
 */
#include "platform.h"
#include <taler/taler_exchange_service.h>
#include <taler/taler_testing_lib.h>
#include "taler_merchant_service.h"
#include "taler_merchant_testing_lib.h"


/**
 * State for a GET /rewards/$REWARD_ID CMD.
 */
struct WalletRewardGetState
{

  /**
   * The merchant base URL.
   */
  const char *merchant_url;

  /**
   * Expected HTTP response code for this CMD.
   */
  unsigned int http_status;

  /**
   * Whether to compare amounts or not.
   */
  bool cmp_amounts;

  /**
   * The expected amount remaining.
   */
  struct TALER_Amount amount_remaining;

  /**
   * The handle to the current GET /rewards/$REWARD_ID request.
   */
  struct TALER_MERCHANT_RewardWalletGetHandle *tgh;

  /**
   * The interpreter state.
   */
  struct TALER_TESTING_Interpreter *is;

  /**
   * Reference to a command that created a reward.
   */
  const char *reward_reference;
};


/**
 * Callback to process a GET /rewards/$REWARD_ID request, it mainly
 * checks that what the backend returned matches the command's
 * expectations.
 *
 * @param cls closure
 * @param wgr response
 */
static void
wallet_reward_get_cb (void *cls,
                      const struct TALER_MERCHANT_RewardWalletGetResponse *wgr)
{
  struct WalletRewardGetState *gts = cls;
  const struct TALER_TESTING_Command *reward_cmd;

  reward_cmd = TALER_TESTING_interpreter_lookup_command (
    gts->is,
    gts->reward_reference);

  gts->tgh = NULL;
  if (gts->http_status != wgr->hr.http_status)
  {
    GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                "Unexpected response code %u (%d) to command %s\n",
                wgr->hr.http_status,
                (int) wgr->hr.ec,
                TALER_TESTING_interpreter_get_current_label (gts->is));
    TALER_TESTING_interpreter_fail (gts->is);
    return;
  }
  switch (wgr->hr.http_status)
  {
  case MHD_HTTP_OK:
    if (gts->cmp_amounts)
    {
      if ((GNUNET_OK !=
           TALER_amount_cmp_currency (&gts->amount_remaining,
                                      &wgr->details.ok.amount_remaining))
          ||
          (0 != TALER_amount_cmp (&gts->amount_remaining,
                                  &wgr->details.ok.amount_remaining)))
      {
        GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                    "Amount remaining on reward does not match\n");
        TALER_TESTING_interpreter_fail (gts->is);
        return;
      }
    }
    {
      const struct GNUNET_TIME_Timestamp *expiration;

      if (GNUNET_OK !=
          TALER_TESTING_get_trait_timestamp (reward_cmd,
                                             0,
                                             &expiration))
        TALER_TESTING_interpreter_fail (gts->is);
      if (GNUNET_TIME_timestamp_cmp (*expiration,
                                     !=,
                                     wgr->details.ok.expiration))
      {
        GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                    "Reward expiration does not match\n");
        TALER_TESTING_interpreter_fail (gts->is);
        return;
      }
    }
    break;
  default:
    GNUNET_log (GNUNET_ERROR_TYPE_WARNING,
                "Unhandled HTTP status.\n");
  }
  TALER_TESTING_interpreter_next (gts->is);
}


/**
 * Run the "GET reward" CMD.
 *
 * @param cls closure.
 * @param cmd command being run now.
 * @param is interpreter state.
 */
static void
wallet_get_reward_run (void *cls,
                       const struct TALER_TESTING_Command *cmd,
                       struct TALER_TESTING_Interpreter *is)
{
  struct WalletRewardGetState *tgs = cls;
  const struct TALER_TESTING_Command *reward_cmd;
  const struct TALER_RewardIdentifierP *reward_id;

  reward_cmd = TALER_TESTING_interpreter_lookup_command (is,
                                                         tgs->reward_reference);

  if (GNUNET_OK !=
      TALER_TESTING_get_trait_reward_id (reward_cmd,
                                         &reward_id))
    TALER_TESTING_FAIL (is);

  tgs->is = is;
  tgs->tgh = TALER_MERCHANT_wallet_reward_get (
    TALER_TESTING_interpreter_get_context (is),
    tgs->merchant_url,
    reward_id,
    &wallet_reward_get_cb,
    tgs);
}


/**
 * Free the state of a "GET reward" CMD, and possibly
 * cancel a pending operation thereof.
 *
 * @param cls closure.
 * @param cmd command being run.
 */
static void
wallet_get_reward_cleanup (void *cls,
                           const struct TALER_TESTING_Command *cmd)
{
  struct WalletRewardGetState *tgs = cls;

  if (NULL != tgs->tgh)
  {
    TALER_LOG_WARNING ("Get reward operation did not complete\n");
    TALER_MERCHANT_wallet_reward_get_cancel (tgs->tgh);
  }
  GNUNET_free (tgs);
}


struct TALER_TESTING_Command
TALER_TESTING_cmd_wallet_get_reward (const char *label,
                                     const char *merchant_url,
                                     const char *reward_reference,
                                     unsigned int http_status)
{
  struct WalletRewardGetState *tgs;

  tgs = GNUNET_new (struct WalletRewardGetState);
  tgs->merchant_url = merchant_url;
  tgs->reward_reference = reward_reference;
  tgs->http_status = http_status;
  {
    struct TALER_TESTING_Command cmd = {
      .cls = tgs,
      .label = label,
      .run = &wallet_get_reward_run,
      .cleanup = &wallet_get_reward_cleanup
    };

    return cmd;
  }
}


struct TALER_TESTING_Command
TALER_TESTING_cmd_wallet_get_reward2 (const char *label,
                                      const char *merchant_url,
                                      const char *reward_reference,
                                      const char *amount_remaining,
                                      unsigned int http_status)
{
  struct WalletRewardGetState *tgs;

  tgs = GNUNET_new (struct WalletRewardGetState);
  tgs->merchant_url = merchant_url;
  tgs->reward_reference = reward_reference;
  tgs->cmp_amounts = true;
  GNUNET_assert (GNUNET_OK == TALER_string_to_amount (amount_remaining,
                                                      &tgs->amount_remaining));
  tgs->http_status = http_status;
  {
    struct TALER_TESTING_Command cmd = {
      .cls = tgs,
      .label = label,
      .run = &wallet_get_reward_run,
      .cleanup = &wallet_get_reward_cleanup
    };

    return cmd;
  }
}


/* end of testing_api_cmd_wallet_get_reward.c */
