/*
  This file is part of TALER
  Copyright (C) 2020-2023 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as
  published by the Free Software Foundation; either version 3, or
  (at your option) any later version.

  TALER is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public
  License along with TALER; see the file COPYING.  If not, see
  <http://www.gnu.org/licenses/>
*/
/**
 * @file testing_api_cmd_get_rewards.c
 * @brief command to test GET /private/rewards
 * @author Jonathan Buchanan
 */
#include "platform.h"
#include <taler/taler_exchange_service.h>
#include <taler/taler_testing_lib.h>
#include "taler_merchant_service.h"
#include "taler_merchant_testing_lib.h"


/**
 * State of a "GET rewards" CMD.
 */
struct GetRewardsState
{

  /**
   * Handle for a "GET rewards" request.
   */
  struct TALER_MERCHANT_RewardsGetHandle *tgh;

  /**
   * The interpreter state.
   */
  struct TALER_TESTING_Interpreter *is;

  /**
   * Base URL of the merchant serving the request.
   */
  const char *merchant_url;

  /**
   * Row to start querying the database from.
   */
  uint64_t offset;

  /**
   * How many rows to return (with direction).
   */
  int64_t limit;

  /**
   * Expected HTTP response code.
   */
  unsigned int http_status;

  /**
   * Length of @e rewards.
   */
  unsigned int rewards_length;

  /**
   * References to rewards that we expect to be found.
   */
  const char **rewards;

};

/**
 * Callback for a GET /private/rewards operation.
 *
 * @param cls closure for this function
 * @param tgr response details
 */
static void
get_rewards_cb (void *cls,
                const struct TALER_MERCHANT_RewardsGetResponse *tgr)
{
  struct GetRewardsState *gts = cls;

  gts->tgh = NULL;
  if (gts->http_status != tgr->hr.http_status)
  {
    GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                "Unexpected response code %u (%d) to command %s\n",
                tgr->hr.http_status,
                (int) tgr->hr.ec,
                TALER_TESTING_interpreter_get_current_label (gts->is));
    TALER_TESTING_interpreter_fail (gts->is);
    return;
  }
  switch (tgr->hr.http_status)
  {
  case MHD_HTTP_OK:
    if (tgr->details.ok.rewards_length != gts->rewards_length)
    {
      GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                  "Rewards length does not match\n");
      TALER_TESTING_interpreter_fail (gts->is);
      return;
    }
    for (unsigned int i = 0; i < tgr->details.ok.rewards_length; ++i)
    {
      const struct TALER_MERCHANT_RewardEntry *reward
        = &tgr->details.ok.rewards[i];
      const struct TALER_TESTING_Command *reward_cmd;

      reward_cmd = TALER_TESTING_interpreter_lookup_command (
        gts->is,
        gts->rewards[i]);
      {
        const struct TALER_RewardIdentifierP *reward_id;

        if (GNUNET_OK !=
            TALER_TESTING_get_trait_reward_id (reward_cmd,
                                               &reward_id))
        {
          GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                      "Could not fetch reward id\n");
          TALER_TESTING_interpreter_fail (gts->is);
          return;
        }
        if (0 != GNUNET_memcmp (reward_id,
                                &reward->reward_id))
        {
          GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                      "Reward id does not match\n");
          TALER_TESTING_interpreter_fail (gts->is);
          return;
        }
      }
      {
        const struct TALER_Amount *reward_amount;

        if (GNUNET_OK !=
            TALER_TESTING_get_trait_amount (reward_cmd,
                                            &reward_amount))
        {
          GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                      "Could not fetch reward amount\n");
          TALER_TESTING_interpreter_fail (gts->is);
          return;
        }
        if ( (GNUNET_OK !=
              TALER_amount_cmp_currency (reward_amount,
                                         &reward->reward_amount)) ||
             (0 !=
              TALER_amount_cmp (reward_amount,
                                &reward->reward_amount)) )
        {
          GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                      "Reward amount does not match\n");
          TALER_TESTING_interpreter_fail (gts->is);
          return;
        }
      }
    }
    break;
  default:
    GNUNET_log (GNUNET_ERROR_TYPE_WARNING,
                "Unhandled HTTP status.\n");
  }
  TALER_TESTING_interpreter_next (gts->is);
}


/**
 * Run the "GET /private/rewards" CMD.
 *
 * @param cls closure.
 * @param cmd command being run now.
 * @param is interpreter state.
 */
static void
get_rewards_run (void *cls,
                 const struct TALER_TESTING_Command *cmd,
                 struct TALER_TESTING_Interpreter *is)
{
  struct GetRewardsState *gts = cls;

  gts->is = is;
  gts->tgh = TALER_MERCHANT_rewards_get2 (
    TALER_TESTING_interpreter_get_context (is),
    gts->merchant_url,
    TALER_EXCHANGE_YNA_NO,
    gts->limit,
    gts->offset,
    &get_rewards_cb,
    gts);

  GNUNET_assert (NULL != gts->tgh);
}


/**
 * Free the state of a "GET rewards" CMD, and possibly
 * cancel a pending operation thereof.
 *
 * @param cls closure.
 * @param cmd command being run.
 */
static void
get_rewards_cleanup (void *cls,
                     const struct TALER_TESTING_Command *cmd)
{
  struct GetRewardsState *gts = cls;

  if (NULL != gts->tgh)
  {
    GNUNET_log (GNUNET_ERROR_TYPE_WARNING,
                "GET /private/rewards operation did not complete\n");
    TALER_MERCHANT_rewards_get_cancel (gts->tgh);
  }
  GNUNET_array_grow (gts->rewards,
                     gts->rewards_length,
                     0);
  GNUNET_free (gts);
}


struct TALER_TESTING_Command
TALER_TESTING_cmd_get_rewards (const char *label,
                               const char *merchant_url,
                               unsigned int http_status,
                               ...)
{
  struct GetRewardsState *gts;

  gts = GNUNET_new (struct GetRewardsState);
  gts->merchant_url = merchant_url;
  gts->offset = INT64_MAX;
  gts->limit = -20;
  gts->http_status = http_status;
  {
    const char *clabel;
    va_list ap;

    va_start (ap, http_status);
    while (NULL != (clabel = va_arg (ap, const char *)))
    {
      GNUNET_array_append (gts->rewards,
                           gts->rewards_length,
                           clabel);
    }
    va_end (ap);
  }
  {
    struct TALER_TESTING_Command cmd = {
      .cls = gts,
      .label = label,
      .run = &get_rewards_run,
      .cleanup = &get_rewards_cleanup
    };

    return cmd;
  }
}


struct TALER_TESTING_Command
TALER_TESTING_cmd_get_rewards2 (const char *label,
                                const char *merchant_url,
                                uint64_t offset,
                                int64_t limit,
                                unsigned int http_status,
                                ...)
{
  struct GetRewardsState *gts;

  gts = GNUNET_new (struct GetRewardsState);
  gts->merchant_url = merchant_url;
  gts->offset = offset;
  gts->limit = limit;
  gts->http_status = http_status;
  {
    const char *clabel;
    va_list ap;

    va_start (ap, http_status);
    while (NULL != (clabel = va_arg (ap, const char *)))
    {
      GNUNET_array_append (gts->rewards,
                           gts->rewards_length,
                           clabel);
    }
    va_end (ap);
  }
  {
    struct TALER_TESTING_Command cmd = {
      .cls = gts,
      .label = label,
      .run = &get_rewards_run,
      .cleanup = &get_rewards_cleanup
    };

    return cmd;
  }
}


/* end of testing_api_cmd_get_rewards.c */
