/*
  This file is part of TALER
  Copyright (C) 2020 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify it under the
  terms of the GNU Lesser General Public License as published by the Free Software
  Foundation; either version 2.1, or (at your option) any later version.

  TALER is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
  A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public License along with
  TALER; see the file COPYING.LGPL.  If not, see
  <http://www.gnu.org/licenses/>
*/
/**
 * @file merchant_api_merchant_get_reward.c
 * @brief Implementation of the GET /private/rewards/$REWARD_ID request of the merchant's HTTP API
 * @author Jonathan Buchanan
 */
#include "platform.h"
#include <curl/curl.h>
#include <jansson.h>
#include <microhttpd.h> /* just for HTTP status codes */
#include <gnunet/gnunet_util_lib.h>
#include <gnunet/gnunet_curl_lib.h>
#include "taler_merchant_service.h"
#include "merchant_api_common.h"
#include "merchant_api_curl_defaults.h"
#include <taler/taler_json_lib.h>
#include <taler/taler_signatures.h>


struct TALER_MERCHANT_RewardMerchantGetHandle
{
  /**
   * The url for this request.
   */
  char *url;

  /**
   * Handle for the request.
   */
  struct GNUNET_CURL_Job *job;

  /**
   * Function to call with the result.
   */
  TALER_MERCHANT_RewardMerchantGetCallback cb;

  /**
   * Closure for @a cb.
   */
  void *cb_cls;

  /**
   * Reference to the execution context.
   */
  struct GNUNET_CURL_Context *ctx;
};


static enum GNUNET_GenericReturnValue
parse_pickups (const json_t *pa,
               struct TALER_MERCHANT_RewardStatusResponse *tsr,
               struct TALER_MERCHANT_RewardMerchantGetHandle *tgh)
{
  unsigned int pa_len = json_array_size (pa);
  struct TALER_MERCHANT_PickupDetail pickups[pa_len];
  size_t index;
  json_t *value;

  json_array_foreach (pa, index, value)
  {
    struct TALER_MERCHANT_PickupDetail *pickup = &pickups[index];
    struct GNUNET_JSON_Specification spec[] = {
      GNUNET_JSON_spec_fixed_auto ("pickup_id",
                                   &pickup->pickup_id),
      GNUNET_JSON_spec_uint64 ("num_planchets",
                               &pickup->num_planchets),
      TALER_JSON_spec_amount_any ("requested_amount",
                                  &pickup->requested_amount),
      GNUNET_JSON_spec_end ()
    };

    if (GNUNET_OK !=
        GNUNET_JSON_parse (value,
                           spec,
                           NULL,
                           NULL))
    {
      GNUNET_break_op (0);
      return GNUNET_SYSERR;
    }
  }
  tsr->details.ok.pickups_length = pa_len;
  tsr->details.ok.pickups = pickups;
  tgh->cb (tgh->cb_cls,
           tsr);
  return GNUNET_OK;
}


/**
 * Function called when we're done processing the
 * GET /private/rewards/$REWARD_ID request.
 *
 * @param cls the `struct TALER_MERCHANT_RewardMerchantGetHandle`
 * @param response_code HTTP response code, 0 on error
 * @param response response body, NULL if not in JSON
 */
static void
handle_merchant_reward_get_finished (void *cls,
                                     long response_code,
                                     const void *response)
{
  struct TALER_MERCHANT_RewardMerchantGetHandle *tgh = cls;
  const json_t *json = response;
  struct TALER_MERCHANT_RewardStatusResponse tsr = {
    .hr.http_status = (unsigned int) response_code,
    .hr.reply = json
  };

  GNUNET_log (GNUNET_ERROR_TYPE_DEBUG,
              "Got /private/rewards/$REWARD_ID response with status code %u\n",
              (unsigned int) response_code);
  tgh->job = NULL;
  switch (response_code)
  {
  case MHD_HTTP_OK:
    {
      struct GNUNET_JSON_Specification spec[] = {
        TALER_JSON_spec_amount_any ("total_authorized",
                                    &tsr.details.ok.total_authorized),
        TALER_JSON_spec_amount_any ("total_picked_up",
                                    &tsr.details.ok.total_picked_up),
        GNUNET_JSON_spec_string ("reason",
                                 &tsr.details.ok.reason),
        GNUNET_JSON_spec_timestamp ("expiration",
                                    &tsr.details.ok.expiration),
        GNUNET_JSON_spec_fixed_auto ("reserve_pub",
                                     &tsr.details.ok.reserve_pub),
        GNUNET_JSON_spec_end ()
      };

      if (GNUNET_OK !=
          GNUNET_JSON_parse (json,
                             spec,
                             NULL, NULL))
      {
        tsr.hr.http_status = 0;
        tsr.hr.ec = TALER_EC_GENERIC_INVALID_RESPONSE;
      }
      else
      {
        json_t *pickups = json_object_get (json,
                                           "pickups");
        if (! json_is_array (pickups))
        {
          tgh->cb (tgh->cb_cls,
                   &tsr);
          TALER_MERCHANT_merchant_reward_get_cancel (tgh);
          return;
        }
        if (GNUNET_OK ==
            parse_pickups (pickups,
                           &tsr,
                           tgh))
        {
          GNUNET_JSON_parse_free (spec);
          TALER_MERCHANT_merchant_reward_get_cancel (tgh);
          return;
        }
        tsr.hr.http_status = 0;
        tsr.hr.ec = TALER_EC_GENERIC_INVALID_RESPONSE;
      }
      GNUNET_JSON_parse_free (spec);
      break;
    }
  case MHD_HTTP_UNAUTHORIZED:
    tsr.hr.ec = TALER_JSON_get_error_code (json);
    tsr.hr.hint = TALER_JSON_get_error_hint (json);
    /* Nothing really to verify, merchant says we need to authenticate. */
    break;
  case MHD_HTTP_NOT_FOUND:
    /* legal, can happen if instance or reward reserve is unknown */
    tsr.hr.ec = TALER_JSON_get_error_code (json);
    tsr.hr.hint = TALER_JSON_get_error_hint (json);
    break;
  case MHD_HTTP_INTERNAL_SERVER_ERROR:
    /* Server had an internal issue; we should retry, but this API
       leaves this to the application */
    tsr.hr.ec = TALER_JSON_get_error_code (json);
    tsr.hr.hint = TALER_JSON_get_error_hint (json);
    break;
  default:
    /* unexpected response code */
    GNUNET_break_op (0);
    TALER_MERCHANT_parse_error_details_ (json,
                                         response_code,
                                         &tsr.hr);
    GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                "Unexpected response code %u/%d\n",
                (unsigned int) response_code,
                (int) tsr.hr.ec);
    break;
  }
  tgh->cb (tgh->cb_cls,
           &tsr);
  TALER_MERCHANT_merchant_reward_get_cancel (tgh);
}


struct TALER_MERCHANT_RewardMerchantGetHandle *
TALER_MERCHANT_merchant_reward_get (struct GNUNET_CURL_Context *ctx,
                                    const char *backend_url,
                                    const struct
                                    TALER_RewardIdentifierP *reward_id,
                                    const struct TALER_Amount *min_pick_up,
                                    struct GNUNET_TIME_Relative lp_timeout,
                                    bool pickups,
                                    TALER_MERCHANT_RewardMerchantGetCallback cb,
                                    void *cb_cls)
{
  struct TALER_MERCHANT_RewardMerchantGetHandle *tgh;
  CURL *eh;

  GNUNET_assert (NULL != backend_url);
  tgh = GNUNET_new (struct TALER_MERCHANT_RewardMerchantGetHandle);
  tgh->ctx = ctx;
  tgh->cb = cb;
  tgh->cb_cls = cb_cls;

  {
    char res_str[sizeof (*reward_id) * 2];
    char arg_str[sizeof (res_str) + 48];
    char timeout_str[32];
    char *end;

    end = GNUNET_STRINGS_data_to_string (reward_id,
                                         sizeof (*reward_id),
                                         res_str,
                                         sizeof (res_str));
    *end = '\0';
    GNUNET_snprintf (arg_str,
                     sizeof (arg_str),
                     "private/rewards/%s",
                     res_str);
    GNUNET_snprintf (timeout_str,
                     sizeof (timeout_str),
                     "%llu",
                     ((unsigned long long)
                      lp_timeout.rel_value_us
                      / GNUNET_TIME_UNIT_MILLISECONDS.rel_value_us));
    tgh->url = TALER_url_join (backend_url,
                               arg_str,
                               "pickups",
                               pickups
                               ? "yes"
                               : NULL,
                               "min_amount",
                               min_pick_up
                               ? TALER_amount2s (min_pick_up)
                               : NULL,
                               "timeout_ms",
                               GNUNET_TIME_relative_is_zero (lp_timeout)
                               ? NULL
                               : timeout_str,
                               NULL);
  }
  if (NULL == tgh->url)
  {
    GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                "Could not construct request URL.\n");
    GNUNET_free (tgh);
    return NULL;
  }

  eh = TALER_MERCHANT_curl_easy_get_ (tgh->url);
  tgh->job = GNUNET_CURL_job_add (ctx,
                                  eh,
                                  &handle_merchant_reward_get_finished,
                                  tgh);
  return tgh;
}


void
TALER_MERCHANT_merchant_reward_get_cancel (
  struct TALER_MERCHANT_RewardMerchantGetHandle *tgh)
{
  if (NULL != tgh->job)
  {
    GNUNET_CURL_job_cancel (tgh->job);
    tgh->job = NULL;
  }
  GNUNET_free (tgh->url);
  GNUNET_free (tgh);
}


/* end of merchant_api_merchant_get_reward.c */
