/*
   This file is part of TALER
   Copyright (C) 2023 Taler Systems SA

   TALER is free software; you can redistribute it and/or modify it under the
   terms of the GNU General Public License as published by the Free Software
   Foundation; either version 3, or (at your option) any later version.

   TALER is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
   A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along with
   TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
/**
 * @file backenddb/pg_lookup_rewards.c
 * @brief Implementation of the lookup_rewards function for Postgres
 * @author Iván Ávalos
 */
#include "platform.h"
#include <taler/taler_error_codes.h>
#include <taler/taler_dbevents.h>
#include <taler/taler_pq_lib.h>
#include "pg_lookup_rewards.h"
#include "pg_helper.h"

/**
 * Context used for postgres_lookup_rewards().
 */
struct LookupMerchantRewardsContext
{
  /**
   * Postgres context.
   */
  struct PostgresClosure *pg;

  /**
   * Function to call with the results.
   */
  TALER_MERCHANTDB_RewardsCallback cb;

  /**
   * Closure for @a cb.
   */
  void *cb_cls;

  /**
   * Internal result.
   */
  enum GNUNET_DB_QueryStatus qs;
};


/**
 * Function to be called with the results of a SELECT statement
 * that has returned @a num_results results about rewards.
 *
 * @param[in,out] cls of type `struct LookupRewardsContext *`
 * @param result the postgres result
 * @param num_results the number of results in @a result
 */
static void
lookup_rewards_cb (void *cls,
                   PGresult *result,
                   unsigned int num_results)
{
  struct LookupMerchantRewardsContext *plc = cls;

  for (unsigned int i = 0; i < num_results; i++)
  {
    uint64_t row_id;
    struct TALER_RewardIdentifierP reward_id;
    struct TALER_Amount reward_amount;
    struct GNUNET_PQ_ResultSpec rs[] = {
      GNUNET_PQ_result_spec_uint64 ("reward_serial",
                                    &row_id),
      GNUNET_PQ_result_spec_auto_from_type ("reward_id",
                                            &reward_id),
      TALER_PQ_result_spec_amount_with_currency ("amount",
                                                 &reward_amount),
      GNUNET_PQ_result_spec_end
    };

    if (GNUNET_OK !=
        GNUNET_PQ_extract_result (result,
                                  rs,
                                  i))
    {
      GNUNET_break (0);
      plc->qs = GNUNET_DB_STATUS_HARD_ERROR;
      return;
    }
    plc->cb (plc->cb_cls,
             row_id,
             reward_id,
             reward_amount);
    GNUNET_PQ_cleanup_result (rs);
  }
}


enum GNUNET_DB_QueryStatus
TMH_PG_lookup_rewards (void *cls,
                       const char *instance_id,
                       enum TALER_EXCHANGE_YesNoAll expired,
                       int64_t limit,
                       uint64_t offset,
                       TALER_MERCHANTDB_RewardsCallback cb,
                       void *cb_cls)
{
  struct PostgresClosure *pg = cls;
  struct LookupMerchantRewardsContext plc = {
    .pg = pg,
    .cb = cb,
    .cb_cls = cb_cls
  };
  uint64_t ulimit = (limit > 0) ? limit : -limit;
  uint8_t bexpired;
  struct GNUNET_TIME_Absolute now = GNUNET_TIME_absolute_get ();
  struct GNUNET_PQ_QueryParam params[] = {
    GNUNET_PQ_query_param_string (instance_id),
    GNUNET_PQ_query_param_uint64 (&ulimit),
    GNUNET_PQ_query_param_uint64 (&offset),
    GNUNET_PQ_query_param_absolute_time (&now),
    GNUNET_PQ_query_param_auto_from_type (&bexpired),
    GNUNET_PQ_query_param_end
  };
  enum GNUNET_DB_QueryStatus qs;
  char stmt[128];

  bexpired = (TALER_EXCHANGE_YNA_YES == expired);
  GNUNET_snprintf (stmt,
                   sizeof (stmt),
                   "lookup_rewards_%s%s",
                   (limit > 0) ? "inc" : "dec",
                   (TALER_EXCHANGE_YNA_ALL == expired) ? "" : "_expired");

  check_connection (pg);
  PREPARE (pg,
           "lookup_rewards_inc",
           "SELECT"
           " reward_serial"
           ",reward_id"
           ",amount"
           ",CAST($4 as BIGINT)" /* otherwise $4 is unused and Postgres unhappy */
           ",CAST($5 as BOOL)" /* otherwise $5 is unused and Postgres unhappy */
           " FROM merchant_rewards"
           " JOIN merchant_reward_reserves USING (reserve_serial)"
           " WHERE merchant_serial ="
           "     (SELECT merchant_serial"
           "        FROM merchant_instances"
           "       WHERE merchant_id=$1)"
           "   AND"
           "    reward_serial > $3"
           " ORDER BY reward_serial ASC"
           " LIMIT $2");
  PREPARE (pg,
           "lookup_rewards_dec",
           "SELECT"
           " reward_serial"
           ",reward_id"
           ",amount"
           ",CAST($4 as BIGINT)" /* otherwise $4 is unused and Postgres unhappy */
           ",CAST($5 as BOOL)" /* otherwise $5 is unused and Postgres unhappy */
           " FROM merchant_rewards"
           " JOIN merchant_reward_reserves USING (reserve_serial)"
           " WHERE merchant_serial ="
           "     (SELECT merchant_serial"
           "        FROM merchant_instances"
           "       WHERE merchant_id=$1)"
           "   AND"
           "    reward_serial < $3"
           " ORDER BY reward_serial DESC"
           " LIMIT $2");
  PREPARE (pg,
           "lookup_rewards_inc_expired",
           "SELECT"
           " reward_serial"
           ",reward_id"
           ",amount"
           " FROM merchant_rewards"
           " JOIN merchant_reward_reserves USING (reserve_serial)"
           " WHERE merchant_serial ="
           "     (SELECT merchant_serial"
           "        FROM merchant_instances"
           "       WHERE merchant_id=$1)"
           "   AND"
           "    reward_serial > $3"
           "   AND"
           "    CAST($5 as BOOL) = (merchant_rewards.expiration < $4)"
           " ORDER BY reward_serial ASC"
           " LIMIT $2");
  PREPARE (pg,
           "lookup_rewards_dec_expired",
           "SELECT"
           " reward_serial"
           ",reward_id"
           ",amount"
           " FROM merchant_rewards"
           " JOIN merchant_reward_reserves USING (reserve_serial)"
           " WHERE merchant_serial ="
           "     (SELECT merchant_serial"
           "        FROM merchant_instances"
           "       WHERE merchant_id=$1)"
           "   AND"
           "    reward_serial < $3"
           "   AND"
           "    CAST($5 as BOOL) = (merchant_rewards.expiration < $4)"
           " ORDER BY reward_serial DESC"
           " LIMIT $2");

  qs = GNUNET_PQ_eval_prepared_multi_select (pg->conn,
                                             stmt,
                                             params,
                                             &lookup_rewards_cb,
                                             &plc);
  if (0 != plc.qs)
    return plc.qs;
  return qs;
}
