/*
   This file is part of TALER
   Copyright (C) 2023 Taler Systems SA

   TALER is free software; you can redistribute it and/or modify it under the
   terms of the GNU General Public License as published by the Free Software
   Foundation; either version 3, or (at your option) any later version.

   TALER is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
   A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along with
   TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
/**
 * @file backenddb/pg_lookup_reward.c
 * @brief Implementation of the lookup_reward function for Postgres
 * @author Iván Ávalos
 */
#include "platform.h"
#include <taler/taler_error_codes.h>
#include <taler/taler_dbevents.h>
#include <taler/taler_pq_lib.h>
#include "pg_lookup_reward.h"
#include "pg_helper.h"

enum GNUNET_DB_QueryStatus
TMH_PG_lookup_reward (void *cls,
                      const char *instance_id,
                      const struct TALER_RewardIdentifierP *reward_id,
                      struct TALER_Amount *total_authorized,
                      struct TALER_Amount *total_picked_up,
                      struct GNUNET_TIME_Timestamp *expiration,
                      char **exchange_url,
                      char **next_url,
                      struct TALER_ReservePrivateKeyP *reserve_priv)
{
  struct PostgresClosure *pg = cls;
  struct GNUNET_PQ_QueryParam params[] = {
    GNUNET_PQ_query_param_string (instance_id),
    GNUNET_PQ_query_param_auto_from_type (reward_id),
    GNUNET_PQ_query_param_end
  };
  struct GNUNET_PQ_ResultSpec rs[] = {
    TALER_PQ_result_spec_amount_with_currency ("amount",
                                               total_authorized),
    TALER_PQ_result_spec_amount_with_currency ("picked_up",
                                               total_picked_up),
    GNUNET_PQ_result_spec_timestamp ("expiration",
                                     expiration),
    GNUNET_PQ_result_spec_string ("exchange_url",
                                  exchange_url),
    GNUNET_PQ_result_spec_string ("next_url",
                                  next_url),
    GNUNET_PQ_result_spec_auto_from_type ("reserve_priv",
                                          reserve_priv),
    GNUNET_PQ_result_spec_end
  };

  check_connection (pg);
  PREPARE (pg,
           "lookup_reward",
           "SELECT"
           " amount"
           ",picked_up"
           ",merchant_rewards.expiration"
           ",exchange_url"
           ",next_url"
           ",reserve_priv"
           " FROM merchant_rewards"
           " JOIN merchant_reward_reserves USING (reserve_serial)"
           " JOIN merchant_reward_reserve_keys USING (reserve_serial)"
           " WHERE reward_id = $2"
           "   AND merchant_serial ="
           "     (SELECT merchant_serial"
           "        FROM merchant_instances"
           "       WHERE merchant_id=$1)");

  return GNUNET_PQ_eval_prepared_singleton_select (pg->conn,
                                                   "lookup_reward",
                                                   params,
                                                   rs);
}
