/*
   This file is part of TALER
   Copyright (C) 2022 Taler Systems SA

   TALER is free software; you can redistribute it and/or modify it under the
   terms of the GNU General Public License as published by the Free Software
   Foundation; either version 3, or (at your option) any later version.

   TALER is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
   A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along with
   TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
/**
 * @file backenddb/pg_lookup_reserves.c
 * @brief Implementation of the lookup_reserves function for Postgres
 * @author Christian Grothoff
 */
#include "platform.h"
#include <taler/taler_error_codes.h>
#include <taler/taler_dbevents.h>
#include <taler/taler_pq_lib.h>
#include "pg_lookup_reserves.h"
#include "pg_helper.h"


/**
 * Closure for #lookup_reserves_cb.
 */
struct LookupReservesContext
{
  /**
   * Postgres context.
   */
  struct PostgresClosure *pg;

  /**
   * Function to call with the results
   */
  TALER_MERCHANTDB_ReservesCallback cb;

  /**
   * Closure for @e cb
   */
  void *cb_cls;

  /**
   * Filter by active reserves.
   */
  enum TALER_EXCHANGE_YesNoAll active;

  /**
   * Filter by failures (mismatch in exchange claimed and
   * merchant claimed initial amounts).
   */
  enum TALER_EXCHANGE_YesNoAll failures;

  /**
   * Set in case of errors.
   */
  enum GNUNET_DB_QueryStatus qs;

};


/**
 * Function to be called with the results of a SELECT statement
 * that has returned @a num_results results about accounts.
 *
 * @param[in,out] cls of type `struct LookupReservesContext *`
 * @param result the postgres result
 * @param num_results the number of results in @a result
 */
static void
lookup_reserves_cb (void *cls,
                    PGresult *result,
                    unsigned int num_results)
{
  struct LookupReservesContext *lrc = cls;

  for (unsigned int i = 0; i < num_results; i++)
  {
    struct TALER_ReservePublicKeyP reserve_pub;
    struct GNUNET_TIME_Timestamp creation_time;
    struct GNUNET_TIME_Timestamp expiration_time;
    struct TALER_Amount merchant_initial_balance;
    struct TALER_Amount exchange_initial_balance;
    struct TALER_Amount pickup_amount;
    struct TALER_Amount committed_amount;
    uint8_t active;
    struct GNUNET_PQ_ResultSpec rs[] = {
      GNUNET_PQ_result_spec_auto_from_type ("reserve_pub",
                                            &reserve_pub),
      GNUNET_PQ_result_spec_timestamp ("creation_time",
                                       &creation_time),
      GNUNET_PQ_result_spec_timestamp ("expiration",
                                       &expiration_time),
      TALER_PQ_result_spec_amount_with_currency ("merchant_initial_balance",
                                                 &merchant_initial_balance),
      TALER_PQ_result_spec_amount_with_currency ("exchange_initial_balance",
                                                 &exchange_initial_balance),
      TALER_PQ_result_spec_amount_with_currency ("rewards_committed",
                                                 &committed_amount),
      TALER_PQ_result_spec_amount_with_currency ("rewards_picked_up",
                                                 &pickup_amount),
      GNUNET_PQ_result_spec_auto_from_type ("active",
                                            &active),
      GNUNET_PQ_result_spec_end
    };

    if (GNUNET_OK !=
        GNUNET_PQ_extract_result (result,
                                  rs,
                                  i))
    {
      GNUNET_break (0);
      lrc->qs = GNUNET_DB_STATUS_HARD_ERROR;
      return;
    }
    switch (lrc->active)
    {
    case TALER_EXCHANGE_YNA_YES:
      if (0 == active)
        continue;
      break;
    case TALER_EXCHANGE_YNA_NO:
      if (0 != active)
        continue;
      break;
    case TALER_EXCHANGE_YNA_ALL:
      break;
    }
    switch (lrc->failures)
    {
    case TALER_EXCHANGE_YNA_YES:
      if (0 ==
          TALER_amount_cmp (&merchant_initial_balance,
                            &exchange_initial_balance))
        continue;
      break;
    case TALER_EXCHANGE_YNA_NO:
      if (0 !=
          TALER_amount_cmp (&merchant_initial_balance,
                            &exchange_initial_balance))
        continue;
      break;
    case TALER_EXCHANGE_YNA_ALL:
      break;
    }
    lrc->cb (lrc->cb_cls,
             &reserve_pub,
             creation_time,
             expiration_time,
             &merchant_initial_balance,
             &exchange_initial_balance,
             &pickup_amount,
             &committed_amount,
             (0 != active));
  }
}


enum GNUNET_DB_QueryStatus
TMH_PG_lookup_reserves (void *cls,
                        const char *instance_id,
                        struct GNUNET_TIME_Timestamp created_after,
                        enum TALER_EXCHANGE_YesNoAll active,
                        enum TALER_EXCHANGE_YesNoAll failures,
                        TALER_MERCHANTDB_ReservesCallback cb,
                        void *cb_cls)
{
  struct PostgresClosure *pg = cls;
  struct LookupReservesContext lrc = {
    .pg = pg,
    .active = active,
    .failures = failures,
    .cb = cb,
    .cb_cls = cb_cls
  };
  struct GNUNET_PQ_QueryParam params[] = {
    GNUNET_PQ_query_param_string (instance_id),
    GNUNET_PQ_query_param_timestamp (&created_after),
    GNUNET_PQ_query_param_end
  };
  enum GNUNET_DB_QueryStatus qs;

  check_connection (pg);
  PREPARE (pg,
           "lookup_reserves",
           "SELECT"
           " reserve_pub"
           ",creation_time"
           ",expiration"
           ",merchant_initial_balance"
           ",exchange_initial_balance"
           ",rewards_committed"
           ",rewards_picked_up"
           ",reserve_priv IS NOT NULL AS active"
           " FROM merchant_reward_reserves"
           " FULL OUTER JOIN merchant_reward_reserve_keys USING (reserve_serial)"
           " WHERE creation_time > $2"
           "   AND merchant_serial ="
           "     (SELECT merchant_serial"
           "        FROM merchant_instances"
           "       WHERE merchant_id=$1)");
  qs = GNUNET_PQ_eval_prepared_multi_select (pg->conn,
                                             "lookup_reserves",
                                             params,
                                             &lookup_reserves_cb,
                                             &lrc);
  if (lrc.qs < 0)
    return lrc.qs;
  return qs;
}
