/*
   This file is part of TALER
   Copyright (C) 2023 Taler Systems SA

   TALER is free software; you can redistribute it and/or modify it under the
   terms of the GNU General Public License as published by the Free Software
   Foundation; either version 3, or (at your option) any later version.

   TALER is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
   A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along with
   TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
/**
 * @file backenddb/pg_lookup_pending_reserves.c
 * @brief Implementation of the lookup_pending_reserves function for Postgres
 * @author Iván Ávalos
 */
#include "platform.h"
#include <taler/taler_error_codes.h>
#include <taler/taler_dbevents.h>
#include <taler/taler_pq_lib.h>
#include "pg_lookup_pending_reserves.h"
#include "pg_helper.h"

/**
 * Closure for #lookup_pending_reserves_cb.
 */
struct LookupPendingReservesContext
{
  /**
   * Postgres context.
   */
  struct PostgresClosure *pg;

  /**
   * Function to call with the results
   */
  TALER_MERCHANTDB_PendingReservesCallback cb;

  /**
   * Closure for @e cb
   */
  void *cb_cls;

  /**
   * Set in case of errors.
   */
  enum GNUNET_DB_QueryStatus qs;

};


/**
 * Function to be called with the results of a SELECT statement
 * that has returned @a num_results results about accounts.
 *
 * @param[in,out] cls of type `struct LookupReservesContext *`
 * @param result the postgres result
 * @param num_results the number of results in @a result
 */
static void
lookup_pending_reserves_cb (void *cls,
                            PGresult *result,
                            unsigned int num_results)
{
  struct LookupPendingReservesContext *lrc = cls;

  for (unsigned int i = 0; i < num_results; i++)
  {
    struct TALER_ReservePublicKeyP reserve_pub;
    struct TALER_Amount merchant_initial_balance;
    char *exchange_url;
    char *instance_id;
    struct GNUNET_PQ_ResultSpec rs[] = {
      GNUNET_PQ_result_spec_auto_from_type ("reserve_pub",
                                            &reserve_pub),
      GNUNET_PQ_result_spec_string ("merchant_id",
                                    &instance_id),
      GNUNET_PQ_result_spec_string ("exchange_url",
                                    &exchange_url),
      TALER_PQ_result_spec_amount_with_currency ("merchant_initial_balance",
                                                 &merchant_initial_balance),
      GNUNET_PQ_result_spec_end
    };

    if (GNUNET_OK !=
        GNUNET_PQ_extract_result (result,
                                  rs,
                                  i))
    {
      GNUNET_break (0);
      lrc->qs = GNUNET_DB_STATUS_HARD_ERROR;
      return;
    }
    lrc->cb (lrc->cb_cls,
             instance_id,
             exchange_url,
             &reserve_pub,
             &merchant_initial_balance);
    GNUNET_PQ_cleanup_result (rs);
  }
}


enum GNUNET_DB_QueryStatus
TMH_PG_lookup_pending_reserves (void *cls,
                                TALER_MERCHANTDB_PendingReservesCallback cb,
                                void *cb_cls)
{
  struct PostgresClosure *pg = cls;
  struct LookupPendingReservesContext lrc = {
    .pg = pg,
    .cb = cb,
    .cb_cls = cb_cls
  };
  struct GNUNET_PQ_QueryParam params[] = {
    GNUNET_PQ_query_param_end
  };
  enum GNUNET_DB_QueryStatus qs;

  check_connection (pg);
  PREPARE (pg,
           "lookup_pending_reserves",
           "SELECT"
           " reserve_pub"
           ",merchant_id"
           ",exchange_url"
           ",merchant_initial_balance"
           " FROM merchant_reward_reserves mrr"
           " JOIN merchant_instances USING (merchant_serial)"
           " JOIN merchant_reward_reserve_keys USING (reserve_serial)"
           " WHERE (mrr.exchange_initial_balance).val=0"
           "   AND (mrr.exchange_initial_balance).frac=0");

  qs = GNUNET_PQ_eval_prepared_multi_select (pg->conn,
                                             "lookup_pending_reserves",
                                             params,
                                             &lookup_pending_reserves_cb,
                                             &lrc);
  if (lrc.qs < 0)
    return lrc.qs;
  return qs;
}
