/*
   This file is part of TALER
   Copyright (C) 2022 Taler Systems SA

   TALER is free software; you can redistribute it and/or modify it under the
   terms of the GNU General Public License as published by the Free Software
   Foundation; either version 3, or (at your option) any later version.

   TALER is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
   A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along with
   TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
/**
 * @file backenddb/pg_insert_pickup.c
 * @brief Implementation of the insert_pickup function for Postgres
 * @author Christian Grothoff
 */
#include "platform.h"
#include <taler/taler_error_codes.h>
#include <taler/taler_dbevents.h>
#include <taler/taler_pq_lib.h>
#include "pg_insert_pickup.h"
#include "pg_helper.h"


enum GNUNET_DB_QueryStatus
TMH_PG_insert_pickup (
  void *cls,
  const char *instance_id,
  const struct TALER_RewardIdentifierP *reward_id,
  const struct TALER_Amount *total_picked_up,
  const struct TALER_PickupIdentifierP *pickup_id,
  const struct TALER_Amount *total_requested)
{
  struct PostgresClosure *pg = cls;
  enum GNUNET_DB_QueryStatus qs;

  PREPARE (pg,
           "insert_pickup",
           "INSERT INTO merchant_reward_pickups"
           "(reward_serial"
           ",pickup_id"
           ",amount"
           ") "
           "SELECT"
           " reward_serial, $3, $4"
           " FROM merchant_rewards"
           " JOIN merchant_reward_reserves USING (reserve_serial)"
           " WHERE reward_id=$2"
           "  AND merchant_serial = "
           "     (SELECT merchant_serial"
           "        FROM merchant_instances"
           "       WHERE merchant_id=$1)");
  PREPARE (pg,
           "update_picked_up_reward",
           "UPDATE merchant_rewards SET"
           " picked_up=$2"
           ",was_picked_up = (CAST($2 AS taler_amount_currency) = amount)"
           " WHERE reward_id = $1");
  PREPARE (pg,
           "lookup_picked_up_reserve",
           "SELECT"
           " reserve_serial"
           ",rewards_picked_up"
           " FROM merchant_reward_reserves"
           " JOIN merchant_rewards USING (reserve_serial)"
           " WHERE reward_id=$2"
           "   AND merchant_serial ="
           "     (SELECT merchant_serial"
           "        FROM merchant_instances"
           "       WHERE merchant_id=$1)");
  PREPARE (pg,
           "update_picked_up_reserve",
           "UPDATE merchant_reward_reserves SET"
           " rewards_picked_up=$2"
           " WHERE reserve_serial = $1");

  {
    struct GNUNET_PQ_QueryParam params[] = {
      GNUNET_PQ_query_param_string (instance_id),
      GNUNET_PQ_query_param_auto_from_type (reward_id),
      GNUNET_PQ_query_param_auto_from_type (pickup_id),
      TALER_PQ_query_param_amount_with_currency (pg->conn,
                                                 total_requested),
      GNUNET_PQ_query_param_end
    };

    qs = GNUNET_PQ_eval_prepared_non_select (pg->conn,
                                             "insert_pickup",
                                             params);
    if (0 > qs)
      return qs;
  }

  {
    struct GNUNET_PQ_QueryParam params[] = {
      GNUNET_PQ_query_param_auto_from_type (reward_id),
      TALER_PQ_query_param_amount_with_currency (pg->conn,
                                                 total_picked_up),
      GNUNET_PQ_query_param_end
    };

    qs = GNUNET_PQ_eval_prepared_non_select (pg->conn,
                                             "update_picked_up_reward",
                                             params);
    if (0 > qs)
      return qs;
  }
  {
    uint64_t reserve_serial;
    struct TALER_Amount reserve_picked_up;
    {
      struct GNUNET_PQ_QueryParam params[] = {
        GNUNET_PQ_query_param_string (instance_id),
        GNUNET_PQ_query_param_auto_from_type (reward_id),
        GNUNET_PQ_query_param_end
      };
      struct GNUNET_PQ_ResultSpec rs[] = {
        GNUNET_PQ_result_spec_uint64 ("reserve_serial",
                                      &reserve_serial),
        TALER_PQ_result_spec_amount_with_currency ("rewards_picked_up",
                                                   &reserve_picked_up),
        GNUNET_PQ_result_spec_end

      };

      qs = GNUNET_PQ_eval_prepared_singleton_select (pg->conn,
                                                     "lookup_picked_up_reserve",
                                                     params,
                                                     rs);
      if (0 > qs)
        return qs;
    }
    if (GNUNET_OK !=
        TALER_amount_cmp_currency (&reserve_picked_up,
                                   total_requested))
    {
      GNUNET_break (0);
      return GNUNET_DB_STATUS_HARD_ERROR;
    }
    if (0 >=
        TALER_amount_add (&reserve_picked_up,
                          &reserve_picked_up,
                          total_requested))
    {
      GNUNET_break (0);
      return GNUNET_DB_STATUS_HARD_ERROR;
    }

    {
      struct GNUNET_PQ_QueryParam params[] = {
        GNUNET_PQ_query_param_uint64 (&reserve_serial),
        TALER_PQ_query_param_amount_with_currency (pg->conn,
                                                   &reserve_picked_up),
        GNUNET_PQ_query_param_end
      };

      qs = GNUNET_PQ_eval_prepared_non_select (pg->conn,
                                               "update_picked_up_reserve",
                                               params);
      if (0 > qs)
        return qs;
    }
  }
  return qs;
}
