/*
  This file is part of TALER
  Copyright (C) 2014-2023 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as
  published by the Free Software Foundation; either version 3, or
  (at your option) any later version.

  TALER is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public
  License along with TALER; see the file COPYING.  If not, see
  <http://www.gnu.org/licenses/>
*/

/**
 * @file testing_api_cmd_reward_authorize.c
 * @brief command to test the rewardping.
 * @author Marcello Stanisci
 */

#include "platform.h"
#include <taler/taler_exchange_service.h>
#include <taler/taler_testing_lib.h>
#include "taler_merchant_service.h"
#include "taler_merchant_testing_lib.h"


/**
 * State for a /reward-authorize CMD.
 */
struct RewardAuthorizeState
{

  /**
   * Merchant base URL.
   */
  const char *merchant_url;

  /**
   * Expected HTTP response code.
   */
  unsigned int http_status;

  /**
   * Reference to the reserv to authorize the reward
   * from (if NULL, the merchant decides).
   */
  const char *reserve_reference;

  /**
   * Human-readable justification for the
   * reward authorization carried on by this CMD.
   */
  const char *justification;

  /**
   * Amount that should be authorized for rewardping.
   */
  struct TALER_Amount amount;

  /**
   * Expected Taler error code for this CMD.
   */
  enum TALER_ErrorCode expected_ec;

  /**
   * Reward taler:// URI.
   */
  char *reward_uri;

  /**
   * The reward id; set when the CMD succeeds.
   */
  struct TALER_RewardIdentifierP reward_id;

  /**
   * Expiration date for this reward.
   */
  struct GNUNET_TIME_Timestamp reward_expiration;

  /**
   * Handle to the on-going /reward-authorize request.
   */
  struct TALER_MERCHANT_RewardAuthorizeHandle *tao;

  /**
   * The interpreter state.
   */
  struct TALER_TESTING_Interpreter *is;

  /**
   * Task used for retries.
   */
  struct GNUNET_SCHEDULER_Task *retry_task;

  /**
   * How long do we wait between retries?
   */
  struct GNUNET_TIME_Relative backoff;

  /**
   * How many retries are left?
   */
  unsigned int retries_left;

};


/**
 * Run the main logic of talking to the merchant.
 *
 * @param cls a `struct RewardAuthorizeState`.
 */
static void
do_retry (void *cls);


/**
 * Callback for a /reward-authorize request.  Set into the state
 * what was returned from the backend (@a reward_id and @a
 * reward_expiration).
 *
 * @param cls closure
 * @param tar response we got
 */
static void
reward_authorize_cb (void *cls,
                     const struct TALER_MERCHANT_RewardAuthorizeResponse *tar)
{
  struct RewardAuthorizeState *tas = cls;

  tas->tao = NULL;
  if (tas->http_status != tar->hr.http_status)
  {
    if ( (MHD_HTTP_NOT_FOUND == tar->hr.http_status) &&
         (0 < tas->retries_left) )
    {
      GNUNET_log (GNUNET_ERROR_TYPE_INFO,
                  "Reserve authorization failed. Reserve may not yet be ready, retrying %u more times.\n",
                  tas->retries_left);
      tas->retries_left--;
      tas->backoff = GNUNET_TIME_randomized_backoff (tas->backoff,
                                                     GNUNET_TIME_UNIT_SECONDS);
      tas->retry_task = GNUNET_SCHEDULER_add_delayed (tas->backoff,
                                                      &do_retry,
                                                      tas);
      return;
    }

    GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                "Unexpected response code %u (%d) to command %s\n",
                tar->hr.http_status,
                tar->hr.ec,
                TALER_TESTING_interpreter_get_current_label (tas->is));
    TALER_TESTING_interpreter_fail (tas->is);
    return;
  }

  if (tas->expected_ec != tar->hr.ec)
  {
    GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                "Unexpected error code %d (%u) to command %s\n",
                (int) tar->hr.ec,
                tar->hr.http_status,
                TALER_TESTING_interpreter_get_current_label (tas->is));
    TALER_TESTING_interpreter_fail (tas->is);
    return;
  }
  if ( (MHD_HTTP_OK == tar->hr.http_status) &&
       (TALER_EC_NONE == tar->hr.ec) )
  {
    tas->reward_uri = GNUNET_strdup (tar->details.ok.reward_uri);
    tas->reward_id = tar->details.ok.reward_id;
    tas->reward_expiration = tar->details.ok.reward_expiration;
  }
  TALER_TESTING_interpreter_next (tas->is);
}


/**
 * Offers information from the /reward-authorize CMD state to other
 * commands.
 *
 * @param cls closure
 * @param[out] ret result (could be anything)
 * @param trait name of the trait
 * @param index index number of the object to extract.
 * @return #GNUNET_OK on success
 */
static enum GNUNET_GenericReturnValue
reward_authorize_traits (void *cls,
                         const void **ret,
                         const char *trait,
                         unsigned int index)
{
  struct RewardAuthorizeState *tas = cls;
  struct TALER_TESTING_Trait traits[] = {
    TALER_TESTING_make_trait_reward_id (&tas->reward_id),
    TALER_TESTING_make_trait_amount (&tas->amount),
    TALER_TESTING_make_trait_reason (tas->justification),
    TALER_TESTING_make_trait_timestamp (0,
                                        &tas->reward_expiration),
    TALER_TESTING_trait_end (),
  };

  return TALER_TESTING_get_trait (traits,
                                  ret,
                                  trait,
                                  index);
}


/**
 * Runs the /reward-authorize CMD
 *
 * @param cls closure
 * @param cmd the CMD representing _this_ command
 * @param is interpreter state
 */
static void
reward_authorize_run (void *cls,
                      const struct TALER_TESTING_Command *cmd,
                      struct TALER_TESTING_Interpreter *is)
{
  struct RewardAuthorizeState *tas = cls;

  tas->retries_left = 16;
  tas->is = is;
  tas->retry_task = GNUNET_SCHEDULER_add_now (&do_retry,
                                              tas);
}


static void
do_retry (void *cls)
{
  struct RewardAuthorizeState *tas = cls;

  tas->retry_task = NULL;
  if (NULL == tas->reserve_reference)
  {
    tas->tao = TALER_MERCHANT_reward_authorize (
      TALER_TESTING_interpreter_get_context (tas->is),
      tas->merchant_url,
      "http://merchant.com/pickup",
      &tas->amount,
      tas->justification,
      &reward_authorize_cb,
      tas);
  }
  else
  {
    const struct TALER_TESTING_Command *reserve_cmd;
    const struct TALER_ReservePublicKeyP *reserve_pub;

    reserve_cmd = TALER_TESTING_interpreter_lookup_command (
      tas->is,
      tas->reserve_reference);
    GNUNET_assert (GNUNET_OK ==
                   TALER_TESTING_get_trait_reserve_pub (reserve_cmd,
                                                        &reserve_pub));
    tas->tao = TALER_MERCHANT_reward_authorize2 (
      TALER_TESTING_interpreter_get_context (tas->is),
      tas->merchant_url,
      reserve_pub,
      "http://merchant.com/pickup",
      &tas->amount,
      tas->justification,
      &reward_authorize_cb,
      tas);
  }
  GNUNET_assert (NULL != tas->tao);
}


/**
 * Run the /reward-authorize CMD, the "fake" version of it.
 *
 * @param cls closure
 * @param cmd the CMD representing _this_ command
 * @param is interpreter state *
 */
static void
reward_authorize_fake_run (void *cls,
                           const struct TALER_TESTING_Command *cmd,
                           struct TALER_TESTING_Interpreter *is)
{
  struct RewardAuthorizeState *tas = cls;

  /* Make up a reward id.  */
  GNUNET_CRYPTO_random_block (GNUNET_CRYPTO_QUALITY_WEAK,
                              &tas->reward_id,
                              sizeof (struct TALER_RewardIdentifierP));
  TALER_TESTING_interpreter_next (is);
}


/**
 * Free the state from a /reward-authorize CMD, and possibly
 * cancel any pending operation.
 *
 * @param cls closure
 * @param cmd the /reward-authorize CMD that is about to be freed.
 */
static void
reward_authorize_cleanup (void *cls,
                          const struct TALER_TESTING_Command *cmd)
{
  struct RewardAuthorizeState *tas = cls;

  if (NULL != tas->tao)
  {
    TALER_LOG_WARNING ("Reward-autorize operation"
                       " did not complete\n");
    TALER_MERCHANT_reward_authorize_cancel (tas->tao);
  }
  if (NULL != tas->retry_task)
  {
    GNUNET_SCHEDULER_cancel (tas->retry_task);
    tas->retry_task = NULL;
  }
  GNUNET_free (tas->reward_uri);
  GNUNET_free (tas);
}


struct TALER_TESTING_Command
TALER_TESTING_cmd_reward_authorize_with_ec (const char *label,
                                            const char *merchant_url,
                                            const char *exchange_url,
                                            unsigned int http_status,
                                            const char *justification,
                                            const char *amount,
                                            enum TALER_ErrorCode ec)
{
  struct RewardAuthorizeState *tas;

  tas = GNUNET_new (struct RewardAuthorizeState);
  tas->merchant_url = merchant_url;
  tas->justification = justification;
  tas->http_status = http_status;
  tas->expected_ec = ec;
  GNUNET_assert (GNUNET_OK ==
                 TALER_string_to_amount (amount,
                                         &tas->amount));
  {
    struct TALER_TESTING_Command cmd = {
      .label = label,
      .cls = tas,
      .run = &reward_authorize_run,
      .cleanup = &reward_authorize_cleanup,
      .traits = &reward_authorize_traits
    };

    return cmd;
  }
}


struct TALER_TESTING_Command
TALER_TESTING_cmd_reward_authorize_from_reserve_with_ec (
  const char *label,
  const char *merchant_url,
  const char *exchange_url,
  const char *reserve_reference,
  unsigned int http_status,
  const char *justification,
  const char *amount,
  enum TALER_ErrorCode ec)
{
  struct RewardAuthorizeState *tas;

  tas = GNUNET_new (struct RewardAuthorizeState);
  tas->merchant_url = merchant_url;
  tas->justification = justification;
  tas->http_status = http_status;
  tas->expected_ec = ec;
  tas->reserve_reference = reserve_reference;
  GNUNET_assert (GNUNET_OK ==
                 TALER_string_to_amount (amount,
                                         &tas->amount));
  {
    struct TALER_TESTING_Command cmd = {
      .label = label,
      .cls = tas,
      .run = &reward_authorize_run,
      .cleanup = &reward_authorize_cleanup,
      .traits = &reward_authorize_traits
    };

    return cmd;
  }
}


struct TALER_TESTING_Command
TALER_TESTING_cmd_reward_authorize (const char *label,
                                    const char *merchant_url,
                                    const char *exchange_url,
                                    unsigned int http_status,
                                    const char *justification,
                                    const char *amount)
{
  struct RewardAuthorizeState *tas;

  tas = GNUNET_new (struct RewardAuthorizeState);
  tas->merchant_url = merchant_url;
  tas->justification = justification;
  tas->http_status = http_status;
  GNUNET_assert (GNUNET_OK ==
                 TALER_string_to_amount (amount,
                                         &tas->amount));
  {
    struct TALER_TESTING_Command cmd = {
      .label = label,
      .cls = tas,
      .run = &reward_authorize_run,
      .cleanup = &reward_authorize_cleanup,
      .traits = &reward_authorize_traits
    };

    return cmd;
  }
}


struct TALER_TESTING_Command
TALER_TESTING_cmd_reward_authorize_from_reserve (const char *label,
                                                 const char *merchant_url,
                                                 const char *exchange_url,
                                                 const char *reserve_reference,
                                                 unsigned int http_status,
                                                 const char *justification,
                                                 const char *amount)
{
  struct RewardAuthorizeState *tas;

  tas = GNUNET_new (struct RewardAuthorizeState);
  tas->merchant_url = merchant_url;
  tas->reserve_reference = reserve_reference;
  tas->justification = justification;
  tas->http_status = http_status;
  GNUNET_assert (GNUNET_OK ==
                 TALER_string_to_amount (amount,
                                         &tas->amount));
  {
    struct TALER_TESTING_Command cmd = {
      .label = label,
      .cls = tas,
      .run = &reward_authorize_run,
      .cleanup = &reward_authorize_cleanup,
      .traits = &reward_authorize_traits
    };

    return cmd;
  }
}


struct TALER_TESTING_Command
TALER_TESTING_cmd_reward_authorize_fake (const char *label)
{
  struct RewardAuthorizeState *tas;

  tas = GNUNET_new (struct RewardAuthorizeState);
  {
    struct TALER_TESTING_Command cmd = {
      .label = label,
      .cls = tas,
      .run = &reward_authorize_fake_run,
      .cleanup = &reward_authorize_cleanup,
      .traits = &reward_authorize_traits
    };

    return cmd;
  }
}


/* end of testing_api_cmd_reward_authorize.c */
