#!/bin/bash
# This file is part of TALER
# Copyright (C) 2014-2023 Taler Systems SA
#
# TALER is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as
# published by the Free Software Foundation; either version 3, or
# (at your option) any later version.
#
# TALER is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with TALER; see the file COPYING.  If not, see
# <http://www.gnu.org/licenses/>
#

set -eu

# Replace with 0 for nexus...
USE_FAKEBANK=1
if [ 1 = "$USE_FAKEBANK" ]
then
    ACCOUNT="exchange-account-2"
    WIRE_METHOD="x-taler-bank"
    BANK_FLAGS="-f -d $WIRE_METHOD -u $ACCOUNT"
    BANK_URL="http://localhost:8082/"
else
    ACCOUNT="exchange-account-1"
    WIRE_METHOD="iban"
    BANK_FLAGS="-ns -d $WIRE_METHOD -u $ACCOUNT"
    BANK_URL="http://localhost:18082/"
fi

. setup.sh
# Launch exchange, merchant and bank.
setup -c "test_template.conf" \
      -em \
      $BANK_FLAGS
LAST_RESPONSE=$(mktemp -p "${TMPDIR:-/tmp}" test_response.conf-XXXXXX)
CONF="test_template.conf.edited"

echo -n "Configuring merchant instance ..."

STATUS=$(curl -H "Content-Type: application/json" -X POST \
    -H 'Authorization: Bearer secret-token:super_secret' \
    http://localhost:9966/management/instances \
    -d '{"auth":{"method":"external"},"id":"default","name":"default","user_type":"business","address":{},"jurisdiction":{},"use_stefan":true,"default_wire_transfer_delay":{"d_us" : 50000000},"default_pay_delay":{"d_us": 60000000}}' \
    -w "%{http_code}" -s -o /dev/null)

if [ "$STATUS" != "204" ]
then
    exit_fail "Expected 204, instance created. Got instead: $STATUS"
fi
echo "Ok"
echo -n "Configuring merchant account ..."
STATUS=$(curl -H "Content-Type: application/json" -X POST \
    -H 'Authorization: Bearer secret-token:super_secret' \
    http://localhost:9966/instances/default/private/accounts \
    -d '{"payto_uri":"payto://x-taler-bank/localhost:18082/fortythree"}' \
    -w "%{http_code}" -s -o /dev/null)

if [ "$STATUS" != "200" ]
then
    exit_fail "Expected 200 OK. Got: $STATUS"
fi

echo "OK"

echo -n "Creating reserve ..."

STATUS=$(curl 'http://localhost:9966/instances/default/private/reserves' \
    -d '{"initial_balance":"TESTKUDOS:2","exchange_url":"http://localhost:8081/","wire_method":"'"$WIRE_METHOD"'"}' \
    -w "%{http_code}" -s -o "$LAST_RESPONSE")

if [ "$STATUS" != "200" ]
then
    cat "$LAST_RESPONSE"
    exit_fail "Expected 200, reserve created. Got instead: $STATUS"
fi

echo "OK"

RESERVE_PUB=$(jq -r .reserve_pub < "$LAST_RESPONSE")

STATUS=$(curl 'http://localhost:9966/instances/default/private/reserves/'"$RESERVE_PUB" \
    -w "%{http_code}" -s -o "$LAST_RESPONSE")

FUNDED=$(jq -r '.merchant_initial_amount == .exchange_initial_amount' < "$LAST_RESPONSE")

if [ "$FUNDED" != "false" ]
then
    cat "$LAST_RESPONSE"
    exit_fail "Should not yet be funded if we just created. Got: $STATUS is founded: $FUNDED"
fi


echo -n "Wire transferring... "
# Exchange wants TESTKUDOS:2 from account 43, under RESERVE_PUB.

if [ 1 = "$USE_FAKEBANK" ]
then
    BODY='{"reserve_pub":"'"$RESERVE_PUB"'","debit_account":"payto://x-taler-bank/localhost/fortythree","amount":"TESTKUDOS:2"}'
    curl -X POST \
         -d "$BODY" \
         -s \
         http://localhost:8082/accounts/exchange/taler-wire-gateway/admin/add-incoming \
         > /dev/null
else
    EXCHANGE_PAYTO=$(get_payto_uri exchange x)
    export LIBEUFIN_SANDBOX_USERNAME=fortythree
    export LIBEUFIN_SANDBOX_PASSWORD=x
    export LIBEUFIN_SANDBOX_URL="http://localhost:18082/"
    libeufin-cli sandbox demobank new-transaction \
                 --bank-account "fortythree" \
                 --payto-with-subject "$EXCHANGE_PAYTO&message=$RESERVE_PUB" \
                 --amount "TESTKUDOS:2"
    unset LIBEUFIN_SANDBOX_USERNAME
    unset LIBEUFIN_SANDBOX_PASSWORD
    unset LIBEUFIN_SANDBOX_URL
    echo "OK"
    echo -n "Give Nexus time to detect the payment (FIXME)... "
    sleep 10 # FIXME-MS: replace with call to Nexus to right now poll the sandbox ...
    # This seems to not work (exchange user unknown). Not sure why.
    #export LIBEUFIN_NEXUS_USERNAME=exchange
    #export LIBEUFIN_NEXUS_PASSWORD=x
    #export LIBEUFIN_NEXUS_URL="http://localhost:8082/"
    #libeufin-cli \
        #    accounts \
        #    fetch-transactions \
        #    exchange \
        #    &> libeufin-transfer-fetch.out
    #unset LIBEUFIN_NEXUS_USERNAME
    #unset LIBEUFIN_NEXUS_PASSWORD
    #unset LIBEUFIN_NEXUS_URL
fi
echo "OK"

echo "Fetch transaction for exchange"
# Run wirewatch here, now, and only once
taler-exchange-wirewatch \
    -c "$CONF" \
    -t \
    -L "INFO" \
    &> taler-exchange-wirewatch.log

STATUS=$(curl 'http://localhost:9966/instances/default/private/reserves/'"$RESERVE_PUB" \
    -w "%{http_code}" -s -o "$LAST_RESPONSE")
FUNDED=$(jq -r '.merchant_initial_amount == .exchange_initial_amount' < "$LAST_RESPONSE")

if [ "$FUNDED" != "true" ]
then
    cat "$LAST_RESPONSE"
    exit_fail "should be funded. got: $STATUS is founded: $FUNDED"
fi

ACCOUNTS=$(jq -r '.accounts|length' < "$LAST_RESPONSE")
if [ "x$ACCOUNTS" != "x1" ]
then
    cat "$LAST_RESPONSE"
    exit_fail "Expected 1 account in response. got: $ACCOUNTS"
fi

echo -n "authorizing reward ..."

STATUS=$(curl 'http://localhost:9966/instances/default/private/reserves/'"$RESERVE_PUB"'/authorize-reward' \
    -d '{"amount":"TESTKUDOS:1","justification":"off course","next_url":"https://taler.net/"}' \
    -w "%{http_code}" -s -o "$LAST_RESPONSE")

if [ "$STATUS" != "200" ]
then
    exit_fail "should respond failed, we did not fund yet. got: $STATUS"
fi
echo "OK"

echo -n "Checking reward ..."
STATUS=$(curl 'http://localhost:9966/instances/default/private/reserves/'"$RESERVE_PUB"'?rewards=yes' \
    -w "%{http_code}" -s -o "$LAST_RESPONSE")

REWARDS_SIZE=$(jq -r ".rewards | length"  < "$LAST_RESPONSE")

if [ "$REWARDS_SIZE" != "1" ]
then
    cat "$LAST_RESPONSE"
    exit_fail "Expected 1 reward. got: $REWARDS_SIZE"
fi

REWARD_ID=$(jq -r .rewards[0].reward_id < "$LAST_RESPONSE")

echo "Found"

echo -n "Checking reward status ..."

STATUS=$(curl 'http://localhost:9966/instances/default/private/rewards/'"$REWARD_ID" \
    -w "%{http_code}" -s -o "$LAST_RESPONSE")

if [ "$STATUS" != "200" ]
then
    cat "$LAST_RESPONSE"
    exit_fail "Expected 200, reward found. got: $STATUS"
fi

echo -n " ... "

STATUS=$(curl 'http://localhost:9966/instances/default/private/rewards/'"$REWARD_ID"'?pickups=yes' \
    -w "%{http_code}" -s -o "$LAST_RESPONSE")

if [ "$STATUS" != "200" ]
then
    cat "$LAST_RESPONSE"
    exit_fail "Expected 200, reward found. got: $STATUS"
fi
echo "OK"

echo -n "trying to create invalid reserve ..."

STATUS=$(curl 'http://localhost:9966/instances/default/private/reserves' \
    -d '{"initial_balance":"INVALID:2","exchange_url":"http://localhost:8081/","wire_method":"iban"}' \
    -w "%{http_code}" -s -o "$LAST_RESPONSE")

if [ "$STATUS" != "400" ]
then
    exit_fail "Expected 400, bad currency. got: $STATUS"
fi
echo "FAILED (which is expected)"

echo "Test PASSED"

exit 0
