/*
   This file is part of TALER
   Copyright (C) 2022 Taler Systems SA

   TALER is free software; you can redistribute it and/or modify it under the
   terms of the GNU General Public License as published by the Free Software
   Foundation; either version 3, or (at your option) any later version.

   TALER is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
   A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along with
   TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
/**
 * @file backenddb/pg_update_product.c
 * @brief Implementation of the update_product function for Postgres
 * @author Christian Grothoff
 * @author Iván Ávalos
 */
#include "platform.h"
#include <taler/taler_error_codes.h>
#include <taler/taler_dbevents.h>
#include <taler/taler_pq_lib.h>
#include "pg_update_product.h"
#include "pg_helper.h"

enum GNUNET_DB_QueryStatus
TMH_PG_update_product (void *cls,
                       const char *instance_id,
                       const char *product_id,
                       const struct TALER_MERCHANTDB_ProductDetails *pd)
{
  struct PostgresClosure *pg = cls;
  struct GNUNET_PQ_QueryParam params[] = {
    GNUNET_PQ_query_param_string (instance_id), /* $1 */
    GNUNET_PQ_query_param_string (product_id),
    GNUNET_PQ_query_param_string (pd->description),
    TALER_PQ_query_param_json (pd->description_i18n),
    GNUNET_PQ_query_param_string (pd->unit),
    GNUNET_PQ_query_param_string (pd->image), /* $6 */
    TALER_PQ_query_param_json (pd->taxes),
    TALER_PQ_query_param_amount_with_currency (pg->conn,
                                 &pd->price),       /* $8 */
    GNUNET_PQ_query_param_uint64 (&pd->total_stock),  /* $9 */
    GNUNET_PQ_query_param_uint64 (&pd->total_lost),
    TALER_PQ_query_param_json (pd->address),
    GNUNET_PQ_query_param_timestamp (&pd->next_restock),
    GNUNET_PQ_query_param_uint32 (&pd->minimum_age),
    GNUNET_PQ_query_param_end
  };

  if ( (pd->total_stock < pd->total_lost + pd->total_sold) ||
       (pd->total_lost < pd->total_lost
        + pd->total_sold) /* integer overflow */)
  {
    GNUNET_break (0);
    return GNUNET_DB_STATUS_HARD_ERROR;
  }
  check_connection (pg);
  PREPARE (pg,
           "update_product",
           "UPDATE merchant_inventory SET"
           " description=$3"
           ",description_i18n=$4"
           ",unit=$5"
           ",image=$6"
           ",taxes=$7"
           ",price=$8"
           ",total_stock=$9"
           ",total_lost=$10"
           ",address=$11"
           ",next_restock=$12"
           ",minimum_age=$13"
           " WHERE merchant_serial="
           "   (SELECT merchant_serial"
           "      FROM merchant_instances"
           "      WHERE merchant_id=$1)"
           "   AND product_id=$2"
           "   AND total_stock <= $9"
           "   AND total_lost <= $10");
  return GNUNET_PQ_eval_prepared_non_select (pg->conn,
                                             "update_product",
                                             params);
}
