/*
   This file is part of TALER
   Copyright (C) 2022 Taler Systems SA

   TALER is free software; you can redistribute it and/or modify it under the
   terms of the GNU General Public License as published by the Free Software
   Foundation; either version 3, or (at your option) any later version.

   TALER is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
   A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along with
   TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
/**
 * @file backenddb/pg_authorize_reward.h
 * @brief implementation of the authorize_reward function for Postgres
 * @author Christian Grothoff
 */
#ifndef PG_AUTHORIZE_REWARD_H
#define PG_AUTHORIZE_REWARD_H

#include <taler/taler_util.h>
#include <taler/taler_json_lib.h>
#include "taler_merchantdb_plugin.h"


/**
 * Authorize a reward over @a amount from reserve @a reserve_pub.  Remember
 * the authorization under @a reward_id for later, together with the
 * @a justification.
 *
 * @param cls closure, typically a connection to the db
 * @param instance_id which instance should generate the reward
 * @param reserve_pub which reserve is debited, NULL to pick one in the DB
 * @param amount how high is the reward (with fees)
 * @param justification why was the reward approved
 * @param next_url where to send the URL post reward pickup
 * @param[out] reward_id set to the unique ID for the reward
 * @param[out] expiration set to when the reward expires
 * @return transaction status,
 *      #TALER_EC_MERCHANT_PRIVATE_POST_REWARD_AUTHORIZE_RESERVE_EXPIRED if the reserve is known but has expired
 *      #TALER_EC_MERCHANT_PRIVATE_POST_REWARD_AUTHORIZE_RESERVE_NOT_FOUND if the reserve is not known
 *      #TALER_EC_MERCHANT_PRIVATE_POST_REWARD_AUTHORIZE_INSUFFICIENT_FUNDS if the reserve has insufficient funds left
 *      #TALER_EC_GENERIC_DB_START_FAILED on hard DB errors
 *      #TALER_EC_GENERIC_DB_FETCH_FAILED on hard DB errors
 *      #TALER_EC_GENERIC_DB_STORE_FAILED on hard DB errors
 *      #TALER_EC_GENERIC_DB_INVARIANT_FAILURE on hard DB errors
 *      #TALER_EC_GENERIC_DB_SOFT_FAILURE soft DB errors (client should retry)
 *      #TALER_EC_NONE upon success
 */
enum TALER_ErrorCode
TMH_PG_authorize_reward (void *cls,
                         const char *instance_id,
                         const struct TALER_ReservePublicKeyP *reserve_pub,
                         const struct TALER_Amount *amount,
                         const char *justification,
                         const char *next_url,
                         struct TALER_RewardIdentifierP *reward_id,
                         struct GNUNET_TIME_Timestamp *expiration);


#endif
