/*
  This file is part of TALER
  Copyright (C) 2022 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as
  published by the Free Software Foundation; either version 3, or
  (at your option) any later version.

  TALER is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public
  License along with TALER; see the file COPYING.  If not, see
  <http://www.gnu.org/licenses/>
*/
/**
 * @file testing_api_cmd_post_using_templates.c
 * @brief command to test POST /using-templates
 * @author Priscilla HUANG
 */
#include "platform.h"
#include <taler/taler_exchange_service.h>
#include <taler/taler_testing_lib.h>
#include "taler_merchant_service.h"
#include "taler_merchant_testing_lib.h"

/**
 * State of a "POST /templates" CMD.
 */
struct PostUsingTemplatesState
{

  /**
   * Handle for a "GET using-template" request.
   */
  struct TALER_MERCHANT_UsingTemplatesPostHandle *iph;

  /**
   * The interpreter state.
   */
  struct TALER_TESTING_Interpreter *is;

  /**
   * Base URL of the merchant serving the request.
   */
  const char *merchant_url;

  /**
   * Summary given by the customer.
   */
  const char *summary;

  /**
   * Amount given by the customer.
   */
  struct TALER_Amount amount;

  /**
   * Label of a command that created the template we should use.
   */
  const char *template_ref;

  /**
   * Expected HTTP response code.
   */
  unsigned int http_status;

};


/**
 * Callback for a POST /using-templates operation.
 *
 * @param cls closure for this function
 * @param por response being processed
 */
static void
post_using_templates_cb (void *cls,
                         const struct TALER_MERCHANT_PostOrdersReply *por)
{
  struct PostUsingTemplatesState *tis = cls;

  tis->iph = NULL;
  if (tis->http_status != por->hr.http_status)
  {
    GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                "Unexpected response code %u (%d) to command %s\n",
                por->hr.http_status,
                (int) por->hr.ec,
                TALER_TESTING_interpreter_get_current_label (tis->is));
    TALER_TESTING_interpreter_fail (tis->is);
    return;
  }
  switch (por->hr.http_status)
  {
  case MHD_HTTP_OK:
    break;
  case MHD_HTTP_CONFLICT:
    break;
  case MHD_HTTP_NOT_FOUND:
    break;
  default:
    GNUNET_break (0);
    GNUNET_log (GNUNET_ERROR_TYPE_WARNING,
                "Unhandled HTTP status %u for POST /templates/$ID.\n",
                por->hr.http_status);
    break;
  }
  TALER_TESTING_interpreter_next (tis->is);
}


/**
 * Run the "POST /using-templates" CMD.
 *
 *
 * @param cls closure.
 * @param cmd command being run now.
 * @param is interpreter state.
 */
static void
post_using_templates_run (void *cls,
                          const struct TALER_TESTING_Command *cmd,
                          struct TALER_TESTING_Interpreter *is)
{
  struct PostUsingTemplatesState *tis = cls;
  const struct TALER_TESTING_Command *ref;
  const char **template_id;

  tis->is = is;
  ref = TALER_TESTING_interpreter_lookup_command (is,
                                                  tis->template_ref);
  if (GNUNET_OK !=
      TALER_TESTING_get_trait_template_id (ref,
                                           &template_id))
    TALER_TESTING_FAIL (is);
  tis->iph = TALER_MERCHANT_using_templates_post (
    is->ctx,
    tis->merchant_url,
    *template_id,
    tis->summary,
    TALER_amount_is_valid (&tis->amount)
    ? &tis->amount
    : NULL,
    &post_using_templates_cb,
    tis);
  GNUNET_assert (NULL != tis->iph);
}


/**
 * Offers information from the POST /using-templates CMD state to other
 * commands.
 *
 * @param cls closure
 * @param[out] ret result (could be anything)
 * @param trait name of the trait
 * @param index index number of the object to extract.
 * @return #GNUNET_OK on success
 */
static enum GNUNET_GenericReturnValue
post_using_templates_traits (void *cls,
                             const void **ret,
                             const char *trait,
                             unsigned int index)
{
  struct PostUsingTemplatesState *pts = cls;
  struct TALER_TESTING_Trait traits[] = {
    TALER_TESTING_trait_end (),
  };

  (void) pts;
  return TALER_TESTING_get_trait (traits,
                                  ret,
                                  trait,
                                  index);
}


/**
 * Free the state of a "POST using-template" CMD, and possibly
 * cancel a pending operation thereof.
 *
 * @param cls closure.
 * @param cmd command being run.
 */
static void
post_using_templates_cleanup (void *cls,
                              const struct TALER_TESTING_Command *cmd)
{
  struct PostUsingTemplatesState *tis = cls;

  if (NULL != tis->iph)
  {
    GNUNET_log (GNUNET_ERROR_TYPE_WARNING,
                "POST /using-templates operation did not complete\n");
    TALER_MERCHANT_using_templates_post_cancel (tis->iph);
  }
  GNUNET_free (tis);
}


struct TALER_TESTING_Command
TALER_TESTING_cmd_merchant_post_using_templates (
  const char *label,
  const char *template_ref,
  const char *merchant_url,
  const char *summary,
  const char *amount,  unsigned int http_status)
{
  struct PostUsingTemplatesState *tis;

  tis = GNUNET_new (struct PostUsingTemplatesState);
  tis->template_ref = template_ref;
  tis->merchant_url = merchant_url;
  tis->http_status = http_status;
  tis->summary = summary;
  if (NULL != amount)
    GNUNET_assert (GNUNET_OK ==
                   TALER_string_to_amount (amount,
                                           &tis->amount));
  {
    struct TALER_TESTING_Command cmd = {
      .cls = tis,
      .label = label,
      .run = &post_using_templates_run,
      .cleanup = &post_using_templates_cleanup,
      .traits = &post_using_templates_traits
    };

    return cmd;
  }
}


/* end of testing_api_cmd_post_using_templates.c */
