/* @licstart  The following is the entire license notice for the
  JavaScript code in this page.

  Copyright (C) 2015, 2016 INRIA

  The JavaScript code in this page is free software: you can
  redistribute it and/or modify it under the terms of the GNU
  Lesser General Public License (GNU LGPL) as published by the Free Software
  Foundation, either version 2.1 of the License, or (at your option)
  any later version.  The code is distributed WITHOUT ANY WARRANTY;
  without even the implied warranty of MERCHANTABILITY or FITNESS
  FOR A PARTICULAR PURPOSE.  See the GNU LGPL for more details.

  As additional permission under GNU LGPL version 2.1 section 7, you
  may distribute non-source (e.g., minimized or compacted) forms of
  that code without the copy of the GNU LGPL normally required by
  section 4, provided you include this license notice and a URL
  through which recipients can access the Corresponding Source.

  @licend  The above is the entire license notice
  for the JavaScript code in this page.

  @author Marcello Stanici
  @author Florian Dold
*/

var taler = (<any>window).taler || {};

(function () {


/**
 * Error handler for things that go wrong in the merchant
 * frontend browser code.
 * TODO: allow the possibility to override this to
 * install a custom handler
 */
function raise_error(message) {
  alert(message);
  throw Error(message);
}

taler.executePayment = function (contractHash, payUrl, offeringUrl) {
  if (!contractHash) {
    raise_error("protocol violation: contract hash must be non-empty string");
  }
  let detail = {
    H_contract: contractHash,
    offering_url: offeringUrl
  };

  function subst(url, contractHash) {
    url = url.replace("${H_contract}", contractHash);
    url = url.replace("${$}", "$");
    return url;
  }

  /**
   * Handle a failed payment.
   *
   * Try to notify the wallet first, before we show a potentially
   * synchronous error message (such as an alert) or leave the page.
   */
  function handleFailedPayment(status) {
    let timeoutHandle = null;
    function err() {
      raise_error("Payment failed: Unexpected status code for $pay_url: " + status);
    }
    function onResp() {
      if (timeoutHandle != null) {
        clearTimeout(timeoutHandle);
        timeoutHandle = null;
      }
      err();
    }
    function onTimeout() {
      timeoutHandle = null
      err();
    }
    let eve = new CustomEvent('taler-payment-failed', {detail: {}});
    document.dispatchEvent(eve);
    document.addEventListener("taler-payment-failed-ok", onResp, false);
    timeoutHandle = setTimeout(onTimeout, 200);
  }

  function handleResponse(evt) {
    console.log("handling taler-notify-payment");
    // Payment timeout in ms.
    let timeout_ms = 1000;
    // Current request.
    let r;
    let timeoutHandle = null;
    function sendPay() {
      r = new XMLHttpRequest();
      r.open("post", payUrl);
      r.setRequestHeader("Content-Type", "application/json;charset=UTF-8");
      r.send(JSON.stringify(evt.detail.payment));
      r.onload = function() {
        switch (r.status) {
          case 200:
            window.location.href = subst(evt.detail.contract.fulfillment_url,
                                         evt.detail.H_contract);
            window.location.reload(true);
            break;
          default:
            handleFailedPayment(r.status);
            break;
        }
        r = null;
        if (timeoutHandle != null) {
          clearTimeout(timeoutHandle);
          timeoutHandle = null;
        }
      };
      function retry() {
        if (r) {
          r.abort();
          r = null;
        }
        timeout_ms = Math.min(timeout_ms * 2, 10 * 1000);
        console.log("sendPay timed out, retrying in ", timeout_ms, "ms");
        sendPay();
      }
      timeoutHandle = setTimeout(retry, timeout_ms);
    }
    sendPay();
  }

  document.addEventListener("taler-notify-payment", handleResponse, false);
  let eve = new CustomEvent('taler-execute-contract', {detail: detail});
  document.dispatchEvent(eve);
}

/**
 * Stop the presence detection, useful for
 * debugging in the browser.
 */
taler.stop = function () {
  document.removeEventListener("taler-wallet-present", probeTaler);
}

var presentHandlers = [];
var absentHandlers = [];

taler.onPresent = function (f) {
  presentHandlers.push(f);
}

taler.onAbsent = function (f) {
  absentHandlers.push(f);
}

function handleInstall() {
  for (var i = 0; i < presentHandlers.length; i++) {
    presentHandlers[i]();
  }
  var show = <any>document.getElementsByClassName("taler-installed-show");
  var hide = <any>document.getElementsByClassName("taler-installed-hide");
  var activate = <any>document.getElementsByClassName("taler-installed-activate");
  for (var i = 0; i < show.length; i++) {
    show[i].style.display = "inherit";
  }
  for (var i = 0; i < hide.length; i++) {
    hide[i].style.display = "none";
  }
  for (var i = 0; i < activate.length; i++) {
    activate[i].disabled = false;
  }
};

function hideElements() {
  var show = <any>document.getElementsByClassName("taler-installed-show");
  var hide = <any>document.getElementsByClassName("taler-installed-hide");
  var activate = <any>document.getElementsByClassName("taler-installed-activate");
  for (var i = 0; i < show.length; i++) {
    show[i].style.display = "none";
  }
  for (var i = 0; i < hide.length; i++) {
    hide[i].style.display = "inherit";
  }
  for (var i = 0; i < activate.length; i++) {
    activate[i].disabled = true;
  }
}

function handleUninstall() {
  for (var i = 0; i < absentHandlers.length; i++) {
    absentHandlers[i]();
  }
  hideElements();
};


var installed = false;
var uninstallCalled = false;
var probeExecuted = false;


function handleProbe() {
  probeExecuted = true;
  if (!installed) {
    console.log("taler install detected");
    handleInstall();
  }
  installed = true;
}

function probeTaler() {
  probeExecuted = false;
  var eve = new Event("taler-probe");
  document.dispatchEvent(eve);
}

function onTimeout() {
  if (!probeExecuted) {
    if (installed || !uninstallCalled) {
      console.log("taler uninstall detected");
      handleUninstall();
    } 
    installed = false;
  }
  // try again, maybe it'll be installed ...
  probeTaler();
}



function handle_contract(contract_wrapper) {
  var cEvent = new CustomEvent("taler-confirm-contract", {
    detail: {
      contract_wrapper: contract_wrapper,
      replace_navigation: true
    }
  });
  document.dispatchEvent(cEvent);
};


/**
 * Offer a contract to the wallet after
 * downloading it from the given URL.
 */
taler.offerContractFrom = function (url) {
  var contract_request = new XMLHttpRequest();
  console.log("downloading contract from '" + url + "'")
  contract_request.open("GET", url, true);
  contract_request.onload = function (e) {
    if (contract_request.readyState == 4) {
      if (contract_request.status == 200) {
        console.log("response text:",
                    contract_request.responseText);
        var contract_wrapper = JSON.parse(contract_request.responseText);
        if (!contract_wrapper) {
          console.error("response text was invalid json");
          alert("Failure to download contract (invalid json)");
          return;
        }
        handle_contract(contract_wrapper);
      } else {
      alert("Failure to download contract from merchant " +
            "(" + contract_request.status + "):\n" +
            contract_request.responseText);
      }
    } 
  };
  contract_request.onerror = function (e) {
    alert("Failure requesting the contract:\n"
          + contract_request.statusText);
    };
  contract_request.send();
}


document.addEventListener("taler-wallet-present", handleProbe, false);

function initTaler() {
  hideElements();
  probeTaler();
  window.setInterval(onTimeout, 300);
}

window.addEventListener("load", initTaler, false);

})();

