--
-- This file is part of TALER
-- Copyright (C) 2014--2023 Taler Systems SA
--
-- TALER is free software; you can redistribute it and/or modify it under the
-- terms of the GNU General Public License as published by the Free Software
-- Foundation; either version 3, or (at your option) any later version.
--
-- TALER is distributed in the hope that it will be useful, but WITHOUT ANY
-- WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
-- A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License along with
-- TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
--

CREATE FUNCTION create_table_reserves_open_requests(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT default 'reserves_open_requests';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(open_request_uuid BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',reserve_pub BYTEA NOT NULL'
      ',request_timestamp INT8 NOT NULL'
      ',expiration_date INT8 NOT NULL'
      ',reserve_sig BYTEA NOT NULL CHECK (LENGTH(reserve_sig)=64)'
      ',reserve_payment taler_amount NOT NULL'
      ',requested_purse_limit INT4 NOT NULL'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (reserve_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table (
     'requests to keep a reserve open'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column (
     'Fee to pay for the request from the reserve balance itself.'
    ,'reserve_payment'
    ,table_name
    ,partition_suffix
  );
END
$$;


CREATE FUNCTION constrain_table_reserves_open_requests(
  IN partition_suffix TEXT
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT default 'reserves_open_requests';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_by_uuid'
    ' PRIMARY KEY (open_request_uuid)'
    ',ADD CONSTRAINT ' || table_name || '_by_time'
    ' UNIQUE (reserve_pub,request_timestamp)'
  );
END
$$;


CREATE FUNCTION foreign_table_reserves_open_requests()
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT default 'reserves_open_requests';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_reserve_pub '
    ' FOREIGN KEY (reserve_pub)'
    ' REFERENCES reserves (reserve_pub) ON DELETE CASCADE'
  );
END
$$;


CREATE OR REPLACE FUNCTION reserves_open_requests_insert_trigger()
  RETURNS trigger
  LANGUAGE plpgsql
  AS $$
BEGIN
  INSERT INTO reserve_history
    (reserve_pub
    ,table_name
    ,serial_id)
  VALUES
    (NEW.reserve_pub
    ,'reserves_open_requests'
    ,NEW.open_request_uuid);
  RETURN NEW;
END $$;
COMMENT ON FUNCTION reserves_open_requests_insert_trigger()
  IS 'Automatically generate reserve history entry.';


CREATE FUNCTION master_table_reserves_open_requests()
RETURNS VOID
LANGUAGE plpgsql
AS $$
BEGIN
  CREATE TRIGGER reserves_open_requests_on_insert
    AFTER INSERT
     ON reserves_open_requests
     FOR EACH ROW EXECUTE FUNCTION reserves_open_requests_insert_trigger();
END $$;


INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('reserves_open_requests'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('reserves_open_requests'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('reserves_open_requests'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE),
    ('reserves_open_requests'
    ,'exchange-0002'
    ,'master'
    ,TRUE
    ,FALSE);
