/*
     This file is part of GNU Taler
     Copyright (C) 2012-2020 Taler Systems SA

     GNU Taler is free software: you can redistribute it and/or modify it
     under the terms of the GNU Lesser General Public License as published
     by the Free Software Foundation, either version 3 of the License,
     or (at your option) any later version.

     GNU Taler is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     Lesser General Public License for more details.

     You should have received a copy of the GNU Lesser General Public License
     along with this program.  If not, see <http://www.gnu.org/licenses/>.

     SPDX-License-Identifier: LGPL3.0-or-later

     Note: the LGPL does not apply to all components of GNU Taler,
     but it does apply to this file.
 */
#include "taler_error_codes.h"
#include <stddef.h>
#include <microhttpd.h>
#include <gettext.h>

/**
 * MHD does not define our value for 0 (client-side generated code).
 */
#define MHD_HTTP_UNINITIALIZED 0

/**
 * A pair containing an error code and its hint.
 */
struct ErrorCodeAndHint
{
  /**
   * The error code.
   */
  enum TALER_ErrorCode ec;

  /**
   * The hint.
   */
  const char *hint;

  /**
   * The HTTP status code.
   */
  unsigned int http_code;
};


/**
 * The list of all error codes with their hints.
 */
static const struct ErrorCodeAndHint code_hint_pairs[] = {

  {
    .ec = TALER_EC_NONE,
    .hint = gettext_noop ("Special code to indicate success (no error)."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_INVALID,
    .hint = gettext_noop ("An error response did not include an error code in the format expected by the client. Most likely, the server does not speak the GNU Taler protocol. Check the URL and/or the network connection to the server."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_GENERIC_CLIENT_INTERNAL_ERROR,
    .hint = gettext_noop ("An internal failure happened on the client side. Details should be in the local logs. Check if you are using the latest available version or file a report with the developers."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_GENERIC_INVALID_RESPONSE,
    .hint = gettext_noop ("The response we got from the server was not in the expected format. Most likely, the server does not speak the GNU Taler protocol. Check the URL and/or the network connection to the server."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_GENERIC_TIMEOUT,
    .hint = gettext_noop ("The operation timed out. Trying again might help. Check the network connection."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_GENERIC_VERSION_MALFORMED,
    .hint = gettext_noop ("The protocol version given by the server does not follow the required format. Most likely, the server does not speak the GNU Taler protocol. Check the URL and/or the network connection to the server."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_GENERIC_REPLY_MALFORMED,
    .hint = gettext_noop ("The service responded with a reply that was in the right data format, but the content did not satisfy the protocol. Please file a bug report."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_GENERIC_CONFIGURATION_INVALID,
    .hint = gettext_noop ("There is an error in the client-side configuration, for example an option is set to an invalid value. Check the logs and fix the local configuration."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_GENERIC_UNEXPECTED_REQUEST_ERROR,
    .hint = gettext_noop ("The client made a request to a service, but received an error response it does not know how to handle. Please file a bug report."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_GENERIC_TOKEN_PERMISSION_INSUFFICIENT,
    .hint = gettext_noop ("The token used by the client to authorize the request does not grant the required permissions for the request. Check the requirements and obtain a suitable authorization token to proceed."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_GENERIC_METHOD_INVALID,
    .hint = gettext_noop ("The HTTP method used is invalid for this endpoint. This is likely a bug in the client implementation. Check if you are using the latest available version and/or file a report with the developers."),
    .http_code = MHD_HTTP_METHOD_NOT_ALLOWED
  },

  {
    .ec = TALER_EC_GENERIC_ENDPOINT_UNKNOWN,
    .hint = gettext_noop ("There is no endpoint defined for the URL provided by the client. Check if you used the correct URL and/or file a report with the developers of the client software."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_GENERIC_JSON_INVALID,
    .hint = gettext_noop ("The JSON in the client's request was malformed. This is likely a bug in the client implementation. Check if you are using the latest available version and/or file a report with the developers."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_GENERIC_HTTP_HEADERS_MALFORMED,
    .hint = gettext_noop ("Some of the HTTP headers provided by the client were malformed and caused the server to not be able to handle the request. This is likely a bug in the client implementation. Check if you are using the latest available version and/or file a report with the developers."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_GENERIC_PAYTO_URI_MALFORMED,
    .hint = gettext_noop ("The payto:// URI provided by the client is malformed. Check that you are using the correct syntax as of RFC 8905 and/or that you entered the bank account number correctly."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_GENERIC_PARAMETER_MISSING,
    .hint = gettext_noop ("A required parameter in the request was missing. This is likely a bug in the client implementation. Check if you are using the latest available version and/or file a report with the developers."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_GENERIC_PARAMETER_MALFORMED,
    .hint = gettext_noop ("A parameter in the request was malformed. This is likely a bug in the client implementation. Check if you are using the latest available version and/or file a report with the developers."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_GENERIC_RESERVE_PUB_MALFORMED,
    .hint = gettext_noop ("The reserve public key was malformed."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_GENERIC_COMPRESSION_INVALID,
    .hint = gettext_noop ("The body in the request could not be decompressed by the server. This is likely a bug in the client implementation. Check if you are using the latest available version and/or file a report with the developers."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_GENERIC_CURRENCY_MISMATCH,
    .hint = gettext_noop ("The currency involved in the operation is not acceptable for this server. Check your configuration and make sure the currency specified for a given service provider is one of the currencies supported by that provider."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_GENERIC_URI_TOO_LONG,
    .hint = gettext_noop ("The URI is longer than the longest URI the HTTP server is willing to parse. If you believe this was a legitimate request, contact the server administrators and/or the software developers to increase the limit."),
    .http_code = MHD_HTTP_URI_TOO_LONG
  },

  {
    .ec = TALER_EC_GENERIC_UPLOAD_EXCEEDS_LIMIT,
    .hint = gettext_noop ("The body is too large to be permissible for the endpoint. If you believe this was a legitimate request, contact the server administrators and/or the software developers to increase the limit."),
    .http_code = MHD_HTTP_CONTENT_TOO_LARGE
  },

  {
    .ec = TALER_EC_GENERIC_UNAUTHORIZED,
    .hint = gettext_noop ("The service refused the request due to lack of proper authorization."),
    .http_code = MHD_HTTP_UNAUTHORIZED
  },

  {
    .ec = TALER_EC_GENERIC_TOKEN_UNKNOWN,
    .hint = gettext_noop ("The service refused the request as the given authorization token is unknown."),
    .http_code = MHD_HTTP_UNAUTHORIZED
  },

  {
    .ec = TALER_EC_GENERIC_TOKEN_EXPIRED,
    .hint = gettext_noop ("The service refused the request as the given authorization token expired."),
    .http_code = MHD_HTTP_UNAUTHORIZED
  },

  {
    .ec = TALER_EC_GENERIC_TOKEN_MALFORMED,
    .hint = gettext_noop ("The service refused the request as the given authorization token is malformed."),
    .http_code = MHD_HTTP_UNAUTHORIZED
  },

  {
    .ec = TALER_EC_GENERIC_FORBIDDEN,
    .hint = gettext_noop ("The service refused the request due to lack of proper rights on the resource."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_GENERIC_DB_SETUP_FAILED,
    .hint = gettext_noop ("The service failed initialize its connection to the database. The system administrator should check that the service has permissions to access the database and that the database is running."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_GENERIC_DB_START_FAILED,
    .hint = gettext_noop ("The service encountered an error event to just start the database transaction. The system administrator should check that the database is running."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_GENERIC_DB_STORE_FAILED,
    .hint = gettext_noop ("The service failed to store information in its database. The system administrator should check that the database is running and review the service logs."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_GENERIC_DB_FETCH_FAILED,
    .hint = gettext_noop ("The service failed to fetch information from its database. The system administrator should check that the database is running and review the service logs."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_GENERIC_DB_COMMIT_FAILED,
    .hint = gettext_noop ("The service encountered an unrecoverable error trying to commit a transaction to the database. The system administrator should check that the database is running and review the service logs."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_GENERIC_DB_SOFT_FAILURE,
    .hint = gettext_noop ("The service encountered an error event to commit the database transaction, even after repeatedly retrying it there was always a conflicting transaction. This indicates a repeated serialization error; it should only happen if some client maliciously tries to create conflicting concurrent transactions. It could also be a sign of a missing index. Check if you are using the latest available version and/or file a report with the developers."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_GENERIC_DB_INVARIANT_FAILURE,
    .hint = gettext_noop ("The service's database is inconsistent and violates service-internal invariants. Check if you are using the latest available version and/or file a report with the developers."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_GENERIC_INTERNAL_INVARIANT_FAILURE,
    .hint = gettext_noop ("The HTTP server experienced an internal invariant failure (bug). Check if you are using the latest available version and/or file a report with the developers."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_GENERIC_FAILED_COMPUTE_JSON_HASH,
    .hint = gettext_noop ("The service could not compute a cryptographic hash over some JSON value. Check if you are using the latest available version and/or file a report with the developers."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_GENERIC_FAILED_COMPUTE_AMOUNT,
    .hint = gettext_noop ("The service could not compute an amount. Check if you are using the latest available version and/or file a report with the developers."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_GENERIC_PARSER_OUT_OF_MEMORY,
    .hint = gettext_noop ("The HTTP server had insufficient memory to parse the request. Restarting services periodically can help, especially if Postgres is using excessive amounts of memory. Check with the system administrator to investigate."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_GENERIC_ALLOCATION_FAILURE,
    .hint = gettext_noop ("The HTTP server failed to allocate memory. Restarting services periodically can help, especially if Postgres is using excessive amounts of memory. Check with the system administrator to investigate."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_GENERIC_JSON_ALLOCATION_FAILURE,
    .hint = gettext_noop ("The HTTP server failed to allocate memory for building JSON reply. Restarting services periodically can help, especially if Postgres is using excessive amounts of memory. Check with the system administrator to investigate."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_GENERIC_CURL_ALLOCATION_FAILURE,
    .hint = gettext_noop ("The HTTP server failed to allocate memory for making a CURL request. Restarting services periodically can help, especially if Postgres is using excessive amounts of memory. Check with the system administrator to investigate."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_GENERIC_FAILED_TO_LOAD_TEMPLATE,
    .hint = gettext_noop ("The backend could not locate a required template to generate an HTML reply. The system administrator should check if the resource files are installed in the correct location and are readable to the service."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_GENERIC_FAILED_TO_EXPAND_TEMPLATE,
    .hint = gettext_noop ("The backend could not expand the template to generate an HTML reply. The system administrator should investigate the logs and check if the templates are well-formed."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_EXCHANGE_GENERIC_BAD_CONFIGURATION,
    .hint = gettext_noop ("Exchange is badly configured and thus cannot operate."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_EXCHANGE_GENERIC_OPERATION_UNKNOWN,
    .hint = gettext_noop ("Operation specified unknown for this endpoint."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_EXCHANGE_GENERIC_WRONG_NUMBER_OF_SEGMENTS,
    .hint = gettext_noop ("The number of segments included in the URI does not match the number of segments expected by the endpoint."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_EXCHANGE_GENERIC_COIN_CONFLICTING_DENOMINATION_KEY,
    .hint = gettext_noop ("The same coin was already used with a different denomination previously."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_EXCHANGE_GENERIC_COINS_INVALID_COIN_PUB,
    .hint = gettext_noop ("The public key of given to a \"/coins/\" endpoint of the exchange was malformed."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_EXCHANGE_GENERIC_DENOMINATION_KEY_UNKNOWN,
    .hint = gettext_noop ("The exchange is not aware of the denomination key the wallet requested for the operation."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_EXCHANGE_DENOMINATION_SIGNATURE_INVALID,
    .hint = gettext_noop ("The signature of the denomination key over the coin is not valid."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_EXCHANGE_GENERIC_KEYS_MISSING,
    .hint = gettext_noop ("The exchange failed to perform the operation as it could not find the private keys. This is a problem with the exchange setup, not with the client's request."),
    .http_code = MHD_HTTP_SERVICE_UNAVAILABLE
  },

  {
    .ec = TALER_EC_EXCHANGE_GENERIC_DENOMINATION_VALIDITY_IN_FUTURE,
    .hint = gettext_noop ("Validity period of the denomination lies in the future."),
    .http_code = MHD_HTTP_PRECONDITION_FAILED
  },

  {
    .ec = TALER_EC_EXCHANGE_GENERIC_DENOMINATION_EXPIRED,
    .hint = gettext_noop ("Denomination key of the coin is past its expiration time for the requested operation."),
    .http_code = MHD_HTTP_GONE
  },

  {
    .ec = TALER_EC_EXCHANGE_GENERIC_DENOMINATION_REVOKED,
    .hint = gettext_noop ("Denomination key of the coin has been revoked."),
    .http_code = MHD_HTTP_GONE
  },

  {
    .ec = TALER_EC_EXCHANGE_GENERIC_SECMOD_TIMEOUT,
    .hint = gettext_noop ("An operation where the exchange interacted with a security module timed out."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_EXCHANGE_GENERIC_INSUFFICIENT_FUNDS,
    .hint = gettext_noop ("The respective coin did not have sufficient residual value for the operation.  The \"history\" in this response provides the \"residual_value\" of the coin, which may be less than its \"original_value\"."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_EXCHANGE_GENERIC_COIN_HISTORY_COMPUTATION_FAILED,
    .hint = gettext_noop ("The exchange had an internal error reconstructing the transaction history of the coin that was being processed."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_EXCHANGE_GENERIC_HISTORY_DB_ERROR_INSUFFICIENT_FUNDS,
    .hint = gettext_noop ("The exchange failed to obtain the transaction history of the given coin from the database while generating an insufficient funds errors."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_EXCHANGE_GENERIC_COIN_CONFLICTING_AGE_HASH,
    .hint = gettext_noop ("The same coin was already used with a different age hash previously."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_EXCHANGE_GENERIC_INVALID_DENOMINATION_CIPHER_FOR_OPERATION,
    .hint = gettext_noop ("The requested operation is not valid for the cipher used by the selected denomination."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_EXCHANGE_GENERIC_CIPHER_MISMATCH,
    .hint = gettext_noop ("The provided arguments for the operation use inconsistent ciphers."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_EXCHANGE_GENERIC_NEW_DENOMS_ARRAY_SIZE_EXCESSIVE,
    .hint = gettext_noop ("The number of denominations specified in the request exceeds the limit of the exchange."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_EXCHANGE_GENERIC_COIN_UNKNOWN,
    .hint = gettext_noop ("The coin is not known to the exchange (yet)."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_EXCHANGE_GENERIC_CLOCK_SKEW,
    .hint = gettext_noop ("The time at the server is too far off from the time specified in the request. Most likely the client system time is wrong."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_EXCHANGE_GENERIC_AMOUNT_EXCEEDS_DENOMINATION_VALUE,
    .hint = gettext_noop ("The specified amount for the coin is higher than the value of the denomination of the coin."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_EXCHANGE_GENERIC_GLOBAL_FEES_MISSING,
    .hint = gettext_noop ("The exchange was not properly configured with global fees."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_EXCHANGE_GENERIC_WIRE_FEES_MISSING,
    .hint = gettext_noop ("The exchange was not properly configured with wire fees."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_EXCHANGE_GENERIC_PURSE_PUB_MALFORMED,
    .hint = gettext_noop ("The purse public key was malformed."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_EXCHANGE_GENERIC_PURSE_UNKNOWN,
    .hint = gettext_noop ("The purse is unknown."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_EXCHANGE_GENERIC_PURSE_EXPIRED,
    .hint = gettext_noop ("The purse has expired."),
    .http_code = MHD_HTTP_GONE
  },

  {
    .ec = TALER_EC_EXCHANGE_GENERIC_RESERVE_UNKNOWN,
    .hint = gettext_noop ("The exchange has no information about the \"reserve_pub\" that was given."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_EXCHANGE_GENERIC_KYC_REQUIRED,
    .hint = gettext_noop ("The exchange is not allowed to proceed with the operation until the client has satisfied a KYC check."),
    .http_code = MHD_HTTP_UNAVAILABLE_FOR_LEGAL_REASONS
  },

  {
    .ec = TALER_EC_EXCHANGE_PURSE_DEPOSIT_COIN_CONFLICTING_ATTEST_VS_AGE_COMMITMENT,
    .hint = gettext_noop ("Inconsistency between provided age commitment and attest: either none or both must be provided"),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_EXCHANGE_PURSE_DEPOSIT_COIN_AGE_ATTESTATION_FAILURE,
    .hint = gettext_noop ("The provided attestation for the minimum age couldn't be verified by the exchange."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_EXCHANGE_GENERIC_PURSE_DELETED,
    .hint = gettext_noop ("The purse was deleted."),
    .http_code = MHD_HTTP_GONE
  },

  {
    .ec = TALER_EC_EXCHANGE_GENERIC_AML_OFFICER_PUB_MALFORMED,
    .hint = gettext_noop ("The public key of the AML officer in the URL was malformed."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_EXCHANGE_GENERIC_AML_OFFICER_GET_SIGNATURE_INVALID,
    .hint = gettext_noop ("The signature affirming the GET request of the AML officer is invalid."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_EXCHANGE_GENERIC_AML_OFFICER_ACCESS_DENIED,
    .hint = gettext_noop ("The specified AML officer does not have access at this time."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_EXCHANGE_GENERIC_AML_PENDING,
    .hint = gettext_noop ("The requested operation is denied pending the resolution of an anti-money laundering investigation by the exchange operator. This is a manual process, please wait and retry later."),
    .http_code = MHD_HTTP_UNAVAILABLE_FOR_LEGAL_REASONS
  },

  {
    .ec = TALER_EC_EXCHANGE_GENERIC_AML_FROZEN,
    .hint = gettext_noop ("The requested operation is denied as the account was frozen on suspicion of money laundering. Please contact the exchange operator."),
    .http_code = MHD_HTTP_UNAVAILABLE_FOR_LEGAL_REASONS
  },

  {
    .ec = TALER_EC_EXCHANGE_GENERIC_KYC_CONVERTER_FAILED,
    .hint = gettext_noop ("The exchange failed to start a KYC attribute conversion helper process. It is likely configured incorrectly."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_EXCHANGE_DEPOSITS_GET_NOT_FOUND,
    .hint = gettext_noop ("The exchange did not find information about the specified transaction in the database."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_EXCHANGE_DEPOSITS_GET_INVALID_H_WIRE,
    .hint = gettext_noop ("The wire hash of given to a \"/deposits/\" handler was malformed."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_EXCHANGE_DEPOSITS_GET_INVALID_MERCHANT_PUB,
    .hint = gettext_noop ("The merchant key of given to a \"/deposits/\" handler was malformed."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_EXCHANGE_DEPOSITS_GET_INVALID_H_CONTRACT_TERMS,
    .hint = gettext_noop ("The hash of the contract terms given to a \"/deposits/\" handler was malformed."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_EXCHANGE_DEPOSITS_GET_INVALID_COIN_PUB,
    .hint = gettext_noop ("The coin public key of given to a \"/deposits/\" handler was malformed."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_EXCHANGE_DEPOSITS_GET_INVALID_SIGNATURE_BY_EXCHANGE,
    .hint = gettext_noop ("The signature returned by the exchange in a /deposits/ request was malformed."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_EXCHANGE_DEPOSITS_GET_MERCHANT_SIGNATURE_INVALID,
    .hint = gettext_noop ("The signature of the merchant is invalid."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_EXCHANGE_DEPOSITS_POLICY_NOT_ACCEPTED,
    .hint = gettext_noop ("The provided policy data was not accepted"),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_EXCHANGE_WITHDRAW_INSUFFICIENT_FUNDS,
    .hint = gettext_noop ("The given reserve does not have sufficient funds to admit the requested withdraw operation at this time.  The response includes the current \"balance\" of the reserve as well as the transaction \"history\" that lead to this balance."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_EXCHANGE_AGE_WITHDRAW_INSUFFICIENT_FUNDS,
    .hint = gettext_noop ("The given reserve does not have sufficient funds to admit the requested age-withdraw operation at this time.  The response includes the current \"balance\" of the reserve as well as the transaction \"history\" that lead to this balance."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_EXCHANGE_WITHDRAW_AMOUNT_FEE_OVERFLOW,
    .hint = gettext_noop ("The amount to withdraw together with the fee exceeds the numeric range for Taler amounts.  This is not a client failure, as the coin value and fees come from the exchange's configuration."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_EXCHANGE_WITHDRAW_SIGNATURE_FAILED,
    .hint = gettext_noop ("The exchange failed to create the signature using the denomination key."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_EXCHANGE_WITHDRAW_RESERVE_SIGNATURE_INVALID,
    .hint = gettext_noop ("The signature of the reserve is not valid."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_EXCHANGE_RESERVE_HISTORY_ERROR_INSUFFICIENT_FUNDS,
    .hint = gettext_noop ("When computing the reserve history, we ended up with a negative overall balance, which should be impossible."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_EXCHANGE_GET_RESERVE_HISTORY_ERROR_INSUFFICIENT_BALANCE,
    .hint = gettext_noop ("The reserve did not have sufficient funds in it to pay for a full reserve history statement."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_EXCHANGE_WITHDRAW_DENOMINATION_KEY_LOST,
    .hint = gettext_noop ("Withdraw period of the coin to be withdrawn is in the past."),
    .http_code = MHD_HTTP_GONE
  },

  {
    .ec = TALER_EC_EXCHANGE_WITHDRAW_UNBLIND_FAILURE,
    .hint = gettext_noop ("The client failed to unblind the blind signature."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_EXCHANGE_WITHDRAW_NONCE_REUSE,
    .hint = gettext_noop ("The client re-used a withdraw nonce, which is not allowed."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_EXCHANGE_AGE_WITHDRAW_COMMITMENT_UNKNOWN,
    .hint = gettext_noop ("The client provided an unknown commitment for an age-withdraw request."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_EXCHANGE_AGE_WITHDRAW_AMOUNT_OVERFLOW,
    .hint = gettext_noop ("The total sum of amounts from the denominations did overflow."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_EXCHANGE_AGE_WITHDRAW_AMOUNT_INCORRECT,
    .hint = gettext_noop ("The total sum of value and fees from the denominations differs from the committed amount with fees."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_EXCHANGE_AGE_WITHDRAW_REVEAL_INVALID_HASH,
    .hint = gettext_noop ("The original commitment differs from the calculated hash"),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_EXCHANGE_AGE_WITHDRAW_MAXIMUM_AGE_TOO_LARGE,
    .hint = gettext_noop ("The maximum age in the commitment is too large for the reserve"),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_EXCHANGE_WITHDRAW_BATCH_IDEMPOTENT_PLANCHET,
    .hint = gettext_noop ("The batch withdraw included a planchet that was already withdrawn. This is not allowed."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_EXCHANGE_DEPOSIT_COIN_SIGNATURE_INVALID,
    .hint = gettext_noop ("The signature made by the coin over the deposit permission is not valid."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_EXCHANGE_DEPOSIT_CONFLICTING_CONTRACT,
    .hint = gettext_noop ("The same coin was already deposited for the same merchant and contract with other details."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_EXCHANGE_DEPOSIT_NEGATIVE_VALUE_AFTER_FEE,
    .hint = gettext_noop ("The stated value of the coin after the deposit fee is subtracted would be negative."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_EXCHANGE_DEPOSIT_REFUND_DEADLINE_AFTER_WIRE_DEADLINE,
    .hint = gettext_noop ("The stated refund deadline is after the wire deadline."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_EXCHANGE_DEPOSIT_WIRE_DEADLINE_IS_NEVER,
    .hint = gettext_noop ("The stated wire deadline is \"never\", which makes no sense."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_EXCHANGE_DEPOSIT_INVALID_WIRE_FORMAT_JSON,
    .hint = gettext_noop ("The exchange failed to canonicalize and hash the given wire format. For example, the merchant failed to provide the \"salt\" or a valid payto:// URI in the wire details.  Note that while the exchange will do some basic sanity checking on the wire details, it cannot warrant that the banking system will ultimately be able to route to the specified address, even if this check passed."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_EXCHANGE_DEPOSIT_INVALID_WIRE_FORMAT_CONTRACT_HASH_CONFLICT,
    .hint = gettext_noop ("The hash of the given wire address does not match the wire hash specified in the proposal data."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_EXCHANGE_DEPOSIT_INVALID_SIGNATURE_BY_EXCHANGE,
    .hint = gettext_noop ("The signature provided by the exchange is not valid."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_EXCHANGE_DEPOSIT_FEE_ABOVE_AMOUNT,
    .hint = gettext_noop ("The deposited amount is smaller than the deposit fee, which would result in a negative contribution."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_EXCHANGE_EXTENSIONS_INVALID_FULFILLMENT,
    .hint = gettext_noop ("The proof of policy fulfillment was invalid."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_EXCHANGE_COIN_HISTORY_BAD_SIGNATURE,
    .hint = gettext_noop ("The coin history was requested with a bad signature."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_EXCHANGE_RESERVE_HISTORY_BAD_SIGNATURE,
    .hint = gettext_noop ("The reserve history was requested with a bad signature."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_EXCHANGE_MELT_FEES_EXCEED_CONTRIBUTION,
    .hint = gettext_noop ("The exchange encountered melt fees exceeding the melted coin's contribution."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_EXCHANGE_MELT_COIN_SIGNATURE_INVALID,
    .hint = gettext_noop ("The signature made with the coin to be melted is invalid."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_EXCHANGE_MELT_COIN_EXPIRED_NO_ZOMBIE,
    .hint = gettext_noop ("The denomination of the given coin has past its expiration date and it is also not a valid zombie (that is, was not refreshed with the fresh coin being subjected to recoup)."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_EXCHANGE_MELT_INVALID_SIGNATURE_BY_EXCHANGE,
    .hint = gettext_noop ("The signature returned by the exchange in a melt request was malformed."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_EXCHANGE_REFRESHES_REVEAL_COMMITMENT_VIOLATION,
    .hint = gettext_noop ("The provided transfer keys do not match up with the original commitment.  Information about the original commitment is included in the response."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_EXCHANGE_REFRESHES_REVEAL_SIGNING_ERROR,
    .hint = gettext_noop ("Failed to produce the blinded signatures over the coins to be returned."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_EXCHANGE_REFRESHES_REVEAL_SESSION_UNKNOWN,
    .hint = gettext_noop ("The exchange is unaware of the refresh session specified in the request."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_EXCHANGE_REFRESHES_REVEAL_CNC_TRANSFER_ARRAY_SIZE_INVALID,
    .hint = gettext_noop ("The size of the cut-and-choose dimension of the private transfer keys request does not match #TALER_CNC_KAPPA - 1."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_EXCHANGE_REFRESHES_REVEAL_NEW_DENOMS_ARRAY_SIZE_MISMATCH,
    .hint = gettext_noop ("The number of envelopes given does not match the number of denomination keys given."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_EXCHANGE_REFRESHES_REVEAL_COST_CALCULATION_OVERFLOW,
    .hint = gettext_noop ("The exchange encountered a numeric overflow totaling up the cost for the refresh operation."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_EXCHANGE_REFRESHES_REVEAL_AMOUNT_INSUFFICIENT,
    .hint = gettext_noop ("The exchange's cost calculation shows that the melt amount is below the costs of the transaction."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_EXCHANGE_REFRESHES_REVEAL_LINK_SIGNATURE_INVALID,
    .hint = gettext_noop ("The signature made with the coin over the link data is invalid."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_EXCHANGE_REFRESHES_REVEAL_INVALID_RCH,
    .hint = gettext_noop ("The refresh session hash given to a /refreshes/ handler was malformed."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_EXCHANGE_REFRESHES_REVEAL_OPERATION_INVALID,
    .hint = gettext_noop ("Operation specified invalid for this endpoint."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_EXCHANGE_REFRESHES_REVEAL_AGE_RESTRICTION_NOT_SUPPORTED,
    .hint = gettext_noop ("The client provided age commitment data, but age restriction is not supported on this server."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_EXCHANGE_REFRESHES_REVEAL_AGE_RESTRICTION_COMMITMENT_INVALID,
    .hint = gettext_noop ("The client provided invalid age commitment data: missing, not an array, or  array of invalid size."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_EXCHANGE_LINK_COIN_UNKNOWN,
    .hint = gettext_noop ("The coin specified in the link request is unknown to the exchange."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_EXCHANGE_TRANSFERS_GET_WTID_MALFORMED,
    .hint = gettext_noop ("The public key of given to a /transfers/ handler was malformed."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_EXCHANGE_TRANSFERS_GET_WTID_NOT_FOUND,
    .hint = gettext_noop ("The exchange did not find information about the specified wire transfer identifier in the database."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_EXCHANGE_TRANSFERS_GET_WIRE_FEE_NOT_FOUND,
    .hint = gettext_noop ("The exchange did not find information about the wire transfer fees it charged."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_EXCHANGE_TRANSFERS_GET_WIRE_FEE_INCONSISTENT,
    .hint = gettext_noop ("The exchange found a wire fee that was above the total transfer value (and thus could not have been charged)."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_EXCHANGE_PURSES_INVALID_WAIT_TARGET,
    .hint = gettext_noop ("The wait target of the URL was not in the set of expected values."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_EXCHANGE_PURSES_GET_INVALID_SIGNATURE_BY_EXCHANGE,
    .hint = gettext_noop ("The signature on the purse status returned by the exchange was invalid."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_EXCHANGE_REFUND_COIN_NOT_FOUND,
    .hint = gettext_noop ("The exchange knows literally nothing about the coin we were asked to refund. But without a transaction history, we cannot issue a refund. This is kind-of OK, the owner should just refresh it directly without executing the refund."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_EXCHANGE_REFUND_CONFLICT_DEPOSIT_INSUFFICIENT,
    .hint = gettext_noop ("We could not process the refund request as the coin's transaction history does not permit the requested refund because then refunds would exceed the deposit amount.  The \"history\" in the response proves this."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_EXCHANGE_REFUND_DEPOSIT_NOT_FOUND,
    .hint = gettext_noop ("The exchange knows about the coin we were asked to refund, but not about the specific /deposit operation.  Hence, we cannot issue a refund (as we do not know if this merchant public key is authorized to do a refund)."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_EXCHANGE_REFUND_MERCHANT_ALREADY_PAID,
    .hint = gettext_noop ("The exchange can no longer refund the customer/coin as the money was already transferred (paid out) to the merchant. (It should be past the refund deadline.)"),
    .http_code = MHD_HTTP_GONE
  },

  {
    .ec = TALER_EC_EXCHANGE_REFUND_FEE_TOO_LOW,
    .hint = gettext_noop ("The refund fee specified for the request is lower than the refund fee charged by the exchange for the given denomination key of the refunded coin."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_EXCHANGE_REFUND_FEE_ABOVE_AMOUNT,
    .hint = gettext_noop ("The refunded amount is smaller than the refund fee, which would result in a negative refund."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_EXCHANGE_REFUND_MERCHANT_SIGNATURE_INVALID,
    .hint = gettext_noop ("The signature of the merchant is invalid."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_EXCHANGE_REFUND_MERCHANT_SIGNING_FAILED,
    .hint = gettext_noop ("Merchant backend failed to create the refund confirmation signature."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_EXCHANGE_REFUND_INVALID_SIGNATURE_BY_EXCHANGE,
    .hint = gettext_noop ("The signature returned by the exchange in a refund request was malformed."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_EXCHANGE_REFUND_INVALID_FAILURE_PROOF_BY_EXCHANGE,
    .hint = gettext_noop ("The failure proof returned by the exchange is incorrect."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_EXCHANGE_REFUND_INCONSISTENT_AMOUNT,
    .hint = gettext_noop ("Conflicting refund granted before with different amount but same refund transaction ID."),
    .http_code = MHD_HTTP_FAILED_DEPENDENCY
  },

  {
    .ec = TALER_EC_EXCHANGE_RECOUP_SIGNATURE_INVALID,
    .hint = gettext_noop ("The given coin signature is invalid for the request."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_EXCHANGE_RECOUP_WITHDRAW_NOT_FOUND,
    .hint = gettext_noop ("The exchange could not find the corresponding withdraw operation. The request is denied."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_EXCHANGE_RECOUP_COIN_BALANCE_ZERO,
    .hint = gettext_noop ("The coin's remaining balance is zero.  The request is denied."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_EXCHANGE_RECOUP_BLINDING_FAILED,
    .hint = gettext_noop ("The exchange failed to reproduce the coin's blinding."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_EXCHANGE_RECOUP_COIN_BALANCE_NEGATIVE,
    .hint = gettext_noop ("The coin's remaining balance is zero.  The request is denied."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_EXCHANGE_RECOUP_NOT_ELIGIBLE,
    .hint = gettext_noop ("The coin's denomination has not been revoked yet."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_EXCHANGE_RECOUP_REFRESH_SIGNATURE_INVALID,
    .hint = gettext_noop ("The given coin signature is invalid for the request."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_EXCHANGE_RECOUP_REFRESH_MELT_NOT_FOUND,
    .hint = gettext_noop ("The exchange could not find the corresponding melt operation. The request is denied."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_EXCHANGE_RECOUP_REFRESH_BLINDING_FAILED,
    .hint = gettext_noop ("The exchange failed to reproduce the coin's blinding."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_EXCHANGE_RECOUP_REFRESH_NOT_ELIGIBLE,
    .hint = gettext_noop ("The coin's denomination has not been revoked yet."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_EXCHANGE_KEYS_TIMETRAVEL_FORBIDDEN,
    .hint = gettext_noop ("This exchange does not allow clients to request /keys for times other than the current (exchange) time."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_EXCHANGE_WIRE_SIGNATURE_INVALID,
    .hint = gettext_noop ("A signature in the server's response was malformed."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_EXCHANGE_WIRE_NO_ACCOUNTS_CONFIGURED,
    .hint = gettext_noop ("No bank accounts are enabled for the exchange. The administrator should enable-account using the taler-exchange-offline tool."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_EXCHANGE_WIRE_INVALID_PAYTO_CONFIGURED,
    .hint = gettext_noop ("The payto:// URI stored in the exchange database for its bank account is malformed."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_EXCHANGE_WIRE_FEES_NOT_CONFIGURED,
    .hint = gettext_noop ("No wire fees are configured for an enabled wire method of the exchange. The administrator must set the wire-fee using the taler-exchange-offline tool."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_EXCHANGE_RESERVES_PURSE_CREATE_CONFLICTING_META_DATA,
    .hint = gettext_noop ("This purse was previously created with different meta data."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_EXCHANGE_RESERVES_PURSE_MERGE_CONFLICTING_META_DATA,
    .hint = gettext_noop ("This purse was previously merged with different meta data."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_EXCHANGE_RESERVES_PURSE_CREATE_INSUFFICIENT_FUNDS,
    .hint = gettext_noop ("The reserve has insufficient funds to create another purse."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_EXCHANGE_RESERVES_PURSE_FEE_TOO_LOW,
    .hint = gettext_noop ("The purse fee specified for the request is lower than the purse fee charged by the exchange at this time."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_EXCHANGE_PURSE_DELETE_ALREADY_DECIDED,
    .hint = gettext_noop ("The payment request cannot be deleted anymore, as it either already completed or timed out."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_EXCHANGE_PURSE_DELETE_SIGNATURE_INVALID,
    .hint = gettext_noop ("The signature affirming the purse deletion is invalid."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_EXCHANGE_RESERVES_AGE_RESTRICTION_REQUIRED,
    .hint = gettext_noop ("Withdrawal from the reserve requires age restriction to be set."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_EXCHANGE_DENOMINATION_HELPER_UNAVAILABLE,
    .hint = gettext_noop ("The exchange failed to talk to the process responsible for its private denomination keys or the helpers had no denominations (properly) configured."),
    .http_code = MHD_HTTP_BAD_GATEWAY
  },

  {
    .ec = TALER_EC_EXCHANGE_DENOMINATION_HELPER_BUG,
    .hint = gettext_noop ("The response from the denomination key helper process was malformed."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_EXCHANGE_DENOMINATION_HELPER_TOO_EARLY,
    .hint = gettext_noop ("The helper refuses to sign with the key, because it is too early: the validity period has not yet started."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_EXCHANGE_PURSE_DEPOSIT_EXCHANGE_SIGNATURE_INVALID,
    .hint = gettext_noop ("The signature of the exchange on the reply was invalid."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_EXCHANGE_SIGNKEY_HELPER_UNAVAILABLE,
    .hint = gettext_noop ("The exchange failed to talk to the process responsible for its private signing keys."),
    .http_code = MHD_HTTP_BAD_GATEWAY
  },

  {
    .ec = TALER_EC_EXCHANGE_SIGNKEY_HELPER_BUG,
    .hint = gettext_noop ("The response from the online signing key helper process was malformed."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_EXCHANGE_SIGNKEY_HELPER_TOO_EARLY,
    .hint = gettext_noop ("The helper refuses to sign with the key, because it is too early: the validity period has not yet started."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_EXCHANGE_RESERVES_PURSE_EXPIRATION_BEFORE_NOW,
    .hint = gettext_noop ("The purse expiration time is in the past at the time of its creation."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_EXCHANGE_RESERVES_PURSE_EXPIRATION_IS_NEVER,
    .hint = gettext_noop ("The purse expiration time is set to never, which is not allowed."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_EXCHANGE_RESERVES_PURSE_MERGE_SIGNATURE_INVALID,
    .hint = gettext_noop ("The signature affirming the merge of the purse is invalid."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_EXCHANGE_RESERVES_RESERVE_MERGE_SIGNATURE_INVALID,
    .hint = gettext_noop ("The signature by the reserve affirming the merge is invalid."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_EXCHANGE_RESERVES_OPEN_BAD_SIGNATURE,
    .hint = gettext_noop ("The signature by the reserve affirming the open operation is invalid."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_EXCHANGE_RESERVES_CLOSE_BAD_SIGNATURE,
    .hint = gettext_noop ("The signature by the reserve affirming the close operation is invalid."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_EXCHANGE_RESERVES_ATTEST_BAD_SIGNATURE,
    .hint = gettext_noop ("The signature by the reserve affirming the attestion request is invalid."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_EXCHANGE_RESERVES_CLOSE_NO_TARGET_ACCOUNT,
    .hint = gettext_noop ("The exchange does not know an origin account to which the remaining reserve balance could be wired to, and the wallet failed to provide one."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_EXCHANGE_RESERVES_OPEN_INSUFFICIENT_FUNDS,
    .hint = gettext_noop ("The reserve balance is insufficient to pay for the open operation."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_EXCHANGE_MANAGEMENT_AUDITOR_NOT_FOUND,
    .hint = gettext_noop ("The auditor that was supposed to be disabled is unknown to this exchange."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_EXCHANGE_MANAGEMENT_AUDITOR_MORE_RECENT_PRESENT,
    .hint = gettext_noop ("The exchange has a more recently signed conflicting instruction and is thus refusing the current change (replay detected)."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_EXCHANGE_MANAGEMENT_AUDITOR_ADD_SIGNATURE_INVALID,
    .hint = gettext_noop ("The signature to add or enable the auditor does not validate."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_EXCHANGE_MANAGEMENT_AUDITOR_DEL_SIGNATURE_INVALID,
    .hint = gettext_noop ("The signature to disable the auditor does not validate."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_EXCHANGE_MANAGEMENT_DENOMINATION_REVOKE_SIGNATURE_INVALID,
    .hint = gettext_noop ("The signature to revoke the denomination does not validate."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_EXCHANGE_MANAGEMENT_SIGNKEY_REVOKE_SIGNATURE_INVALID,
    .hint = gettext_noop ("The signature to revoke the online signing key does not validate."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_EXCHANGE_MANAGEMENT_WIRE_MORE_RECENT_PRESENT,
    .hint = gettext_noop ("The exchange has a more recently signed conflicting instruction and is thus refusing the current change (replay detected)."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_EXCHANGE_MANAGEMENT_KEYS_SIGNKEY_UNKNOWN,
    .hint = gettext_noop ("The signingkey specified is unknown to the exchange."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_EXCHANGE_MANAGEMENT_WIRE_DETAILS_SIGNATURE_INVALID,
    .hint = gettext_noop ("The signature to publish wire account does not validate."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_EXCHANGE_MANAGEMENT_WIRE_ADD_SIGNATURE_INVALID,
    .hint = gettext_noop ("The signature to add the wire account does not validate."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_EXCHANGE_MANAGEMENT_WIRE_DEL_SIGNATURE_INVALID,
    .hint = gettext_noop ("The signature to disable the wire account does not validate."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_EXCHANGE_MANAGEMENT_WIRE_NOT_FOUND,
    .hint = gettext_noop ("The wire account to be disabled is unknown to the exchange."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_EXCHANGE_MANAGEMENT_WIRE_FEE_SIGNATURE_INVALID,
    .hint = gettext_noop ("The signature to affirm wire fees does not validate."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_EXCHANGE_MANAGEMENT_WIRE_FEE_MISMATCH,
    .hint = gettext_noop ("The signature conflicts with a previous signature affirming different fees."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_EXCHANGE_MANAGEMENT_KEYS_DENOMKEY_ADD_SIGNATURE_INVALID,
    .hint = gettext_noop ("The signature affirming the denomination key is invalid."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_EXCHANGE_MANAGEMENT_KEYS_SIGNKEY_ADD_SIGNATURE_INVALID,
    .hint = gettext_noop ("The signature affirming the signing key is invalid."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_EXCHANGE_MANAGEMENT_GLOBAL_FEE_MISMATCH,
    .hint = gettext_noop ("The signature conflicts with a previous signature affirming different fees."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_EXCHANGE_MANAGEMENT_GLOBAL_FEE_SIGNATURE_INVALID,
    .hint = gettext_noop ("The signature affirming the fee structure is invalid."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_EXCHANGE_MANAGEMENT_DRAIN_PROFITS_SIGNATURE_INVALID,
    .hint = gettext_noop ("The signature affirming the profit drain is invalid."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_EXCHANGE_AML_DECISION_ADD_SIGNATURE_INVALID,
    .hint = gettext_noop ("The signature affirming the AML decision is invalid."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_EXCHANGE_AML_DECISION_INVALID_OFFICER,
    .hint = gettext_noop ("The AML officer specified is not allowed to make AML decisions right now."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_EXCHANGE_AML_DECISION_MORE_RECENT_PRESENT,
    .hint = gettext_noop ("There is a more recent AML decision on file. The decision was rejected as timestamps of AML decisions must be monotonically increasing."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_EXCHANGE_AML_DECISION_UNKNOWN_CHECK,
    .hint = gettext_noop ("There AML decision would impose an AML check of a type that is not provided by any KYC provider known to the exchange."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_EXCHANGE_MANAGEMENT_UPDATE_AML_OFFICER_SIGNATURE_INVALID,
    .hint = gettext_noop ("The signature affirming the change in the AML officer status is invalid."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_EXCHANGE_MANAGEMENT_AML_OFFICERS_MORE_RECENT_PRESENT,
    .hint = gettext_noop ("A more recent decision about the AML officer status is known to the exchange."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_EXCHANGE_PURSE_CREATE_CONFLICTING_META_DATA,
    .hint = gettext_noop ("The purse was previously created with different meta data."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_EXCHANGE_PURSE_CREATE_CONFLICTING_CONTRACT_STORED,
    .hint = gettext_noop ("The purse was previously created with a different contract."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_EXCHANGE_PURSE_CREATE_COIN_SIGNATURE_INVALID,
    .hint = gettext_noop ("A coin signature for a deposit into the purse is invalid."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_EXCHANGE_PURSE_CREATE_EXPIRATION_BEFORE_NOW,
    .hint = gettext_noop ("The purse expiration time is in the past."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_EXCHANGE_PURSE_CREATE_EXPIRATION_IS_NEVER,
    .hint = gettext_noop ("The purse expiration time is \"never\"."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_EXCHANGE_PURSE_CREATE_SIGNATURE_INVALID,
    .hint = gettext_noop ("The purse signature over the purse meta data is invalid."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_EXCHANGE_PURSE_ECONTRACT_SIGNATURE_INVALID,
    .hint = gettext_noop ("The signature over the encrypted contract is invalid."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_EXCHANGE_PURSE_CREATE_EXCHANGE_SIGNATURE_INVALID,
    .hint = gettext_noop ("The signature from the exchange over the confirmation is invalid."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_EXCHANGE_PURSE_DEPOSIT_CONFLICTING_META_DATA,
    .hint = gettext_noop ("The coin was previously deposited with different meta data."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_EXCHANGE_PURSE_ECONTRACT_CONFLICTING_META_DATA,
    .hint = gettext_noop ("The encrypted contract was previously uploaded with different meta data."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_EXCHANGE_CREATE_PURSE_NEGATIVE_VALUE_AFTER_FEE,
    .hint = gettext_noop ("The deposited amount is less than the purse fee."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_EXCHANGE_PURSE_MERGE_INVALID_MERGE_SIGNATURE,
    .hint = gettext_noop ("The signature using the merge key is invalid."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_EXCHANGE_PURSE_MERGE_INVALID_RESERVE_SIGNATURE,
    .hint = gettext_noop ("The signature using the reserve key is invalid."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_EXCHANGE_PURSE_NOT_FULL,
    .hint = gettext_noop ("The targeted purse is not yet full and thus cannot be merged. Retrying the request later may succeed."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_EXCHANGE_PURSE_MERGE_EXCHANGE_SIGNATURE_INVALID,
    .hint = gettext_noop ("The signature from the exchange over the confirmation is invalid."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_EXCHANGE_MERGE_PURSE_PARTNER_UNKNOWN,
    .hint = gettext_noop ("The exchange of the target account is not a partner of this exchange."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_EXCHANGE_MANAGEMENT_ADD_PARTNER_SIGNATURE_INVALID,
    .hint = gettext_noop ("The signature affirming the new partner is invalid."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_EXCHANGE_MANAGEMENT_ADD_PARTNER_DATA_CONFLICT,
    .hint = gettext_noop ("Conflicting data for the partner already exists with the exchange."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_EXCHANGE_AUDITORS_AUDITOR_SIGNATURE_INVALID,
    .hint = gettext_noop ("The auditor signature over the denomination meta data is invalid."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_EXCHANGE_AUDITORS_AUDITOR_UNKNOWN,
    .hint = gettext_noop ("The auditor that was specified is unknown to this exchange."),
    .http_code = MHD_HTTP_PRECONDITION_FAILED
  },

  {
    .ec = TALER_EC_EXCHANGE_AUDITORS_AUDITOR_INACTIVE,
    .hint = gettext_noop ("The auditor that was specified is no longer used by this exchange."),
    .http_code = MHD_HTTP_GONE
  },

  {
    .ec = TALER_EC_EXCHANGE_KYC_WALLET_SIGNATURE_INVALID,
    .hint = gettext_noop ("The signature affirming the wallet's KYC request was invalid."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_EXCHANGE_KYC_PROOF_BACKEND_INVALID_RESPONSE,
    .hint = gettext_noop ("The exchange received an unexpected malformed response from its KYC backend."),
    .http_code = MHD_HTTP_BAD_GATEWAY
  },

  {
    .ec = TALER_EC_EXCHANGE_KYC_PROOF_BACKEND_ERROR,
    .hint = gettext_noop ("The backend signaled an unexpected failure."),
    .http_code = MHD_HTTP_BAD_GATEWAY
  },

  {
    .ec = TALER_EC_EXCHANGE_KYC_PROOF_BACKEND_AUTHORIZATION_FAILED,
    .hint = gettext_noop ("The backend signaled an authorization failure."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_EXCHANGE_KYC_PROOF_REQUEST_UNKNOWN,
    .hint = gettext_noop ("The exchange is unaware of having made an the authorization request."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_EXCHANGE_KYC_CHECK_AUTHORIZATION_FAILED,
    .hint = gettext_noop ("The payto-URI hash did not match. Hence the request was denied."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_EXCHANGE_KYC_GENERIC_LOGIC_UNKNOWN,
    .hint = gettext_noop ("The request used a logic specifier that is not known to the exchange."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_EXCHANGE_KYC_GENERIC_LOGIC_GONE,
    .hint = gettext_noop ("The request requires a logic which is no longer configured at the exchange."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_EXCHANGE_KYC_GENERIC_LOGIC_BUG,
    .hint = gettext_noop ("The logic plugin had a bug in its interaction with the KYC provider."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_EXCHANGE_KYC_GENERIC_PROVIDER_ACCESS_REFUSED,
    .hint = gettext_noop ("The exchange could not process the request with its KYC provider because the provider refused access to the service. This indicates some configuration issue at the Taler exchange operator."),
    .http_code = MHD_HTTP_NETWORK_AUTHENTICATION_REQUIRED
  },

  {
    .ec = TALER_EC_EXCHANGE_KYC_GENERIC_PROVIDER_TIMEOUT,
    .hint = gettext_noop ("There was a timeout in the interaction between the exchange and the KYC provider. The most likely cause is some networking problem. Trying again later might succeed."),
    .http_code = MHD_HTTP_GATEWAY_TIMEOUT
  },

  {
    .ec = TALER_EC_EXCHANGE_KYC_GENERIC_PROVIDER_UNEXPECTED_REPLY,
    .hint = gettext_noop ("The KYC provider responded with a status that was completely unexpected by the KYC logic of the exchange."),
    .http_code = MHD_HTTP_BAD_GATEWAY
  },

  {
    .ec = TALER_EC_EXCHANGE_KYC_GENERIC_PROVIDER_RATE_LIMIT_EXCEEDED,
    .hint = gettext_noop ("The rate limit of the exchange at the KYC provider has been exceeded. Trying much later might work."),
    .http_code = MHD_HTTP_SERVICE_UNAVAILABLE
  },

  {
    .ec = TALER_EC_EXCHANGE_KYC_WEBHOOK_UNAUTHORIZED,
    .hint = gettext_noop ("The request to the webhook lacked proper authorization or authentication data."),
    .http_code = MHD_HTTP_UNAUTHORIZED
  },

  {
    .ec = TALER_EC_EXCHANGE_CONTRACTS_UNKNOWN,
    .hint = gettext_noop ("The exchange does not know a contract under the given contract public key."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_EXCHANGE_CONTRACTS_INVALID_CONTRACT_PUB,
    .hint = gettext_noop ("The URL does not encode a valid exchange public key in its path."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_EXCHANGE_CONTRACTS_DECRYPTION_FAILED,
    .hint = gettext_noop ("The returned encrypted contract did not decrypt."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_EXCHANGE_CONTRACTS_SIGNATURE_INVALID,
    .hint = gettext_noop ("The signature on the encrypted contract did not validate."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_EXCHANGE_CONTRACTS_DECODING_FAILED,
    .hint = gettext_noop ("The decrypted contract was malformed."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_EXCHANGE_PURSE_DEPOSIT_COIN_SIGNATURE_INVALID,
    .hint = gettext_noop ("A coin signature for a deposit into the purse is invalid."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_EXCHANGE_PURSE_DEPOSIT_DECIDED_ALREADY,
    .hint = gettext_noop ("It is too late to deposit coins into the purse."),
    .http_code = MHD_HTTP_GONE
  },

  {
    .ec = TALER_EC_EXCHANGE_TOTP_KEY_INVALID,
    .hint = gettext_noop ("TOTP key is not valid."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_MERCHANT_GENERIC_INSTANCE_UNKNOWN,
    .hint = gettext_noop ("The backend could not find the merchant instance specified in the request."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_MERCHANT_GENERIC_HOLE_IN_WIRE_FEE_STRUCTURE,
    .hint = gettext_noop ("The start and end-times in the wire fee structure leave a hole. This is not allowed."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_MERCHANT_GENERIC_EXCHANGE_WIRE_REQUEST_FAILED,
    .hint = gettext_noop ("The merchant was unable to obtain a valid answer to /wire from the exchange."),
    .http_code = MHD_HTTP_BAD_GATEWAY
  },

  {
    .ec = TALER_EC_MERCHANT_GENERIC_ORDER_UNKNOWN,
    .hint = gettext_noop ("The proposal is not known to the backend."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_MERCHANT_GENERIC_PRODUCT_UNKNOWN,
    .hint = gettext_noop ("The order provided to the backend could not be completed, because a product to be completed via inventory data is not actually in our inventory."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_MERCHANT_GENERIC_REWARD_ID_UNKNOWN,
    .hint = gettext_noop ("The reward ID is unknown.  This could happen if the reward has expired."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_MERCHANT_GENERIC_DB_CONTRACT_CONTENT_INVALID,
    .hint = gettext_noop ("The contract obtained from the merchant backend was malformed."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_MERCHANT_GENERIC_CONTRACT_HASH_DOES_NOT_MATCH_ORDER,
    .hint = gettext_noop ("The order we found does not match the provided contract hash."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_MERCHANT_GENERIC_EXCHANGE_KEYS_FAILURE,
    .hint = gettext_noop ("The exchange failed to provide a valid response to the merchant's /keys request."),
    .http_code = MHD_HTTP_BAD_GATEWAY
  },

  {
    .ec = TALER_EC_MERCHANT_GENERIC_EXCHANGE_TIMEOUT,
    .hint = gettext_noop ("The exchange failed to respond to the merchant on time."),
    .http_code = MHD_HTTP_GATEWAY_TIMEOUT
  },

  {
    .ec = TALER_EC_MERCHANT_GENERIC_EXCHANGE_CONNECT_FAILURE,
    .hint = gettext_noop ("The merchant failed to talk to the exchange."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_MERCHANT_GENERIC_EXCHANGE_REPLY_MALFORMED,
    .hint = gettext_noop ("The exchange returned a maformed response."),
    .http_code = MHD_HTTP_BAD_GATEWAY
  },

  {
    .ec = TALER_EC_MERCHANT_GENERIC_EXCHANGE_UNEXPECTED_STATUS,
    .hint = gettext_noop ("The exchange returned an unexpected response status."),
    .http_code = MHD_HTTP_BAD_GATEWAY
  },

  {
    .ec = TALER_EC_MERCHANT_GENERIC_UNAUTHORIZED,
    .hint = gettext_noop ("The merchant refused the request due to lack of authorization."),
    .http_code = MHD_HTTP_UNAUTHORIZED
  },

  {
    .ec = TALER_EC_MERCHANT_GENERIC_INSTANCE_DELETED,
    .hint = gettext_noop ("The merchant instance specified in the request was deleted."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_MERCHANT_GENERIC_TRANSFER_UNKNOWN,
    .hint = gettext_noop ("The backend could not find the inbound wire transfer specified in the request."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_MERCHANT_GENERIC_TEMPLATE_UNKNOWN,
    .hint = gettext_noop ("The backend could not find the template(id) because it is not exist."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_MERCHANT_GENERIC_WEBHOOK_UNKNOWN,
    .hint = gettext_noop ("The backend could not find the webhook(id) because it is not exist."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_MERCHANT_GENERIC_PENDING_WEBHOOK_UNKNOWN,
    .hint = gettext_noop ("The backend could not find the webhook(serial) because it is not exist."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_MERCHANT_GENERIC_OTP_DEVICE_UNKNOWN,
    .hint = gettext_noop ("The backend could not find the OTP device(id) because it is not exist."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_MERCHANT_GENERIC_ACCOUNT_UNKNOWN,
    .hint = gettext_noop ("The account is not known to the backend."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_MERCHANT_GENERIC_H_WIRE_MALFORMED,
    .hint = gettext_noop ("The wire hash was malformed."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_MERCHANT_GENERIC_CURRENCY_MISMATCH,
    .hint = gettext_noop ("The currency specified in the operation does not work with the current state of the given resource."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_MERCHANT_GET_ORDERS_EXCHANGE_TRACKING_FAILURE,
    .hint = gettext_noop ("The exchange failed to provide a valid answer to the tracking request, thus those details are not in the response."),
    .http_code = MHD_HTTP_OK
  },

  {
    .ec = TALER_EC_MERCHANT_GET_ORDERS_ID_EXCHANGE_REQUEST_FAILURE,
    .hint = gettext_noop ("The merchant backend failed to construct the request for tracking to the exchange, thus tracking details are not in the response."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_MERCHANT_GET_ORDERS_ID_EXCHANGE_LOOKUP_START_FAILURE,
    .hint = gettext_noop ("The merchant backend failed trying to contact the exchange for tracking details, thus those details are not in the response."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_MERCHANT_GET_ORDERS_ID_INVALID_TOKEN,
    .hint = gettext_noop ("The claim token used to authenticate the client is invalid for this order."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_MERCHANT_GET_ORDERS_ID_INVALID_CONTRACT_HASH,
    .hint = gettext_noop ("The contract terms hash used to authenticate the client is invalid for this order."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_MERCHANT_POST_ORDERS_ID_PAY_INSUFFICIENT_FUNDS,
    .hint = gettext_noop ("The exchange responded saying that funds were insufficient (for example, due to double-spending)."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_MERCHANT_POST_ORDERS_ID_PAY_DENOMINATION_KEY_NOT_FOUND,
    .hint = gettext_noop ("The denomination key used for payment is not listed among the denomination keys of the exchange."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_MERCHANT_POST_ORDERS_ID_PAY_DENOMINATION_KEY_AUDITOR_FAILURE,
    .hint = gettext_noop ("The denomination key used for payment is not audited by an auditor approved by the merchant."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_MERCHANT_POST_ORDERS_ID_PAY_AMOUNT_OVERFLOW,
    .hint = gettext_noop ("There was an integer overflow totaling up the amounts or deposit fees in the payment."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_MERCHANT_POST_ORDERS_ID_PAY_FEES_EXCEED_PAYMENT,
    .hint = gettext_noop ("The deposit fees exceed the total value of the payment."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_MERCHANT_POST_ORDERS_ID_PAY_INSUFFICIENT_DUE_TO_FEES,
    .hint = gettext_noop ("After considering deposit and wire fees, the payment is insufficient to satisfy the required amount for the contract.  The client should revisit the logic used to calculate fees it must cover."),
    .http_code = MHD_HTTP_NOT_ACCEPTABLE
  },

  {
    .ec = TALER_EC_MERCHANT_POST_ORDERS_ID_PAY_PAYMENT_INSUFFICIENT,
    .hint = gettext_noop ("Even if we do not consider deposit and wire fees, the payment is insufficient to satisfy the required amount for the contract."),
    .http_code = MHD_HTTP_NOT_ACCEPTABLE
  },

  {
    .ec = TALER_EC_MERCHANT_POST_ORDERS_ID_PAY_COIN_SIGNATURE_INVALID,
    .hint = gettext_noop ("The signature over the contract of one of the coins was invalid."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_MERCHANT_POST_ORDERS_ID_PAY_EXCHANGE_LOOKUP_FAILED,
    .hint = gettext_noop ("When we tried to find information about the exchange to issue the deposit, we failed.  This usually only happens if the merchant backend is somehow unable to get its own HTTP client logic to work."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_MERCHANT_POST_ORDERS_ID_PAY_REFUND_DEADLINE_PAST_WIRE_TRANSFER_DEADLINE,
    .hint = gettext_noop ("The refund deadline in the contract is after the transfer deadline."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_MERCHANT_POST_ORDERS_ID_PAY_ALREADY_PAID,
    .hint = gettext_noop ("The order was already paid (maybe by another wallet)."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_MERCHANT_POST_ORDERS_ID_PAY_OFFER_EXPIRED,
    .hint = gettext_noop ("The payment is too late, the offer has expired."),
    .http_code = MHD_HTTP_GONE
  },

  {
    .ec = TALER_EC_MERCHANT_POST_ORDERS_ID_PAY_MERCHANT_FIELD_MISSING,
    .hint = gettext_noop ("The \"merchant\" field is missing in the proposal data. This is an internal error as the proposal is from the merchant's own database at this point."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_MERCHANT_POST_ORDERS_ID_PAY_WIRE_HASH_UNKNOWN,
    .hint = gettext_noop ("Failed to locate merchant's account information matching the wire hash given in the proposal."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_MERCHANT_POST_ORDERS_ID_PAY_DENOMINATION_DEPOSIT_EXPIRED,
    .hint = gettext_noop ("The deposit time for the denomination has expired."),
    .http_code = MHD_HTTP_GONE
  },

  {
    .ec = TALER_EC_MERCHANT_POST_ORDERS_ID_PAY_EXCHANGE_WIRE_FEE_ADDITION_FAILED,
    .hint = gettext_noop ("The exchange of the deposited coin charges a wire fee that could not be added to the total (total amount too high)."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_MERCHANT_POST_ORDERS_ID_PAY_REFUNDED,
    .hint = gettext_noop ("The contract was not fully paid because of refunds. Note that clients MAY treat this as paid if, for example, contracts must be executed despite of refunds."),
    .http_code = MHD_HTTP_PAYMENT_REQUIRED
  },

  {
    .ec = TALER_EC_MERCHANT_POST_ORDERS_ID_PAY_REFUNDS_EXCEED_PAYMENTS,
    .hint = gettext_noop ("According to our database, we have refunded more than we were paid (which should not be possible)."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_DEAD_QQQ_PAY_MERCHANT_POST_ORDERS_ID_ABORT_REFUND_REFUSED_PAYMENT_COMPLETE,
    .hint = gettext_noop ("Legacy stuff. Remove me with protocol v1."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_MERCHANT_POST_ORDERS_ID_PAY_EXCHANGE_FAILED,
    .hint = gettext_noop ("The payment failed at the exchange."),
    .http_code = MHD_HTTP_BAD_GATEWAY
  },

  {
    .ec = TALER_EC_MERCHANT_POST_ORDERS_ID_PAY_AGE_COMMITMENT_MISSING,
    .hint = gettext_noop ("The payment required a minimum age but one of the coins (of a denomination with support for age restriction) did not provide any age_commitment."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_MERCHANT_POST_ORDERS_ID_PAY_AGE_COMMITMENT_SIZE_MISMATCH,
    .hint = gettext_noop ("The payment required a minimum age but one of the coins provided an age_commitment that contained a wrong number of public keys compared to the number of age groups defined in the denomination of the coin."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_MERCHANT_POST_ORDERS_ID_PAY_AGE_VERIFICATION_FAILED,
    .hint = gettext_noop ("The payment required a minimum age but one of the coins provided a minimum_age_sig that couldn't be verified with the given age_commitment for that particular minimum age."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_MERCHANT_POST_ORDERS_ID_PAY_AGE_COMMITMENT_HASH_MISSING,
    .hint = gettext_noop ("The payment required no minimum age but one of the coins (of a denomination with support for age restriction) did not provide the required h_age_commitment."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_MERCHANT_POST_ORDERS_ID_PAY_WIRE_METHOD_UNSUPPORTED,
    .hint = gettext_noop ("The exchange does not support the selected bank account of the merchant. Likely the merchant had stale data on the bank accounts of the exchange and thus selected an inappropriate exchange when making the offer."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_MERCHANT_POST_ORDERS_ID_PAID_CONTRACT_HASH_MISMATCH,
    .hint = gettext_noop ("The contract hash does not match the given order ID."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_MERCHANT_POST_ORDERS_ID_PAID_COIN_SIGNATURE_INVALID,
    .hint = gettext_noop ("The signature of the merchant is not valid for the given contract hash."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_MERCHANT_POST_ORDERS_ID_ABORT_EXCHANGE_REFUND_FAILED,
    .hint = gettext_noop ("The merchant failed to send the exchange the refund request."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_MERCHANT_POST_ORDERS_ID_ABORT_EXCHANGE_LOOKUP_FAILED,
    .hint = gettext_noop ("The merchant failed to find the exchange to process the lookup."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_MERCHANT_POST_ORDERS_ID_ABORT_CONTRACT_NOT_FOUND,
    .hint = gettext_noop ("The merchant could not find the contract."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_MERCHANT_POST_ORDERS_ID_ABORT_REFUND_REFUSED_PAYMENT_COMPLETE,
    .hint = gettext_noop ("The payment was already completed and thus cannot be aborted anymore."),
    .http_code = MHD_HTTP_PRECONDITION_FAILED
  },

  {
    .ec = TALER_EC_MERCHANT_POST_ORDERS_ID_ABORT_CONTRACT_HASH_MISSMATCH,
    .hint = gettext_noop ("The hash provided by the wallet does not match the order."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_MERCHANT_POST_ORDERS_ID_ABORT_COINS_ARRAY_EMPTY,
    .hint = gettext_noop ("The array of coins cannot be empty."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_MERCHANT_EXCHANGE_TRANSFERS_AWAITING_KEYS,
    .hint = gettext_noop ("We are waiting for the exchange to provide us with key material before checking the wire transfer."),
    .http_code = MHD_HTTP_ACCEPTED
  },

  {
    .ec = TALER_EC_MERCHANT_EXCHANGE_TRANSFERS_AWAITING_LIST,
    .hint = gettext_noop ("We are waiting for the exchange to provide us with the list of aggregated transactions."),
    .http_code = MHD_HTTP_ACCEPTED
  },

  {
    .ec = TALER_EC_MERCHANT_EXCHANGE_TRANSFERS_FATAL_NO_EXCHANGE,
    .hint = gettext_noop ("The endpoint indicated in the wire transfer does not belong to a GNU Taler exchange."),
    .http_code = MHD_HTTP_OK
  },

  {
    .ec = TALER_EC_MERCHANT_EXCHANGE_TRANSFERS_FATAL_NOT_FOUND,
    .hint = gettext_noop ("The exchange indicated in the wire transfer claims to know nothing about the wire transfer."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_MERCHANT_EXCHANGE_TRANSFERS_RATE_LIMITED,
    .hint = gettext_noop ("The interaction with the exchange is delayed due to rate limiting."),
    .http_code = MHD_HTTP_ACCEPTED
  },

  {
    .ec = TALER_EC_MERCHANT_EXCHANGE_TRANSFERS_TRANSIENT_FAILURE,
    .hint = gettext_noop ("We experienced a transient failure in our interaction with the exchange."),
    .http_code = MHD_HTTP_ACCEPTED
  },

  {
    .ec = TALER_EC_MERCHANT_EXCHANGE_TRANSFERS_HARD_FAILURE,
    .hint = gettext_noop ("The response from the exchange was unacceptable and should be reviewed with an auditor."),
    .http_code = MHD_HTTP_OK
  },

  {
    .ec = TALER_EC_MERCHANT_POST_ORDERS_ID_CLAIM_NOT_FOUND,
    .hint = gettext_noop ("We could not claim the order because the backend is unaware of it."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_MERCHANT_POST_ORDERS_ID_CLAIM_ALREADY_CLAIMED,
    .hint = gettext_noop ("We could not claim the order because someone else claimed it first."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_MERCHANT_POST_ORDERS_ID_CLAIM_CLIENT_INTERNAL_FAILURE,
    .hint = gettext_noop ("The client-side experienced an internal failure."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_MERCHANT_POST_ORDERS_ID_REFUND_SIGNATURE_FAILED,
    .hint = gettext_noop ("The backend failed to sign the refund request."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_MERCHANT_REWARD_PICKUP_UNBLIND_FAILURE,
    .hint = gettext_noop ("The client failed to unblind the signature returned by the merchant."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_MERCHANT_REWARD_PICKUP_EXCHANGE_ERROR,
    .hint = gettext_noop ("The exchange returned a failure code for the withdraw operation."),
    .http_code = MHD_HTTP_BAD_GATEWAY
  },

  {
    .ec = TALER_EC_MERCHANT_REWARD_PICKUP_SUMMATION_FAILED,
    .hint = gettext_noop ("The merchant failed to add up the amounts to compute the pick up value."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_MERCHANT_REWARD_PICKUP_HAS_EXPIRED,
    .hint = gettext_noop ("The reward expired."),
    .http_code = MHD_HTTP_GONE
  },

  {
    .ec = TALER_EC_MERCHANT_REWARD_PICKUP_AMOUNT_EXCEEDS_REWARD_REMAINING,
    .hint = gettext_noop ("The requested withdraw amount exceeds the amount remaining to be picked up."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_MERCHANT_REWARD_PICKUP_DENOMINATION_UNKNOWN,
    .hint = gettext_noop ("The merchant did not find the specified denomination key in the exchange's key set."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_MERCHANT_PRIVATE_POST_ORDERS_INSTANCE_CONFIGURATION_LACKS_WIRE,
    .hint = gettext_noop ("The merchant instance has no active bank accounts configured. However, at least one bank account must be available to create new orders."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_MERCHANT_PRIVATE_POST_ORDERS_NO_LOCALTIME,
    .hint = gettext_noop ("The proposal had no timestamp and the merchant backend failed to obtain the current local time."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_MERCHANT_PRIVATE_POST_ORDERS_PROPOSAL_PARSE_ERROR,
    .hint = gettext_noop ("The order provided to the backend could not be parsed; likely some required fields were missing or ill-formed."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_MERCHANT_PRIVATE_POST_ORDERS_ALREADY_EXISTS,
    .hint = gettext_noop ("A conflicting order (sharing the same order identifier) already exists at this merchant backend instance."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_MERCHANT_PRIVATE_POST_ORDERS_REFUND_AFTER_WIRE_DEADLINE,
    .hint = gettext_noop ("The order creation request is invalid because the given wire deadline is before the refund deadline."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_MERCHANT_PRIVATE_POST_ORDERS_DELIVERY_DATE_IN_PAST,
    .hint = gettext_noop ("The order creation request is invalid because the delivery date given is in the past."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_MERCHANT_PRIVATE_POST_ORDERS_WIRE_DEADLINE_IS_NEVER,
    .hint = gettext_noop ("The order creation request is invalid because a wire deadline of \"never\" is not allowed."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_MERCHANT_PRIVATE_POST_ORDERS_PAY_DEADLINE_IN_PAST,
    .hint = gettext_noop ("The order ceration request is invalid because the given payment deadline is in the past."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_MERCHANT_PRIVATE_POST_ORDERS_REFUND_DEADLINE_IN_PAST,
    .hint = gettext_noop ("The order creation request is invalid because the given refund deadline is in the past."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_MERCHANT_PRIVATE_POST_ORDERS_NO_EXCHANGES_FOR_WIRE_METHOD,
    .hint = gettext_noop ("The backend does not trust any exchange that would allow funds to be wired to any bank account of this instance using the wire method specified with the order. Note that right now, we do not support the use of exchange bank accounts with mandatory currency conversion."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_MERCHANT_PRIVATE_PATCH_ORDERS_ID_FORGET_PATH_SYNTAX_INCORRECT,
    .hint = gettext_noop ("One of the paths to forget is malformed."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_MERCHANT_PRIVATE_PATCH_ORDERS_ID_FORGET_PATH_NOT_FORGETTABLE,
    .hint = gettext_noop ("One of the paths to forget was not marked as forgettable."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_MERCHANT_PRIVATE_DELETE_ORDERS_AWAITING_PAYMENT,
    .hint = gettext_noop ("The order provided to the backend could not be deleted, our offer is still valid and awaiting payment. Deletion may work later after the offer has expired if it remains unpaid."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_MERCHANT_PRIVATE_DELETE_ORDERS_ALREADY_PAID,
    .hint = gettext_noop ("The order provided to the backend could not be deleted as the order was already paid."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_MERCHANT_PRIVATE_POST_ORDERS_ID_REFUND_INCONSISTENT_AMOUNT,
    .hint = gettext_noop ("The amount to be refunded is inconsistent: either is lower than the previous amount being awarded, or it exceeds the original price paid by the customer."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_MERCHANT_PRIVATE_POST_ORDERS_ID_REFUND_ORDER_UNPAID,
    .hint = gettext_noop ("Only paid orders can be refunded, and the frontend specified an unpaid order to issue a refund for."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_MERCHANT_PRIVATE_POST_ORDERS_ID_REFUND_NOT_ALLOWED_BY_CONTRACT,
    .hint = gettext_noop ("The refund delay was set to 0 and thus no refunds are ever allowed for this order."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_MERCHANT_PRIVATE_POST_TRANSFERS_EXCHANGE_UNKNOWN,
    .hint = gettext_noop ("The exchange says it does not know this transfer."),
    .http_code = MHD_HTTP_BAD_GATEWAY
  },

  {
    .ec = TALER_EC_MERCHANT_PRIVATE_POST_TRANSFERS_REQUEST_ERROR,
    .hint = gettext_noop ("We internally failed to execute the /track/transfer request."),
    .http_code = MHD_HTTP_BAD_GATEWAY
  },

  {
    .ec = TALER_EC_MERCHANT_PRIVATE_POST_TRANSFERS_CONFLICTING_TRANSFERS,
    .hint = gettext_noop ("The amount transferred differs between what was submitted and what the exchange claimed."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_MERCHANT_PRIVATE_POST_TRANSFERS_CONFLICTING_REPORTS,
    .hint = gettext_noop ("The exchange gave conflicting information about a coin which has been wire transferred."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_MERCHANT_PRIVATE_POST_TRANSFERS_BAD_WIRE_FEE,
    .hint = gettext_noop ("The exchange charged a different wire fee than what it originally advertised, and it is higher."),
    .http_code = MHD_HTTP_BAD_GATEWAY
  },

  {
    .ec = TALER_EC_MERCHANT_PRIVATE_POST_TRANSFERS_ACCOUNT_NOT_FOUND,
    .hint = gettext_noop ("We did not find the account that the transfer was made to."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_MERCHANT_PRIVATE_DELETE_TRANSFERS_ALREADY_CONFIRMED,
    .hint = gettext_noop ("The backend could not delete the transfer as the echange already replied to our inquiry about it and we have integrated the result."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_MERCHANT_PRIVATE_POST_TRANSFERS_CONFLICTING_SUBMISSION,
    .hint = gettext_noop ("The backend was previously informed about a wire transfer with the same ID but a different amount. Multiple wire transfers with the same ID are not allowed. If the new amount is correct, the old transfer should first be deleted."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_MERCHANT_EXCHANGE_TRANSFERS_CONFLICTING_TRANSFERS,
    .hint = gettext_noop ("The amount transferred differs between what was submitted and what the exchange claimed."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_MERCHANT_PRIVATE_POST_INSTANCES_ALREADY_EXISTS,
    .hint = gettext_noop ("The merchant backend cannot create an instance under the given identifier as one already exists. Use PATCH to modify the existing entry."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_MERCHANT_PRIVATE_POST_INSTANCES_BAD_AUTH,
    .hint = gettext_noop ("The merchant backend cannot create an instance because the authentication configuration field is malformed."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_MERCHANT_PRIVATE_POST_INSTANCE_AUTH_BAD_AUTH,
    .hint = gettext_noop ("The merchant backend cannot update an instance's authentication settings because the provided authentication settings are malformed."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_MERCHANT_PRIVATE_POST_INSTANCES_PURGE_REQUIRED,
    .hint = gettext_noop ("The merchant backend cannot create an instance under the given identifier, the previous one was deleted but must be purged first."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_MERCHANT_PRIVATE_PATCH_INSTANCES_PURGE_REQUIRED,
    .hint = gettext_noop ("The merchant backend cannot update an instance under the given identifier, the previous one was deleted but must be purged first."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_MERCHANT_PRIVATE_ACCOUNT_DELETE_UNKNOWN_ACCOUNT,
    .hint = gettext_noop ("The bank account referenced in the requested operation was not found."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_MERCHANT_PRIVATE_ACCOUNT_EXISTS,
    .hint = gettext_noop ("The bank account specified in the request already exists at the merchant."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_MERCHANT_PRIVATE_POST_PRODUCTS_CONFLICT_PRODUCT_EXISTS,
    .hint = gettext_noop ("The product ID exists."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_MERCHANT_PRIVATE_PATCH_PRODUCTS_TOTAL_LOST_REDUCED,
    .hint = gettext_noop ("The update would have reduced the total amount of product lost, which is not allowed."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_MERCHANT_PRIVATE_PATCH_PRODUCTS_TOTAL_LOST_EXCEEDS_STOCKS,
    .hint = gettext_noop ("The update would have mean that more stocks were lost than what remains from total inventory after sales, which is not allowed."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_MERCHANT_PRIVATE_PATCH_PRODUCTS_TOTAL_STOCKED_REDUCED,
    .hint = gettext_noop ("The update would have reduced the total amount of product in stock, which is not allowed."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_MERCHANT_PRIVATE_PATCH_PRODUCTS_TOTAL_SOLD_REDUCED,
    .hint = gettext_noop ("The update would have reduced the total amount of product sold, which is not allowed."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_MERCHANT_PRIVATE_POST_PRODUCTS_LOCK_INSUFFICIENT_STOCKS,
    .hint = gettext_noop ("The lock request is for more products than we have left (unlocked) in stock."),
    .http_code = MHD_HTTP_GONE
  },

  {
    .ec = TALER_EC_MERCHANT_PRIVATE_DELETE_PRODUCTS_CONFLICTING_LOCK,
    .hint = gettext_noop ("The deletion request is for a product that is locked."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_MERCHANT_PRIVATE_POST_RESERVES_UNSUPPORTED_WIRE_METHOD,
    .hint = gettext_noop ("The requested wire method is not supported by the exchange."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_MERCHANT_PRIVATE_POST_RESERVES_REWARDS_NOT_ALLOWED,
    .hint = gettext_noop ("The requested exchange does not allow rewards."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_MERCHANT_PRIVATE_DELETE_RESERVES_NO_SUCH_RESERVE,
    .hint = gettext_noop ("The reserve could not be deleted because it is unknown."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_MERCHANT_PRIVATE_POST_REWARD_AUTHORIZE_RESERVE_EXPIRED,
    .hint = gettext_noop ("The reserve that was used to fund the rewards has expired."),
    .http_code = MHD_HTTP_GONE
  },

  {
    .ec = TALER_EC_MERCHANT_PRIVATE_POST_REWARD_AUTHORIZE_RESERVE_UNKNOWN,
    .hint = gettext_noop ("The reserve that was used to fund the rewards was not found in the DB."),
    .http_code = MHD_HTTP_SERVICE_UNAVAILABLE
  },

  {
    .ec = TALER_EC_MERCHANT_PRIVATE_POST_REWARD_AUTHORIZE_INSUFFICIENT_FUNDS,
    .hint = gettext_noop ("The backend knows the instance that was supposed to support the reward, and it was configured for rewardping. However, the funds remaining are insufficient to cover the reward, and the merchant should top up the reserve."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_MERCHANT_PRIVATE_POST_REWARD_AUTHORIZE_RESERVE_NOT_FOUND,
    .hint = gettext_noop ("The backend failed to find a reserve needed to authorize the reward."),
    .http_code = MHD_HTTP_SERVICE_UNAVAILABLE
  },

  {
    .ec = TALER_EC_MERCHANT_PRIVATE_GET_ORDERS_ID_AMOUNT_ARITHMETIC_FAILURE,
    .hint = gettext_noop ("The merchant backend encountered a failure in computing the deposit total."),
    .http_code = MHD_HTTP_OK
  },

  {
    .ec = TALER_EC_MERCHANT_PRIVATE_POST_TEMPLATES_CONFLICT_TEMPLATE_EXISTS,
    .hint = gettext_noop ("The template ID already exists."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_MERCHANT_PRIVATE_POST_OTP_DEVICES_CONFLICT_OTP_DEVICE_EXISTS,
    .hint = gettext_noop ("The OTP device ID already exists."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_MERCHANT_POST_USING_TEMPLATES_AMOUNT_CONFLICT_TEMPLATES_CONTRACT_AMOUNT,
    .hint = gettext_noop ("Amount given in the using template and in the template contract. There is a conflict."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_MERCHANT_POST_USING_TEMPLATES_SUMMARY_CONFLICT_TEMPLATES_CONTRACT_SUBJECT,
    .hint = gettext_noop ("Subject given in the using template and in the template contract. There is a conflict."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_MERCHANT_POST_USING_TEMPLATES_NO_AMOUNT,
    .hint = gettext_noop ("Amount not given in the using template and in the template contract. There is a conflict."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_MERCHANT_POST_USING_TEMPLATES_NO_SUMMARY,
    .hint = gettext_noop ("Subject not given in the using template and in the template contract. There is a conflict."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_MERCHANT_PRIVATE_POST_WEBHOOKS_CONFLICT_WEBHOOK_EXISTS,
    .hint = gettext_noop ("The webhook ID elready exists."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_MERCHANT_PRIVATE_POST_PENDING_WEBHOOKS_CONFLICT_PENDING_WEBHOOK_EXISTS,
    .hint = gettext_noop ("The webhook serial elready exists."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_AUDITOR_DEPOSIT_CONFIRMATION_SIGNATURE_INVALID,
    .hint = gettext_noop ("The signature from the exchange on the deposit confirmation is invalid."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_AUDITOR_EXCHANGE_SIGNING_KEY_REVOKED,
    .hint = gettext_noop ("The exchange key used for the signature on the deposit confirmation was revoked."),
    .http_code = MHD_HTTP_GONE
  },

  {
    .ec = TALER_EC_BANK_SAME_ACCOUNT,
    .hint = gettext_noop ("Wire transfer attempted with credit and debit party being the same bank account."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_BANK_UNALLOWED_DEBIT,
    .hint = gettext_noop ("Wire transfer impossible, due to financial limitation of the party that attempted the payment."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_BANK_NEGATIVE_NUMBER_AMOUNT,
    .hint = gettext_noop ("Negative numbers are not allowed (as value and/or fraction) to instantiate an amount object."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_BANK_NUMBER_TOO_BIG,
    .hint = gettext_noop ("A too big number was used (as value and/or fraction) to instantiate an amount object."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_BANK_UNKNOWN_ACCOUNT,
    .hint = gettext_noop ("The bank account referenced in the requested operation was not found."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_BANK_TRANSACTION_NOT_FOUND,
    .hint = gettext_noop ("The transaction referenced in the requested operation (typically a reject operation), was not found."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_BANK_BAD_FORMAT_AMOUNT,
    .hint = gettext_noop ("Bank received a malformed amount string."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_BANK_REJECT_NO_RIGHTS,
    .hint = gettext_noop ("The client does not own the account credited by the transaction which is to be rejected, so it has no rights do reject it."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_BANK_UNMANAGED_EXCEPTION,
    .hint = gettext_noop ("This error code is returned when no known exception types captured the exception."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_BANK_SOFT_EXCEPTION,
    .hint = gettext_noop ("This error code is used for all those exceptions that do not really need a specific error code to return to the client. Used for example when a client is trying to register with a unavailable username."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_BANK_TRANSFER_REQUEST_UID_REUSED,
    .hint = gettext_noop ("The request UID for a request to transfer funds has already been used, but with different details for the transfer."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_BANK_WITHDRAWAL_OPERATION_RESERVE_SELECTION_CONFLICT,
    .hint = gettext_noop ("The withdrawal operation already has a reserve selected.  The current request conflicts with the existing selection."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_BANK_DUPLICATE_RESERVE_PUB_SUBJECT,
    .hint = gettext_noop ("The wire transfer subject duplicates an existing reserve public key. But wire transfer subjects must be unique."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_BANK_ANCIENT_TRANSACTION_GONE,
    .hint = gettext_noop ("The client requested a transaction that is so far in the past, that it has been forgotten by the bank."),
    .http_code = MHD_HTTP_GONE
  },

  {
    .ec = TALER_EC_BANK_ABORT_CONFIRM_CONFLICT,
    .hint = gettext_noop ("The client attempted to abort a transaction that was already confirmed."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_BANK_CONFIRM_ABORT_CONFLICT,
    .hint = gettext_noop ("The client attempted to confirm a transaction that was already aborted."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_BANK_REGISTER_CONFLICT,
    .hint = gettext_noop ("The client attempted to register an account with the same name."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_BANK_POST_WITHDRAWAL_OPERATION_REQUIRED,
    .hint = gettext_noop ("The client attempted to confirm a withdrawal operation before the wallet posted the required details."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_BANK_RESERVED_USERNAME_CONFLICT,
    .hint = gettext_noop ("The client tried to register a new account under a reserved username (like 'admin' for example)."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_BANK_REGISTER_USERNAME_REUSE,
    .hint = gettext_noop ("The client tried to register a new account with an username already in use."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_BANK_REGISTER_PAYTO_URI_REUSE,
    .hint = gettext_noop ("The client tried to register a new account with a payto:// URI already in use."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_BANK_ACCOUNT_BALANCE_NOT_ZERO,
    .hint = gettext_noop ("The client tried to delete an account with a non null balance."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_BANK_UNKNOWN_CREDITOR,
    .hint = gettext_noop ("The client tried to create a transaction or an operation that credit an unknown account."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_BANK_UNKNOWN_DEBTOR,
    .hint = gettext_noop ("The client tried to create a transaction or an operation that debit an unknown account."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_BANK_ACCOUNT_IS_EXCHANGE,
    .hint = gettext_noop ("The client tried to perform an action prohibited for exchange accounts."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_BANK_ACCOUNT_IS_NOT_EXCHANGE,
    .hint = gettext_noop ("The client tried to perform an action reserved for exchange accounts."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_BANK_BAD_CONVERSION,
    .hint = gettext_noop ("Received currency conversion is wrong."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_BANK_MISSING_TAN_INFO,
    .hint = gettext_noop ("The account referenced in this operation is missing tan info for the chosen channel."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_BANK_CONFIRM_INCOMPLETE,
    .hint = gettext_noop ("The client attempted to confirm a transaction with incomplete info."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_BANK_TAN_RATE_LIMITED,
    .hint = gettext_noop ("The request rate is too high. The server is refusing requests to guard against brute-force attacks."),
    .http_code = MHD_HTTP_TOO_MANY_REQUESTS
  },

  {
    .ec = TALER_EC_BANK_TAN_CHANNEL_NOT_SUPPORTED,
    .hint = gettext_noop ("This TAN channel is not supported."),
    .http_code = MHD_HTTP_NOT_IMPLEMENTED
  },

  {
    .ec = TALER_EC_BANK_TAN_CHANNEL_SCRIPT_FAILED,
    .hint = gettext_noop ("Failed to send TAN using the helper script. Either script is not found, or script timeout, or script terminated with a non-successful result."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_BANK_TAN_CHALLENGE_FAILED,
    .hint = gettext_noop ("The client's response to the challenge was invalid."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_BANK_NON_ADMIN_PATCH_LEGAL_NAME,
    .hint = gettext_noop ("A non-admin user has tried to change their legal name."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_BANK_NON_ADMIN_PATCH_DEBT_LIMIT,
    .hint = gettext_noop ("A non-admin user has tried to change their debt limit."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_BANK_NON_ADMIN_PATCH_MISSING_OLD_PASSWORD,
    .hint = gettext_noop ("A non-admin user has tried to change their password whihout providing the current one."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_BANK_PATCH_BAD_OLD_PASSWORD,
    .hint = gettext_noop ("Provided old password does not match current password."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_BANK_PATCH_ADMIN_EXCHANGE,
    .hint = gettext_noop ("An admin user has tried to become an exchange."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_BANK_NON_ADMIN_PATCH_CASHOUT,
    .hint = gettext_noop ("A non-admin user has tried to change their cashout account."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_BANK_NON_ADMIN_PATCH_CONTACT,
    .hint = gettext_noop ("A non-admin user has tried to change their contact info."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_BANK_ADMIN_CREDITOR,
    .hint = gettext_noop ("The client tried to create a transaction that credit the admin account."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_BANK_CHALLENGE_NOT_FOUND,
    .hint = gettext_noop ("The referenced challenge was not found."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_BANK_TAN_CHALLENGE_EXPIRED,
    .hint = gettext_noop ("The referenced challenge has expired."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_BANK_NON_ADMIN_SET_TAN_CHANNEL,
    .hint = gettext_noop ("A non-admin user has tried to create an account with 2fa."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_SYNC_ACCOUNT_UNKNOWN,
    .hint = gettext_noop ("The sync service failed find the account in its database."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_SYNC_BAD_IF_NONE_MATCH,
    .hint = gettext_noop ("The SHA-512 hash provided in the If-None-Match header is malformed."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_SYNC_BAD_IF_MATCH,
    .hint = gettext_noop ("The SHA-512 hash provided in the If-Match header is malformed or missing."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_SYNC_BAD_SYNC_SIGNATURE,
    .hint = gettext_noop ("The signature provided in the \"Sync-Signature\" header is malformed or missing."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_SYNC_INVALID_SIGNATURE,
    .hint = gettext_noop ("The signature provided in the \"Sync-Signature\" header does not match the account, old or new Etags."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_SYNC_MALFORMED_CONTENT_LENGTH,
    .hint = gettext_noop ("The \"Content-length\" field for the upload is not a number."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_SYNC_EXCESSIVE_CONTENT_LENGTH,
    .hint = gettext_noop ("The \"Content-length\" field for the upload is too big based on the server's terms of service."),
    .http_code = MHD_HTTP_CONTENT_TOO_LARGE
  },

  {
    .ec = TALER_EC_SYNC_OUT_OF_MEMORY_ON_CONTENT_LENGTH,
    .hint = gettext_noop ("The server is out of memory to handle the upload. Trying again later may succeed."),
    .http_code = MHD_HTTP_CONTENT_TOO_LARGE
  },

  {
    .ec = TALER_EC_SYNC_INVALID_UPLOAD,
    .hint = gettext_noop ("The uploaded data does not match the Etag."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_SYNC_PAYMENT_GENERIC_TIMEOUT,
    .hint = gettext_noop ("HTTP server experienced a timeout while awaiting promised payment."),
    .http_code = MHD_HTTP_REQUEST_TIMEOUT
  },

  {
    .ec = TALER_EC_SYNC_PAYMENT_CREATE_BACKEND_ERROR,
    .hint = gettext_noop ("Sync could not setup the payment request with its own backend."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_SYNC_PREVIOUS_BACKUP_UNKNOWN,
    .hint = gettext_noop ("The sync service failed find the backup to be updated in its database."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_SYNC_MISSING_CONTENT_LENGTH,
    .hint = gettext_noop ("The \"Content-length\" field for the upload is missing."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_SYNC_GENERIC_BACKEND_ERROR,
    .hint = gettext_noop ("Sync had problems communicating with its payment backend."),
    .http_code = MHD_HTTP_BAD_GATEWAY
  },

  {
    .ec = TALER_EC_SYNC_GENERIC_BACKEND_TIMEOUT,
    .hint = gettext_noop ("Sync experienced a timeout communicating with its payment backend."),
    .http_code = MHD_HTTP_GATEWAY_TIMEOUT
  },

  {
    .ec = TALER_EC_WALLET_EXCHANGE_PROTOCOL_VERSION_INCOMPATIBLE,
    .hint = gettext_noop ("The wallet does not implement a version of the exchange protocol that is compatible with the protocol version of the exchange."),
    .http_code = MHD_HTTP_NOT_IMPLEMENTED
  },

  {
    .ec = TALER_EC_WALLET_UNEXPECTED_EXCEPTION,
    .hint = gettext_noop ("The wallet encountered an unexpected exception.  This is likely a bug in the wallet implementation."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_WALLET_RECEIVED_MALFORMED_RESPONSE,
    .hint = gettext_noop ("The wallet received a response from a server, but the response can't be parsed."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_WALLET_NETWORK_ERROR,
    .hint = gettext_noop ("The wallet tried to make a network request, but it received no response."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_WALLET_HTTP_REQUEST_THROTTLED,
    .hint = gettext_noop ("The wallet tried to make a network request, but it was throttled."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_WALLET_UNEXPECTED_REQUEST_ERROR,
    .hint = gettext_noop ("The wallet made a request to a service, but received an error response it does not know how to handle."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_WALLET_EXCHANGE_DENOMINATIONS_INSUFFICIENT,
    .hint = gettext_noop ("The denominations offered by the exchange are insufficient.  Likely the exchange is badly configured or not maintained."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_WALLET_CORE_API_OPERATION_UNKNOWN,
    .hint = gettext_noop ("The wallet does not support the operation requested by a client."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_WALLET_INVALID_TALER_PAY_URI,
    .hint = gettext_noop ("The given taler://pay URI is invalid."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_WALLET_EXCHANGE_COIN_SIGNATURE_INVALID,
    .hint = gettext_noop ("The signature on a coin by the exchange's denomination key is invalid after unblinding it."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_WALLET_EXCHANGE_WITHDRAW_RESERVE_UNKNOWN_AT_EXCHANGE,
    .hint = gettext_noop ("The exchange does not know about the reserve (yet), and thus withdrawal can't progress."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_WALLET_CORE_NOT_AVAILABLE,
    .hint = gettext_noop ("The wallet core service is not available."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_WALLET_WITHDRAWAL_OPERATION_ABORTED_BY_BANK,
    .hint = gettext_noop ("The bank has aborted a withdrawal operation, and thus a withdrawal can't complete."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_WALLET_HTTP_REQUEST_GENERIC_TIMEOUT,
    .hint = gettext_noop ("An HTTP request made by the wallet timed out."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_WALLET_ORDER_ALREADY_CLAIMED,
    .hint = gettext_noop ("The order has already been claimed by another wallet."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_WALLET_WITHDRAWAL_GROUP_INCOMPLETE,
    .hint = gettext_noop ("A group of withdrawal operations (typically for the same reserve at the same exchange) has errors and will be tried again later."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_WALLET_REWARD_COIN_SIGNATURE_INVALID,
    .hint = gettext_noop ("The signature on a coin by the exchange's denomination key (obtained through the merchant via a reward) is invalid after unblinding it."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_WALLET_BANK_INTEGRATION_PROTOCOL_VERSION_INCOMPATIBLE,
    .hint = gettext_noop ("The wallet does not implement a version of the bank integration API that is compatible with the version offered by the bank."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_WALLET_CONTRACT_TERMS_BASE_URL_MISMATCH,
    .hint = gettext_noop ("The wallet processed a taler://pay URI, but the merchant base URL in the downloaded contract terms does not match the merchant base URL derived from the URI."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_WALLET_CONTRACT_TERMS_SIGNATURE_INVALID,
    .hint = gettext_noop ("The merchant's signature on the contract terms is invalid."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_WALLET_CONTRACT_TERMS_MALFORMED,
    .hint = gettext_noop ("The contract terms given by the merchant are malformed."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_WALLET_PENDING_OPERATION_FAILED,
    .hint = gettext_noop ("A pending operation failed, and thus the request can't be completed."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_WALLET_PAY_MERCHANT_SERVER_ERROR,
    .hint = gettext_noop ("A payment was attempted, but the merchant had an internal server error (5xx)."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_WALLET_CRYPTO_WORKER_ERROR,
    .hint = gettext_noop ("The crypto worker failed."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_WALLET_CRYPTO_WORKER_BAD_REQUEST,
    .hint = gettext_noop ("The crypto worker received a bad request."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_WALLET_WITHDRAWAL_KYC_REQUIRED,
    .hint = gettext_noop ("A KYC step is required before withdrawal can proceed."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_WALLET_DEPOSIT_GROUP_INSUFFICIENT_BALANCE,
    .hint = gettext_noop ("The wallet does not have sufficient balance to create a deposit group."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_WALLET_PEER_PUSH_PAYMENT_INSUFFICIENT_BALANCE,
    .hint = gettext_noop ("The wallet does not have sufficient balance to create a peer push payment."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_WALLET_PEER_PULL_PAYMENT_INSUFFICIENT_BALANCE,
    .hint = gettext_noop ("The wallet does not have sufficient balance to pay for an invoice."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_WALLET_REFRESH_GROUP_INCOMPLETE,
    .hint = gettext_noop ("A group of refresh operations has errors and will be tried again later."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_WALLET_EXCHANGE_BASE_URL_MISMATCH,
    .hint = gettext_noop ("The exchange's self-reported base URL does not match the one that the wallet is using."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_WALLET_ORDER_ALREADY_PAID,
    .hint = gettext_noop ("The order has already been paid by another wallet."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_WALLET_EXCHANGE_UNAVAILABLE,
    .hint = gettext_noop ("An exchange that is required for some request is currently not available."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_WALLET_EXCHANGE_ENTRY_USED,
    .hint = gettext_noop ("An exchange entry is still used by the exchange, thus it can't be deleted without purging."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_WALLET_DB_UNAVAILABLE,
    .hint = gettext_noop ("The wallet database is unavailable and the wallet thus is not operational."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_ANASTASIS_GENERIC_BACKEND_TIMEOUT,
    .hint = gettext_noop ("We encountered a timeout with our payment backend."),
    .http_code = MHD_HTTP_GATEWAY_TIMEOUT
  },

  {
    .ec = TALER_EC_ANASTASIS_GENERIC_INVALID_PAYMENT_REQUEST,
    .hint = gettext_noop ("The backend requested payment, but the request is malformed."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_ANASTASIS_GENERIC_BACKEND_ERROR,
    .hint = gettext_noop ("The backend got an unexpected reply from the payment processor."),
    .http_code = MHD_HTTP_BAD_GATEWAY
  },

  {
    .ec = TALER_EC_ANASTASIS_GENERIC_MISSING_CONTENT_LENGTH,
    .hint = gettext_noop ("The \"Content-length\" field for the upload is missing."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_ANASTASIS_GENERIC_MALFORMED_CONTENT_LENGTH,
    .hint = gettext_noop ("The \"Content-length\" field for the upload is malformed."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_ANASTASIS_GENERIC_ORDER_CREATE_BACKEND_ERROR,
    .hint = gettext_noop ("The backend failed to setup an order with the payment processor."),
    .http_code = MHD_HTTP_BAD_GATEWAY
  },

  {
    .ec = TALER_EC_ANASTASIS_GENERIC_PAYMENT_CHECK_UNAUTHORIZED,
    .hint = gettext_noop ("The backend was not authorized to check for payment with the payment processor."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_ANASTASIS_GENERIC_PAYMENT_CHECK_START_FAILED,
    .hint = gettext_noop ("The backend could not check payment status with the payment processor."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_ANASTASIS_GENERIC_PROVIDER_UNREACHABLE,
    .hint = gettext_noop ("The Anastasis provider could not be reached."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_ANASTASIS_PAYMENT_GENERIC_TIMEOUT,
    .hint = gettext_noop ("HTTP server experienced a timeout while awaiting promised payment."),
    .http_code = MHD_HTTP_REQUEST_TIMEOUT
  },

  {
    .ec = TALER_EC_ANASTASIS_TRUTH_UNKNOWN,
    .hint = gettext_noop ("The key share is unknown to the provider."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_ANASTASIS_TRUTH_AUTHORIZATION_METHOD_NO_LONGER_SUPPORTED,
    .hint = gettext_noop ("The authorization method used for the key share is no longer supported by the provider."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_ANASTASIS_TRUTH_CHALLENGE_RESPONSE_REQUIRED,
    .hint = gettext_noop ("The client needs to respond to the challenge."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_ANASTASIS_TRUTH_CHALLENGE_FAILED,
    .hint = gettext_noop ("The client's response to the challenge was invalid."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_ANASTASIS_TRUTH_CHALLENGE_UNKNOWN,
    .hint = gettext_noop ("The backend is not aware of having issued the provided challenge code. Either this is the wrong code, or it has expired."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_ANASTASIS_TRUTH_AUTHORIZATION_START_FAILED,
    .hint = gettext_noop ("The backend failed to initiate the authorization process."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_ANASTASIS_TRUTH_KEY_SHARE_GONE,
    .hint = gettext_noop ("The authorization succeeded, but the key share is no longer available."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_ANASTASIS_TRUTH_ORDER_DISAPPEARED,
    .hint = gettext_noop ("The backend forgot the order we asked the client to pay for"),
    .http_code = MHD_HTTP_BAD_GATEWAY
  },

  {
    .ec = TALER_EC_ANASTASIS_TRUTH_BACKEND_EXCHANGE_BAD,
    .hint = gettext_noop ("The backend itself reported a bad exchange interaction."),
    .http_code = MHD_HTTP_BAD_GATEWAY
  },

  {
    .ec = TALER_EC_ANASTASIS_TRUTH_UNEXPECTED_PAYMENT_STATUS,
    .hint = gettext_noop ("The backend reported a payment status we did not expect."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_ANASTASIS_TRUTH_PAYMENT_CREATE_BACKEND_ERROR,
    .hint = gettext_noop ("The backend failed to setup the order for payment."),
    .http_code = MHD_HTTP_BAD_GATEWAY
  },

  {
    .ec = TALER_EC_ANASTASIS_TRUTH_DECRYPTION_FAILED,
    .hint = gettext_noop ("The decryption of the key share failed with the provided key."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_ANASTASIS_TRUTH_RATE_LIMITED,
    .hint = gettext_noop ("The request rate is too high. The server is refusing requests to guard against brute-force attacks."),
    .http_code = MHD_HTTP_TOO_MANY_REQUESTS
  },

  {
    .ec = TALER_EC_ANASTASIS_TRUTH_CHALLENGE_WRONG_METHOD,
    .hint = gettext_noop ("A request to issue a challenge is not valid for this authentication method."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_ANASTASIS_TRUTH_UPLOAD_UUID_EXISTS,
    .hint = gettext_noop ("The backend failed to store the key share because the UUID is already in use."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_ANASTASIS_TRUTH_UPLOAD_METHOD_NOT_SUPPORTED,
    .hint = gettext_noop ("The backend failed to store the key share because the authorization method is not supported."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_ANASTASIS_SMS_PHONE_INVALID,
    .hint = gettext_noop ("The provided phone number is not an acceptable number."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_ANASTASIS_SMS_HELPER_EXEC_FAILED,
    .hint = gettext_noop ("Failed to run the SMS transmission helper process."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_ANASTASIS_SMS_HELPER_COMMAND_FAILED,
    .hint = gettext_noop ("Provider failed to send SMS. Helper terminated with a non-successful result."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_ANASTASIS_EMAIL_INVALID,
    .hint = gettext_noop ("The provided email address is not an acceptable address."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_ANASTASIS_EMAIL_HELPER_EXEC_FAILED,
    .hint = gettext_noop ("Failed to run the E-mail transmission helper process."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_ANASTASIS_EMAIL_HELPER_COMMAND_FAILED,
    .hint = gettext_noop ("Provider failed to send E-mail. Helper terminated with a non-successful result."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_ANASTASIS_POST_INVALID,
    .hint = gettext_noop ("The provided postal address is not an acceptable address."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_ANASTASIS_POST_HELPER_EXEC_FAILED,
    .hint = gettext_noop ("Failed to run the mail transmission helper process."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_ANASTASIS_POST_HELPER_COMMAND_FAILED,
    .hint = gettext_noop ("Provider failed to send mail. Helper terminated with a non-successful result."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_ANASTASIS_IBAN_INVALID,
    .hint = gettext_noop ("The provided IBAN address is not an acceptable IBAN."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_ANASTASIS_IBAN_MISSING_TRANSFER,
    .hint = gettext_noop ("The provider has not yet received the IBAN wire transfer authorizing the disclosure of the key share."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_ANASTASIS_TOTP_KEY_MISSING,
    .hint = gettext_noop ("The backend did not find a TOTP key in the data provided."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_ANASTASIS_TOTP_KEY_INVALID,
    .hint = gettext_noop ("The key provided does not satisfy the format restrictions for an Anastasis TOTP key."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_ANASTASIS_POLICY_BAD_IF_NONE_MATCH,
    .hint = gettext_noop ("The given if-none-match header is malformed."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_ANASTASIS_POLICY_OUT_OF_MEMORY_ON_CONTENT_LENGTH,
    .hint = gettext_noop ("The server is out of memory to handle the upload. Trying again later may succeed."),
    .http_code = MHD_HTTP_CONTENT_TOO_LARGE
  },

  {
    .ec = TALER_EC_ANASTASIS_POLICY_BAD_SIGNATURE,
    .hint = gettext_noop ("The signature provided in the \"Anastasis-Policy-Signature\" header is malformed or missing."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_ANASTASIS_POLICY_BAD_IF_MATCH,
    .hint = gettext_noop ("The given if-match header is malformed."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_ANASTASIS_POLICY_INVALID_UPLOAD,
    .hint = gettext_noop ("The uploaded data does not match the Etag."),
    .http_code = MHD_HTTP_BAD_REQUEST
  },

  {
    .ec = TALER_EC_ANASTASIS_POLICY_NOT_FOUND,
    .hint = gettext_noop ("The provider is unaware of the requested policy."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_ANASTASIS_REDUCER_ACTION_INVALID,
    .hint = gettext_noop ("The given action is invalid for the current state of the reducer."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_ANASTASIS_REDUCER_STATE_INVALID,
    .hint = gettext_noop ("The given state of the reducer is invalid."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_ANASTASIS_REDUCER_INPUT_INVALID,
    .hint = gettext_noop ("The given input to the reducer is invalid."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_ANASTASIS_REDUCER_AUTHENTICATION_METHOD_NOT_SUPPORTED,
    .hint = gettext_noop ("The selected authentication method does not work for the Anastasis provider."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_ANASTASIS_REDUCER_INPUT_INVALID_FOR_STATE,
    .hint = gettext_noop ("The given input and action do not work for the current state."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_ANASTASIS_REDUCER_BACKEND_FAILURE,
    .hint = gettext_noop ("We experienced an unexpected failure interacting with the backend."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_ANASTASIS_REDUCER_RESOURCE_MALFORMED,
    .hint = gettext_noop ("The contents of a resource file did not match our expectations."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_ANASTASIS_REDUCER_RESOURCE_MISSING,
    .hint = gettext_noop ("A required resource file is missing."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_ANASTASIS_REDUCER_INPUT_REGEX_FAILED,
    .hint = gettext_noop ("An input did not match the regular expression."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_ANASTASIS_REDUCER_INPUT_VALIDATION_FAILED,
    .hint = gettext_noop ("An input did not match the custom validation logic."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_ANASTASIS_REDUCER_POLICY_LOOKUP_FAILED,
    .hint = gettext_noop ("Our attempts to download the recovery document failed with all providers. Most likely the personal information you entered differs from the information you provided during the backup process and you should go back to the previous step. Alternatively, if you used a backup provider that is unknown to this application, you should add that provider manually."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_ANASTASIS_REDUCER_BACKUP_PROVIDER_FAILED,
    .hint = gettext_noop ("Anastasis provider reported a fatal failure."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_ANASTASIS_REDUCER_PROVIDER_CONFIG_FAILED,
    .hint = gettext_noop ("Anastasis provider failed to respond to the configuration request."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_ANASTASIS_REDUCER_POLICY_MALFORMED,
    .hint = gettext_noop ("The policy we downloaded is malformed. Must have been a client error while creating the backup."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_ANASTASIS_REDUCER_NETWORK_FAILED,
    .hint = gettext_noop ("We failed to obtain the policy, likely due to a network issue."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_ANASTASIS_REDUCER_SECRET_MALFORMED,
    .hint = gettext_noop ("The recovered secret did not match the required syntax."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_ANASTASIS_REDUCER_CHALLENGE_DATA_TOO_BIG,
    .hint = gettext_noop ("The challenge data provided is too large for the available providers."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_ANASTASIS_REDUCER_SECRET_TOO_BIG,
    .hint = gettext_noop ("The provided core secret is too large for some of the providers."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_ANASTASIS_REDUCER_PROVIDER_INVALID_CONFIG,
    .hint = gettext_noop ("The provider returned in invalid configuration."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_ANASTASIS_REDUCER_INTERNAL_ERROR,
    .hint = gettext_noop ("The reducer encountered an internal error, likely a bug that needs to be reported."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_ANASTASIS_REDUCER_PROVIDERS_ALREADY_SYNCED,
    .hint = gettext_noop ("The reducer already synchronized with all providers."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_DONAU_GENERIC_KEYS_MISSING,
    .hint = gettext_noop ("The donau failed to perform the operation as it could not find the private keys. This is a problem with the donau setup, not with the client's request."),
    .http_code = MHD_HTTP_SERVICE_UNAVAILABLE
  },

  {
    .ec = TALER_EC_LIBEUFIN_NEXUS_GENERIC_ERROR,
    .hint = gettext_noop ("A generic error happened in the LibEuFin nexus.  See the enclose details JSON for more information."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_LIBEUFIN_NEXUS_UNCAUGHT_EXCEPTION,
    .hint = gettext_noop ("An uncaught exception happened in the LibEuFin nexus service."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_LIBEUFIN_SANDBOX_GENERIC_ERROR,
    .hint = gettext_noop ("A generic error happened in the LibEuFin sandbox.  See the enclose details JSON for more information."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },

  {
    .ec = TALER_EC_LIBEUFIN_SANDBOX_UNCAUGHT_EXCEPTION,
    .hint = gettext_noop ("An uncaught exception happened in the LibEuFin sandbox service."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_TALDIR_METHOD_NOT_SUPPORTED,
    .hint = gettext_noop ("This validation method is not supported by the service."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_TALDIR_REGISTER_RATE_LIMITED,
    .hint = gettext_noop ("Number of allowed attempts for initiating a challenge exceeded."),
    .http_code = MHD_HTTP_TOO_MANY_REQUESTS
  },

  {
    .ec = TALER_EC_CHALLENGER_GENERIC_CLIENT_UNKNOWN,
    .hint = gettext_noop ("The client is unknown or unauthorized."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_CHALLENGER_GENERIC_CLIENT_FORBIDDEN_BAD_REDIRECT_URI,
    .hint = gettext_noop ("The client is not authorized to use the given redirect URI."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_CHALLENGER_HELPER_EXEC_FAILED,
    .hint = gettext_noop ("The service failed to execute its helper process to send the challenge."),
    .http_code = MHD_HTTP_INTERNAL_SERVER_ERROR
  },

  {
    .ec = TALER_EC_CHALLENGER_GRANT_UNKNOWN,
    .hint = gettext_noop ("The grant is unknown to the service (it could also have expired)."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_CHALLENGER_CLIENT_FORBIDDEN_BAD_CODE,
    .hint = gettext_noop ("The code given is not even well-formed."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_CHALLENGER_GENERIC_VALIDATION_UNKNOWN,
    .hint = gettext_noop ("The service is not aware of the referenced validation process."),
    .http_code = MHD_HTTP_NOT_FOUND
  },

  {
    .ec = TALER_EC_CHALLENGER_CLIENT_FORBIDDEN_INVALID_CODE,
    .hint = gettext_noop ("The code given is not valid."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_CHALLENGER_TOO_MANY_ATTEMPTS,
    .hint = gettext_noop ("Too many attempts have been made, validation is temporarily disabled for this address."),
    .http_code = MHD_HTTP_TOO_MANY_REQUESTS
  },

  {
    .ec = TALER_EC_CHALLENGER_INVALID_PIN,
    .hint = gettext_noop ("The PIN code provided is incorrect."),
    .http_code = MHD_HTTP_FORBIDDEN
  },

  {
    .ec = TALER_EC_CHALLENGER_MISSING_ADDRESS,
    .hint = gettext_noop ("The token cannot be valid as no address was ever provided by the client."),
    .http_code = MHD_HTTP_CONFLICT
  },

  {
    .ec = TALER_EC_END,
    .hint = gettext_noop ("End of error code range."),
    .http_code = MHD_HTTP_UNINITIALIZED
  },


};


/**
 * The length of @e code_hint_pairs.
 */
static const unsigned int code_hint_pairs_length = 571;



const char *
TALER_ErrorCode_get_hint (enum TALER_ErrorCode ec)
{
  unsigned int lower = 0;
  unsigned int upper = code_hint_pairs_length - 1;
  unsigned int mid = upper / 2;
  while (lower <= upper)
  {
    mid = (upper + lower) / 2;
    if (code_hint_pairs[mid].ec < ec)
    {
      lower = mid + 1;
    }
    else if (code_hint_pairs[mid].ec > ec)
    {
      upper = mid - 1;
    }
    else
    {
      return code_hint_pairs[mid].hint;
    }
  }
  return "<no hint found>";
}


unsigned int
TALER_ErrorCode_get_http_status (enum TALER_ErrorCode ec)
{
  unsigned int lower = 0;
  unsigned int upper = code_hint_pairs_length - 1;
  unsigned int mid = upper / 2;
  while (lower <= upper)
  {
    mid = (upper + lower) / 2;
    if (code_hint_pairs[mid].ec < ec)
    {
      lower = mid + 1;
    }
    else if (code_hint_pairs[mid].ec > ec)
    {
      upper = mid - 1;
    }
    else
    {
      return code_hint_pairs[mid].http_code;
    }
  }
  return UINT_MAX;
}


unsigned int
TALER_ErrorCode_get_http_status_safe (enum TALER_ErrorCode ec)
{
  unsigned int hc;

  hc = TALER_ErrorCode_get_http_status (ec);
  if ( (0 == hc) ||
       (UINT_MAX == hc) )
    return MHD_HTTP_INTERNAL_SERVER_ERROR;
  return hc;
}
