/*
  This file is part of TALER
  Copyright (C) 2014-2024 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify it under the
  terms of the GNU General Public License as published by the Free Software
  Foundation; either version 3, or (at your option) any later version.

  TALER is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
  A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

  You should have received a copy of the GNU General Public License along with
  TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
*/
/**
 * @file include/taler_auditordb_plugin.h
 * @brief Low-level (statement-level) database access for the auditor
 * @author Florian Dold
 * @author Christian Grothoff
 */
#ifndef TALER_AUDITORDB_PLUGIN_H
#define TALER_AUDITORDB_PLUGIN_H

#include <jansson.h>
#include <gnunet/gnunet_util_lib.h>
#include <gnunet/gnunet_db_lib.h>
#include "taler_util.h"
#include "taler_auditordb_lib.h"
#include "taler_signatures.h"


/**
 * Function called with the results of select_historic_denom_revenue()
 *
 * @param cls closure
 * @param denom_pub_hash hash of the denomination key
 * @param revenue_timestamp when did this profit get realized
 * @param revenue_balance what was the total profit made from
 *                        deposit fees, melting fees, refresh fees
 *                        and coins that were never returned?
 * @param loss_balance what was the total loss
 * @return sets the return value of select_denomination_info(),
 *         #GNUNET_OK to continue,
 *         #GNUNET_NO to stop processing further rows
 *         #GNUNET_SYSERR or other values on error.
 */
typedef enum GNUNET_GenericReturnValue
(*TALER_AUDITORDB_HistoricDenominationRevenueDataCallback)(
  void *cls,
  const struct TALER_DenominationHashP *denom_pub_hash,
  struct GNUNET_TIME_Timestamp revenue_timestamp,
  const struct TALER_Amount *revenue_balance,
  const struct TALER_Amount *loss_balance);


/**
 * Function called with the results of select_historic_reserve_revenue()
 *
 * @param cls closure
 * @param start_time beginning of aggregated time interval
 * @param end_time end of aggregated time interval
 * @param reserve_profits total profits made
 *
 * @return sets the return value of select_denomination_info(),
 *         #GNUNET_OK to continue,
 *         #GNUNET_NO to stop processing further rows
 *         #GNUNET_SYSERR or other values on error.
 */
typedef enum GNUNET_GenericReturnValue
(*TALER_AUDITORDB_HistoricReserveRevenueDataCallback)(
  void *cls,
  struct GNUNET_TIME_Timestamp start_time,
  struct GNUNET_TIME_Timestamp end_time,
  const struct TALER_Amount *reserve_profits);


/**
 * Information about a signing key of an exchange.
 */
struct TALER_AUDITORDB_ExchangeSigningKey
{

  /**
   * When does @e exchange_pub start to be used?
   */
  struct GNUNET_TIME_Timestamp ep_start;

  /**
   * When will the exchange stop signing with @e exchange_pub?
   */
  struct GNUNET_TIME_Timestamp ep_expire;

  /**
   * When does the signing key expire (for legal disputes)?
   */
  struct GNUNET_TIME_Timestamp ep_end;

  /**
   * What is the public offline signing key this is all about?
   */
  struct TALER_ExchangePublicKeyP exchange_pub;

  /**
   * Signature by the offline master key affirming the above.
   */
  struct TALER_MasterSignatureP master_sig;
};


/**
 * Information about a deposit confirmation we received from
 * a merchant.
 */
struct TALER_AUDITORDB_DepositConfirmation
{

  /**
   * Hash over the contract for which this deposit is made.
   */
  struct TALER_PrivateContractHashP h_contract_terms;

  /**
   * Hash over the policy extension for the deposit.
   */
  struct TALER_ExtensionPolicyHashP h_policy;

  /**
   * Hash over the wiring information of the merchant.
   */
  struct TALER_MerchantWireHashP h_wire;

  /**
   * Time when this deposit confirmation was generated by the exchange.
   */
  struct GNUNET_TIME_Timestamp exchange_timestamp;

  /**
   * How much time does the @e merchant have to issue a refund
   * request?  Zero if refunds are not allowed.  After this time, the
   * coin cannot be refunded.  Note that the wire transfer will not be
   * performed by the exchange until the refund deadline.  This value
   * is taken from the original deposit request.
   */
  struct GNUNET_TIME_Timestamp refund_deadline;

  /**
   * How much time does the @e exchange have to wire the funds?
   */
  struct GNUNET_TIME_Timestamp wire_deadline;

  /**
   * Amount to be deposited, excluding fee.  Calculated from the
   * amount with fee and the fee from the deposit request.
   */
  struct TALER_Amount total_without_fee;

  /**
   * Length of the @e coin_pubs and @e coin_sigs arrays.
   */
  unsigned int num_coins;

  /**
   * Array of the coin public keys involved in the
   * batch deposit operation.
   */
  const struct TALER_CoinSpendPublicKeyP *coin_pubs;

  /**
   * Array of coin deposit signatures from the deposit operation.
   */
  const struct TALER_CoinSpendSignatureP *coin_sigs;

  /**
   * The Merchant's public key.  Allows the merchant to later refund
   * the transaction or to inquire about the wire transfer identifier.
   */
  struct TALER_MerchantPublicKeyP merchant;

  /**
   * Signature from the exchange of type
   * #TALER_SIGNATURE_EXCHANGE_CONFIRM_DEPOSIT.
   */
  struct TALER_ExchangeSignatureP exchange_sig;

  /**
   * Public signing key from the exchange matching @e exchange_sig.
   */
  struct TALER_ExchangePublicKeyP exchange_pub;

  /**
   * Exchange master signature over @e exchange_sig.
   */
  struct TALER_MasterSignatureP master_sig;

};


/**
 * Balance values for a reserve (or all reserves).
 */
struct TALER_AUDITORDB_ReserveFeeBalance
{
  /**
   * Remaining funds.
   */
  struct TALER_Amount reserve_balance;

  /**
   * Losses from operations that should not have
   * happened (e.g. negative balance).
   */
  struct TALER_Amount reserve_loss;

  /**
   * Fees charged for withdraw.
   */
  struct TALER_Amount withdraw_fee_balance;

  /**
   * Fees charged for closing.
   */
  struct TALER_Amount close_fee_balance;

  /**
   * Fees charged for purse creation.
   */
  struct TALER_Amount purse_fee_balance;

  /**
   * Opening fees charged.
   */
  struct TALER_Amount open_fee_balance;

  /**
   * History fees charged.
   */
  struct TALER_Amount history_fee_balance;
};


/**
 * Balance data for denominations in circulation.
 */
struct TALER_AUDITORDB_DenominationCirculationData
{
  /**
   * Amount of outstanding coins in circulation.
   */
  struct TALER_Amount denom_balance;

  /**
   * Amount lost due coins illicitly accepted (effectively, a
   * negative @a denom_balance).
   */
  struct TALER_Amount denom_loss;

  /**
   * Total amount that could still be theoretically lost in the future due to
   * recoup operations.  (Total put into circulation minus @e recoup_loss).
   */
  struct TALER_Amount denom_risk;

  /**
   * Amount lost due to recoups.
   */
  struct TALER_Amount recoup_loss;

  /**
   * Number of coins of this denomination that the exchange signed into
   * existence.
   */
  uint64_t num_issued;
};


/**
 * Function called with deposit confirmations stored in
 * the auditor's database.
 *
 * @param cls closure
 * @param serial_id location of the @a dc in the database
 * @param dc the deposit confirmation itself
 * @return #GNUNET_OK to continue to iterate, #GNUNET_SYSERR to stop iterating
 */
typedef enum GNUNET_GenericReturnValue
(*TALER_AUDITORDB_DepositConfirmationCallback)(
  void *cls,
  uint64_t serial_id,
  const struct TALER_AUDITORDB_DepositConfirmation *dc);


/**
 * Function called on deposits that are past their due date
 * and have not yet seen a wire transfer.
 *
 * @param cls closure
 * @param batch_deposit_serial_id where in the table are we
 * @param total_amount value of all missing deposits, including fees
 * @param wire_target_h_payto hash of the recipient account's payto URI
 * @param deadline what was the earliest requested wire transfer deadline
 */
typedef void
(*TALER_AUDITORDB_WireMissingCallback)(
  void *cls,
  uint64_t batch_deposit_serial_id,
  const struct TALER_Amount *total_amount,
  const struct TALER_PaytoHashP *wire_target_h_payto,
  struct GNUNET_TIME_Timestamp deadline);


/**
 * Function called on expired purses.
 *
 * @param cls closure
 * @param purse_pub public key of the purse
 * @param balance amount of money in the purse
 * @param expiration_date when did the purse expire?
 * @return #GNUNET_OK to continue to iterate
 */
typedef enum GNUNET_GenericReturnValue
(*TALER_AUDITORDB_ExpiredPurseCallback)(
  void *cls,
  const struct TALER_PurseContractPublicKeyP *purse_pub,
  const struct TALER_Amount *balance,
  struct GNUNET_TIME_Timestamp expiration_date);


/**
 * @brief The plugin API, returned from the plugin's "init" function.
 * The argument given to "init" is simply a configuration handle.
 *
 * Functions starting with "get_" return one result, functions starting
 * with "select_" return multiple results via callbacks.
 */
struct TALER_AUDITORDB_Plugin
{

  /**
   * Closure for all callbacks.
   */
  void *cls;

  /**
   * Name of the library which generated this plugin.  Set by the
   * plugin loader.
   */
  char *library_name;

  /**
   * Fully connect to the db if the connection does not exist yet
   * and check that there is no transaction currently running.
   *
   * @param cls the @e cls of this struct with the plugin-specific state
   * @return #GNUNET_OK on success
   *         #GNUNET_NO if we rolled back an earlier transaction
   *         #GNUNET_SYSERR if we have no DB connection
   */
  enum GNUNET_GenericReturnValue
  (*preflight)(void *cls);


  /**
   * Register callback to be invoked on events of type @a es.
   *
   * @param cls database context to use
   * @param es specification of the event to listen for
   * @param timeout how long to wait for the event
   * @param cb function to call when the event happens, possibly
   *         mulrewardle times (until cancel is invoked)
   * @param cb_cls closure for @a cb
   * @return handle useful to cancel the listener
   */
  struct GNUNET_DB_EventHandler *
  (*event_listen)(void *cls,
                  const struct GNUNET_DB_EventHeaderP *es,
                  struct GNUNET_TIME_Relative timeout,
                  GNUNET_DB_EventCallback cb,
                  void *cb_cls);

  /**
   * Stop notifications.
   *
   * @param eh handle to unregister.
   */
  void
  (*event_listen_cancel)(struct GNUNET_DB_EventHandler *eh);


  /**
   * Notify all that listen on @a es of an event.
   *
   * @param cls database context to use
   * @param es specification of the event to generate
   * @param extra additional event data provided
   * @param extra_size number of bytes in @a extra
   */
  void
  (*event_notify)(void *cls,
                  const struct GNUNET_DB_EventHeaderP *es,
                  const void *extra,
                  size_t extra_size);


  /**
   * Drop all auditor tables OR deletes recoverable auditor state.
   * This should only be used by testcases or when restarting the
   * auditor from scratch.
   *
   * @param cls the `struct PostgresClosure` with the plugin-specific state
   * @param drop_exchangelist drop all tables, including schema versioning
   *        and the exchange and deposit_confirmations table; NOT to be
   *        used when restarting the auditor
   * @return #GNUNET_OK upon success; #GNUNET_SYSERR upon failure
   */
  enum GNUNET_GenericReturnValue
  (*drop_tables)(void *cls,
                 bool drop_exchangelist);


  /**
   * Create the necessary tables if they are not present
   *
   * @param cls the @e cls of this struct with the plugin-specific state
   * @param support_partitions true to support partitioning
   * @param num_partitions number of partitions to use
   * @return #GNUNET_OK upon success; #GNUNET_SYSERR upon failure
   */
  enum GNUNET_GenericReturnValue
  (*create_tables)(void *cls,
                   bool support_partitions,
                   uint32_t num_partitions);


  /**
   * Start a transaction.
   *
   * @param cls the @e cls of this struct with the plugin-specific state
   * @return #GNUNET_OK on success
   */
  enum GNUNET_GenericReturnValue
  (*start)(void *cls);


  /**
   * Commit a transaction.
   *
   * @param cls the @e cls of this struct with the plugin-specific state
   * @return transaction status code
   */
  enum GNUNET_DB_QueryStatus
  (*commit)(void *cls);


  /**
   * Abort/rollback a transaction.
   *
   * @param cls the @e cls of this struct with the plugin-specific state
   */
  void
  (*rollback) (void *cls);


  /**
   * Function called to perform "garbage collection" on the
   * database, expiring records we no longer require.
   *
   * @param cls closure
   * @return #GNUNET_OK on success,
   *         #GNUNET_SYSERR on DB errors
   */
  enum GNUNET_GenericReturnValue
  (*gc)(void *cls);


  /**
   * Insert information about the auditor's progress with an exchange's
   * data.
   *
   * @param cls the @e cls of this struct with the plugin-specific state
   * @param progress_key name of the progress indicator
   * @param progress_offset offset until which we have made progress
   * @param ... NULL terminated list of additional key-value pairs to insert
   * @return transaction status code
   */
  enum GNUNET_DB_QueryStatus
  (*insert_auditor_progress)(
    void *cls,
    const char *progress_key,
    uint64_t progress_offset,
    ...);


  /**
   * Update information about the progress of the auditor.  There
   * must be an existing record for the exchange.
   *
   * @param cls the @e cls of this struct with the plugin-specific state
   * @param progress_key name of the progress indicator
   * @param progress_offset offset until which we have made progress
   * @param ... NULL terminated list of additional key-value pairs to update
   * @return transaction status code
   */
  enum GNUNET_DB_QueryStatus
  (*update_auditor_progress)(
    void *cls,
    const char *progress_key,
    uint64_t progress_offset,
    ...);


  /**
   * Get information about the progress of the auditor.
   *
   * @param cls the @e cls of this struct with the plugin-specific state
   * @param progress_key name of the progress indicator
   * @param[out] progress_offset set to offset until which we have made progress, set to 0 if key was not found
   * @param ... NULL terminated list of additional key-value pairs to fetch
   * @return transaction status code
   */
  enum GNUNET_DB_QueryStatus
  (*get_auditor_progress)(void *cls,
                          const char *progress_key,
                          uint64_t *progress_offset,
                          ...);


  /**
   * Insert information about a balance tracked by the auditor. There must not be an
   * existing record.
   *
   * @param cls the @e cls of this struct with the plugin-specific state
   * @param balance_key key of the balance to store
   * @param balance_value value to store
   * @param ... NULL terminated list of additional key-value pairs to insert
   * @return transaction status code
   */
  enum GNUNET_DB_QueryStatus
  (*insert_balance)(void *cls,
                    const char *balance_key,
                    const struct TALER_Amount *balance_value,
                    ...);


  /**
   * Insert information about a balance tracked by the auditor.  Destructively updates an
   * existing record, which must already exist.
   *
   * @param cls the @e cls of this struct with the plugin-specific state
   * @param balance_key key of the balance to store
   * @param balance_amount value to store
   * @param ... NULL terminated list of additional key-value pairs to update
   * @return transaction status code
   */
  enum GNUNET_DB_QueryStatus
  (*update_balance)(void *cls,
                    const char *balance_key,
                    const struct TALER_Amount *balance_amount,
                    ...);


  /**
   * Get summary information about balance tracked by the auditor.
   *
   * @param cls the @e cls of this struct with the plugin-specific state
   * @param balance_key key of the balance to store
   * @param[out] balance_value set to amount stored under @a balance_key, set to invalid amount (all zero) if key was not found
   * @param ... NULL terminated list of additional key-value pairs to fetch
   * @return transaction status code
   */
  enum GNUNET_DB_QueryStatus
  (*get_balance)(void *cls,
                 const char *balance_key,
                 struct TALER_Amount *balance_value,
                 ...);


  /**
   * Insert information about a signing key of the exchange.
   *
   * @param cls the @e cls of this struct with the plugin-specific state
   * @param sk signing key information to store
   * @return query result status
   */
  enum GNUNET_DB_QueryStatus
  (*insert_exchange_signkey)(
    void *cls,
    const struct TALER_AUDITORDB_ExchangeSigningKey *sk);


  /**
   * Insert information about a deposit confirmation into the database.
   *
   * @param cls the @e cls of this struct with the plugin-specific state
   * @param dc deposit confirmation information to store
   * @return query result status
   */
  enum GNUNET_DB_QueryStatus
  (*insert_deposit_confirmation)(
    void *cls,
    const struct TALER_AUDITORDB_DepositConfirmation *dc);


  /**
   * Get information about deposit confirmations from the database.
   *
   * @param cls the @e cls of this struct with the plugin-specific state
   * @param start_id row/serial ID where to start the iteration (0 from
   *                  the start, exclusive, i.e. serial_ids must start from 1)
   * @param return_suppressed should suppressed rows be returned anyway?
   * @param cb function to call with results
   * @param cb_cls closure for @a cb
   * @return query result status
   */
  enum GNUNET_DB_QueryStatus
  (*get_deposit_confirmations)(
    void *cls,
    uint64_t start_id,
    bool return_suppressed,
    TALER_AUDITORDB_DepositConfirmationCallback cb,
    void *cb_cls);


  /**
   * Delete information about a deposit confirmation from the database.
   *
   * @param cls the @e cls of this struct with the plugin-specific state
   * @param row_id row to delete
   * @return query result status
   */
  enum GNUNET_DB_QueryStatus
  (*delete_deposit_confirmation)(
    void *cls,
    uint64_t row_id);


  /**
   * Insert information about a reserve.  There must not be an
   * existing record for the reserve.
   *
   * @param cls the @e cls of this struct with the plugin-specific state
   * @param reserve_pub public key of the reserve
   * @param rfb balance amounts for the reserve
   * @param expiration_date expiration date of the reserve
   * @param origin_account where did the money in the reserve originally come from
   * @return transaction status code
   */
  enum GNUNET_DB_QueryStatus
  (*insert_reserve_info)(
    void *cls,
    const struct TALER_ReservePublicKeyP *reserve_pub,
    const struct TALER_AUDITORDB_ReserveFeeBalance *rfb,
    struct GNUNET_TIME_Timestamp expiration_date,
    const char *origin_account);


  /**
   * Update information about a reserve.  Destructively updates an
   * existing record, which must already exist.
   *
   * @param cls the @e cls of this struct with the plugin-specific state
   * @param reserve_pub public key of the reserve
   * @param rfb balance amounts for the reserve
   * @param expiration_date expiration date of the reserve
   * @return transaction status code
   */
  enum GNUNET_DB_QueryStatus
  (*update_reserve_info)(
    void *cls,
    const struct TALER_ReservePublicKeyP *reserve_pub,
    const struct TALER_AUDITORDB_ReserveFeeBalance *rfb,
    struct GNUNET_TIME_Timestamp expiration_date);


  /**
   * Get information about a reserve.
   *
   * @param cls the @e cls of this struct with the plugin-specific state
   * @param reserve_pub public key of the reserve
   * @param[out] rowid which row did we get the information from
   * @param[out] rfb set to balances associated with the reserve
   * @param[out] expiration_date expiration date of the reserve
   * @param[out] sender_account from where did the money in the reserve originally come from
   * @return transaction status code
   */
  enum GNUNET_DB_QueryStatus
  (*get_reserve_info)(
    void *cls,
    const struct TALER_ReservePublicKeyP *reserve_pub,
    uint64_t *rowid,
    struct TALER_AUDITORDB_ReserveFeeBalance *rfb,
    struct GNUNET_TIME_Timestamp *expiration_date,
    char **sender_account);


  /**
   * Delete information about a reserve.
   *
   * @param cls the @e cls of this struct with the plugin-specific state
   * @param reserve_pub public key of the reserve
   * @return transaction status code
   */
  enum GNUNET_DB_QueryStatus
  (*del_reserve_info)(void *cls,
                      const struct TALER_ReservePublicKeyP *reserve_pub);


  /**
   * Insert new row into the pending deposits table.
   *
   * @param cls the @e cls of this struct with the plugin-specific state
   * @param batch_deposit_serial_id where in the table are we
   * @param total_amount value of all missing deposits, including fees
   * @param wire_target_h_payto hash of the recipient account's payto URI
   * @param deadline what was the requested wire transfer deadline
   * @return transaction status code
   */
  enum GNUNET_DB_QueryStatus
  (*insert_pending_deposit)(
    void *cls,
    uint64_t batch_deposit_serial_id,
    const struct TALER_PaytoHashP *wire_target_h_payto,
    const struct TALER_Amount *total_amount,
    struct GNUNET_TIME_Timestamp deadline);


  /**
   * Delete a row from the pending deposit table.
   * Usually done when the respective wire transfer
   * was finally detected.
   *
   * @param cls the @e cls of this struct with the plugin-specific state
   * @param batch_deposit_serial_id which entry to delete
   * @return transaction status code
   */
  enum GNUNET_DB_QueryStatus
  (*delete_pending_deposit)(
    void *cls,
    uint64_t batch_deposit_serial_id);


  /**
   * Return (batch) deposits for which we have not yet
   * seen the required wire transfer.
   *
   * @param deadline only return up to this deadline
   * @param cb function to call on each entry
   * @param cb_cls closure for @a cb
   * @return transaction status code
   */
  enum GNUNET_DB_QueryStatus
  (*select_pending_deposits)(
    void *cls,
    struct GNUNET_TIME_Absolute deadline,
    TALER_AUDITORDB_WireMissingCallback cb,
    void *cb_cls);


  /**
   * Insert information about a purse.  There must not be an
   * existing record for the purse.
   *
   * @param cls the @e cls of this struct with the plugin-specific state
   * @param purse_pub public key of the purse
   * @param balance balance of the purse
   * @param expiration_date expiration date of the purse
   * @return transaction status code
   */
  enum GNUNET_DB_QueryStatus
  (*insert_purse_info)(
    void *cls,
    const struct TALER_PurseContractPublicKeyP *purse_pub,
    const struct TALER_Amount *balance,
    struct GNUNET_TIME_Timestamp expiration_date);


  /**
   * Update information about a purse.  Destructively updates an
   * existing record, which must already exist.
   *
   * @param cls the @e cls of this struct with the plugin-specific state
   * @param purse_pub public key of the purse
   * @param balance new balance for the purse
   * @return transaction status code
   */
  enum GNUNET_DB_QueryStatus
  (*update_purse_info)(
    void *cls,
    const struct TALER_PurseContractPublicKeyP *purse_pub,
    const struct TALER_Amount *balance);


  /**
   * Get information about a purse.
   *
   * @param cls the @e cls of this struct with the plugin-specific state
   * @param purse_pub public key of the purse
   * @param[out] rowid which row did we get the information from
   * @param[out] balance set to balance of the purse
   * @param[out] expiration_date expiration date of the purse
   * @return transaction status code
   */
  enum GNUNET_DB_QueryStatus
  (*get_purse_info)(
    void *cls,
    const struct TALER_PurseContractPublicKeyP *purse_pub,
    uint64_t *rowid,
    struct TALER_Amount *balance,
    struct GNUNET_TIME_Timestamp *expiration_date);


  /**
   * Delete information about a purse.
   *
   * @param cls the @e cls of this struct with the plugin-specific state
   * @param purse_pub public key of the reserve
   * @return transaction status code
   */
  enum GNUNET_DB_QueryStatus
  (*delete_purse_info)(
    void *cls,
    const struct TALER_PurseContractPublicKeyP *purse_pub);


  /**
   * Get information about expired purses.
   *
   * @param cls the @e cls of this struct with the plugin-specific state
   * @param cb function to call on expired purses
   * @param cb_cls closure for @a cb
   * @return transaction status code
   */
  enum GNUNET_DB_QueryStatus
  (*select_purse_expired)(
    void *cls,
    TALER_AUDITORDB_ExpiredPurseCallback cb,
    void *cb_cls);


  /**
   * Insert information about a denomination key's balances.  There
   * must not be an existing record for the denomination key.
   *
   * @param cls the @e cls of this struct with the plugin-specific state
   * @param denom_pub_hash hash of the denomination public key
   * @param dcd denomination circulation data to store
   * @return transaction status code
   */
  enum GNUNET_DB_QueryStatus
  (*insert_denomination_balance)(
    void *cls,
    const struct TALER_DenominationHashP *denom_pub_hash,
    const struct TALER_AUDITORDB_DenominationCirculationData *dcd);


  /**
   * Update information about a denomination key's balances.  There
   * must be an existing record for the denomination key.
   *
   * @param cls the @e cls of this struct with the plugin-specific state
   * @param denom_pub_hash hash of the denomination public key
   * @param dcd denomination circulation data to store
   * @return transaction status code
   */
  enum GNUNET_DB_QueryStatus
  (*update_denomination_balance)(
    void *cls,
    const struct TALER_DenominationHashP *denom_pub_hash,
    const struct TALER_AUDITORDB_DenominationCirculationData *dcd);

  /**
   * Delete information about a denomination key's balances.
   *
   * @param cls the @e cls of this struct with the plugin-specific state
   * @param denom_pub_hash hash of the denomination public key
   * @return transaction status code
   */
  enum GNUNET_DB_QueryStatus
  (*del_denomination_balance)(
    void *cls,
    const struct TALER_DenominationHashP *denom_pub_hash);


  /**
   * Get information about a denomination key's balances.
   *
   * @param cls the @e cls of this struct with the plugin-specific state
   * @param denom_pub_hash hash of the denomination public key
   * @param[out] dcd denomination circulation data to initialize
   * @return transaction status code
   */
  enum GNUNET_DB_QueryStatus
  (*get_denomination_balance)(
    void *cls,
    const struct TALER_DenominationHashP *denom_pub_hash,
    struct TALER_AUDITORDB_DenominationCirculationData *dcd);


  /**
   * Insert information about an exchange's historic
   * revenue about a denomination key.
   *
   * @param cls the @e cls of this struct with the plugin-specific state
   * @param denom_pub_hash hash of the denomination key
   * @param revenue_timestamp when did this profit get realized
   * @param revenue_balance what was the total profit made from
   *                        deposit fees, melting fees, refresh fees
   *                        and coins that were never returned?
   * @param recoup_loss_balance total losses from recoups of revoked denominations
   * @return transaction status code
   */
  enum GNUNET_DB_QueryStatus
  (*insert_historic_denom_revenue)(
    void *cls,
    const struct TALER_DenominationHashP *denom_pub_hash,
    struct GNUNET_TIME_Timestamp revenue_timestamp,
    const struct TALER_Amount *revenue_balance,
    const struct TALER_Amount *recoup_loss_balance);


  /**
   * Obtain all of the historic denomination key revenue.
   *
   * @param cls the @e cls of this struct with the plugin-specific state
   * @param cb function to call with the results
   * @param cb_cls closure for @a cb
   * @return transaction status code
   */
  enum GNUNET_DB_QueryStatus
  (*select_historic_denom_revenue)(
    void *cls,
    TALER_AUDITORDB_HistoricDenominationRevenueDataCallback cb,
    void *cb_cls);


  /**
   * Insert information about an exchange's historic revenue from reserves.
   *
   * @param cls the @e cls of this struct with the plugin-specific state
   * @param start_time beginning of aggregated time interval
   * @param end_time end of aggregated time interval
   * @param reserve_profits total profits made
   * @return transaction status code
   */
  enum GNUNET_DB_QueryStatus
  (*insert_historic_reserve_revenue)(
    void *cls,
    struct GNUNET_TIME_Timestamp start_time,
    struct GNUNET_TIME_Timestamp end_time,
    const struct TALER_Amount *reserve_profits);


  /**
   * Return information about an exchange's historic revenue from reserves.
   *
   * @param cls the @e cls of this struct with the plugin-specific state
   * @param cb function to call with results
   * @param cb_cls closure for @a cb
   * @return transaction status code
   */
  enum GNUNET_DB_QueryStatus
  (*select_historic_reserve_revenue)(
    void *cls,
    TALER_AUDITORDB_HistoricReserveRevenueDataCallback cb,
    void *cb_cls);

};


#endif /* _TALER_AUDITOR_DB_H */
