BEGIN;
SELECT _v.register_patch('auditor-0002', NULL, NULL);
SET search_path TO auditor;
DO $$ BEGIN
    CREATE TYPE taler_amount
      AS
      (val INT8
      ,frac INT4
      );
    COMMENT ON TYPE taler_amount
      IS 'Stores an amount, fraction is in units of 1/100000000 of the base value';
EXCEPTION
    WHEN duplicate_object THEN null;
END $$;
CREATE TABLE auditor_amount_arithmetic_inconsistency
(
    row_id BIGINT GENERATED BY DEFAULT AS IDENTITY UNIQUE PRIMARY KEY,
    operation BYTEA,
    exchange_amount taler_amount,
    auditor_amount taler_amount,
    profitable BOOLEAN
);
COMMENT ON TABLE auditor_amount_arithmetic_inconsistency
    IS 'Report a (serious) inconsistency in the exchange''s database with respect to calculations involving amounts';
CREATE TABLE IF NOT EXISTS auditor_balances
(
   balance_key TEXT PRIMARY KEY NOT NULL
  ,balance_value taler_amount
);
COMMENT
ON TABLE auditor_balances
  IS 'table storing various global balances of the auditor';
COMMENT
ON COLUMN auditor_balances.balance_key
 IS 'unique name for the balance value';
COMMENT
ON COLUMN auditor_balances.balance_value
 IS 'balance amount';
CREATE TABLE auditor_denomination_pending
  (denom_pub_hash BYTEA PRIMARY KEY CHECK (LENGTH(denom_pub_hash)=64)
  ,denom_balance taler_amount NOT NULL
  ,denom_loss taler_amount NOT NULL
  ,num_issued INT8 NOT NULL
  ,denom_risk taler_amount NOT NULL
  ,recoup_loss taler_amount NOT NULL
);
COMMENT ON TABLE auditor_denomination_pending
  IS 'outstanding denomination coins that the exchange is aware of and what the respective balances are (outstanding as well as issued overall which implies the maximum value at risk).';
COMMENT ON COLUMN auditor_denomination_pending.num_issued
  IS 'counts the number of coins issued (withdraw, refresh) of this denomination';
COMMENT ON COLUMN auditor_denomination_pending.denom_risk
  IS 'amount that could theoretically be lost in the future due to recoup operations';
COMMENT ON COLUMN auditor_denomination_pending.denom_loss
  IS 'amount that was lost due to failures by the exchange';
COMMENT ON COLUMN auditor_denomination_pending.recoup_loss
  IS 'amount actually lost due to recoup operations after a revocation';
CREATE TABLE auditor_exchange_signkeys
  (exchange_pub BYTEA PRIMARY KEY CHECK (LENGTH(exchange_pub)=32)
  ,master_sig BYTEA NOT NULL CHECK (LENGTH(master_sig)=64)
  ,ep_valid_from INT8 NOT NULL
  ,ep_expire_sign INT8 NOT NULL
  ,ep_expire_legal INT8 NOT NULL
  );
COMMENT ON TABLE auditor_exchange_signkeys
  IS 'list of the online signing keys of exchanges we are auditing';
COMMENT ON COLUMN auditor_exchange_signkeys.exchange_pub
  IS 'Public online signing key of the exchange.';
COMMENT ON COLUMN auditor_exchange_signkeys.master_sig
  IS 'Signature affirming the validity of the signing key of purpose TALER_SIGNATURE_MASTER_SIGNING_KEY_VALIDITY.';
COMMENT ON COLUMN auditor_exchange_signkeys.ep_valid_from
  IS 'Time when this online signing key will first be used to sign messages.';
COMMENT ON COLUMN auditor_exchange_signkeys.ep_expire_sign
  IS 'Time when this online signing key will no longer be used to sign.';
COMMENT ON COLUMN auditor_exchange_signkeys.ep_expire_legal
  IS 'Time when this online signing key legally expires.';
CREATE TABLE auditor_historic_denomination_revenue
  (denom_pub_hash BYTEA PRIMARY KEY CHECK (LENGTH(denom_pub_hash)=64)
  ,revenue_timestamp INT8 NOT NULL
  ,revenue_balance taler_amount NOT NULL
  ,loss_balance taler_amount NOT NULL
  );
COMMENT ON TABLE auditor_historic_denomination_revenue
  IS 'Table with historic profits; basically, when a denom_pub has expired and everything associated with it is garbage collected, the final profits end up in here; note that the denom_pub here is not a foreign key, we just keep it as a reference point.';
COMMENT ON COLUMN auditor_historic_denomination_revenue.denom_pub_hash
  IS 'hash of the denomination public key that created this revenue';
COMMENT ON COLUMN auditor_historic_denomination_revenue.revenue_timestamp
  IS 'when was this revenue realized (by the denomination public key expiring)';
COMMENT ON COLUMN auditor_historic_denomination_revenue.revenue_balance
  IS 'the sum of all of the profits we made on the denomination except for withdraw fees (which are in historic_reserve_revenue); so this includes the deposit, melt and refund fees';
COMMENT ON COLUMN auditor_historic_denomination_revenue.loss_balance
  IS 'the sum of all of the losses we made on the denomination (for example, because the signing key was compromised and thus we redeemed coins we never issued); of course should be zero in practice in most cases';
SET search_path TO auditor;
CREATE TABLE IF NOT EXISTS auditor_historic_reserve_summary
  (start_date INT8 PRIMARY KEY
  ,end_date INT8 NOT NULL
  ,reserve_profits taler_amount NOT NULL
  );
COMMENT ON TABLE auditor_historic_reserve_summary
  IS 'historic profits from reserves; we eventually GC auditor_historic_reserve_revenue, and then store the totals in here (by time intervals).';
COMMENT ON COLUMN auditor_historic_reserve_summary.start_date
  IS 'start date of the time interval over which we made these profits from reserves';
COMMENT ON COLUMN auditor_historic_reserve_summary.end_date
  IS 'end date (exclusive) of the time interval over which we made these profits from reserves';
COMMENT ON COLUMN auditor_historic_reserve_summary.reserve_profits
  IS 'total amount in profits made';
CREATE TABLE IF NOT EXISTS auditor_progress
  (progress_key TEXT PRIMARY KEY NOT NULL
  ,progress_offset INT8 NOT NULL
  );
COMMENT ON TABLE auditor_progress
  IS 'Information about to the point until which the audit has progressed.  Used for SELECTing the statements to process.';
COMMENT ON COLUMN auditor_progress.progress_key
  IS 'Name of the progress indicator';
COMMENT ON COLUMN auditor_progress.progress_offset
  IS 'Table offset or timestamp or counter until which the audit has progressed';
CREATE TABLE auditor_purses
  (auditor_purses_rowid BIGINT GENERATED BY DEFAULT AS IDENTITY UNIQUE
  ,purse_pub BYTEA PRIMARY KEY CHECK(LENGTH(purse_pub)=32)
  ,balance taler_amount NOT NULL DEFAULT(0,0)
  ,target taler_amount NOT NULL
  ,expiration_date INT8 NOT NULL
  );
COMMENT ON TABLE auditor_purses
  IS 'all of the purses and their respective balances that the auditor is aware of';
CREATE TABLE auditor_reserves
  (auditor_reserves_rowid BIGINT GENERATED BY DEFAULT AS IDENTITY UNIQUE
  ,reserve_pub BYTEA PRIMARY KEY CHECK(LENGTH(reserve_pub)=32)
  ,reserve_balance taler_amount NOT NULL
  ,reserve_loss taler_amount NOT NULL
  ,withdraw_fee_balance taler_amount NOT NULL
  ,close_fee_balance taler_amount NOT NULL
  ,purse_fee_balance taler_amount NOT NULL
  ,open_fee_balance taler_amount NOT NULL
  ,history_fee_balance taler_amount NOT NULL
  ,expiration_date INT8 NOT NULL
  ,origin_account TEXT
  );
COMMENT ON TABLE auditor_reserves
  IS 'all of the customer reserves and their respective balances that the auditor is aware of';
CREATE TABLE auditor_deposit_confirmations
(deposit_confirmation_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY UNIQUE
    ,h_contract_terms BYTEA NOT NULL CHECK (LENGTH(h_contract_terms)=64)
    ,h_policy BYTEA NOT NULL CHECK (LENGTH(h_policy)=64)
    ,h_wire BYTEA NOT NULL CHECK (LENGTH(h_wire)=64)
    ,exchange_timestamp INT8 NOT NULL
    ,refund_deadline INT8 NOT NULL
    ,wire_deadline INT8 NOT NULL
    ,total_without_fee taler_amount NOT NULL
    ,coin_pubs BYTEA[] NOT NULL CHECK (CARDINALITY(coin_pubs)>0)
    ,coin_sigs BYTEA[] NOT NULL CHECK (CARDINALITY(coin_sigs)=CARDINALITY(coin_pubs))
    ,merchant_pub BYTEA NOT NULL CHECK (LENGTH(merchant_pub)=32)
    ,exchange_sig BYTEA NOT NULL CHECK (LENGTH(exchange_sig)=64)
    ,exchange_pub BYTEA NOT NULL CHECK (LENGTH(exchange_pub)=32)
    ,master_sig BYTEA NOT NULL CHECK (LENGTH(master_sig)=64)
    ,suppressed BOOLEAN NOT NULL DEFAULT FALSE
    ,ancient BOOLEAN NOT NULL DEFAULT FALSE
    ,PRIMARY KEY (h_contract_terms,h_wire,merchant_pub,exchange_sig,exchange_pub,master_sig)
    );
COMMENT ON TABLE auditor_deposit_confirmations
  IS 'deposit confirmation sent to us by merchants; we must check that the exchange reported these properly.';
CREATE INDEX IF NOT EXISTS auditor_deposit_confirmations_not_ancient
    ON auditor_deposit_confirmations
    (exchange_timestamp ASC)
    WHERE NOT ancient;
CREATE OR REPLACE FUNCTION auditor_new_transactions_trigger()
    RETURNS trigger
    LANGUAGE plpgsql
AS $$
BEGIN
    NOTIFY XX81AFHF88YGN6ESNH39KR5VQE9MHD7GSSNMTCXB82SZ6T99ARHE0;
    RETURN NEW;
END $$;
COMMENT ON FUNCTION auditor_new_transactions_trigger()
    IS 'Call auditor_call_db_notify on new entry';
CREATE TRIGGER auditor_notify_helper_deposits
    AFTER INSERT
    ON auditor.auditor_deposit_confirmations
EXECUTE PROCEDURE auditor_new_transactions_trigger();
COMMIT;
