--
-- This file is part of TALER
-- Copyright (C) 2014--2022 Taler Systems SA
--
-- TALER is free software; you can redistribute it and/or modify it under the
-- terms of the GNU General Public License as published by the Free Software
-- Foundation; either version 3, or (at your option) any later version.
--
-- TALER is distributed in the hope that it will be useful, but WITHOUT ANY
-- WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
-- A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License along with
-- TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
--

CREATE FUNCTION create_table_refresh_revealed_coins(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'refresh_revealed_coins';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(rrc_serial BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',melt_serial_id INT8 NOT NULL'
      ',freshcoin_index INT4 NOT NULL'
      ',link_sig BYTEA NOT NULL CHECK(LENGTH(link_sig)=64)'
      ',denominations_serial INT8 NOT NULL'
      ',coin_ev BYTEA NOT NULL'
      ',h_coin_ev BYTEA NOT NULL CHECK(LENGTH(h_coin_ev)=64)'
      ',ev_sig BYTEA NOT NULL'
      ',ewv BYTEA NOT NULL'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (melt_serial_id)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'Revelations about the new coins that are to be created during a melting session.'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'needed for exchange-auditor replication logic'
    ,'rrc_serial'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Identifies the refresh commitment (rc) of the melt operation.'
    ,'melt_serial_id'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'index of the fresh coin being created (one melt operation may result in multiple fresh coins)'
    ,'freshcoin_index'
    ,table_name
    ,partition_suffix
  );
  --
  -- FIXME: Add comment for link_sig
  --
  PERFORM comment_partitioned_column(
     'envelope of the new coin to be signed'
    ,'coin_ev'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'exchange contributed values in the creation of the fresh coin (see /csr)'
    ,'ewv'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'hash of the envelope of the new coin to be signed (for lookups)'
    ,'h_coin_ev'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'exchange signature over the envelope'
    ,'ev_sig'
    ,table_name
    ,partition_suffix
  );
END
$$;


CREATE FUNCTION constrain_table_refresh_revealed_coins(
  IN partition_suffix TEXT
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'refresh_revealed_coins';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_coins_by_melt_serial_id_index '
    'ON ' || table_name || ' '
    '(melt_serial_id);'
  );
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_rrc_serial_key'
    ' UNIQUE (rrc_serial) '
    ',ADD CONSTRAINT ' || table_name || '_coin_ev_key'
    ' UNIQUE (coin_ev) '
    ',ADD CONSTRAINT ' || table_name || '_h_coin_ev_key'
    ' UNIQUE (h_coin_ev) '
    ',ADD PRIMARY KEY (melt_serial_id, freshcoin_index)'
  );
END
$$;


CREATE FUNCTION foreign_table_refresh_revealed_coins()
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'refresh_revealed_coins';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_melt'
    ' FOREIGN KEY (melt_serial_id)'
    ' REFERENCES refresh_commitments (melt_serial_id) ON DELETE CASCADE'
    ',ADD CONSTRAINT ' || table_name || '_foreign_denom'
    ' FOREIGN KEY (denominations_serial)'
    ' REFERENCES denominations (denominations_serial) ON DELETE CASCADE'
  );
END
$$;


INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('refresh_revealed_coins'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('refresh_revealed_coins'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('refresh_revealed_coins'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE);
