--
-- This file is part of TALER
-- Copyright (C) 2014--2022 Taler Systems SA
--
-- TALER is free software; you can redistribute it and/or modify it under the
-- terms of the GNU General Public License as published by the Free Software
-- Foundation; either version 3, or (at your option) any later version.
--
-- TALER is distributed in the hope that it will be useful, but WITHOUT ANY
-- WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
-- A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License along with
-- TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
--

CREATE FUNCTION create_table_legitimization_requirements(
  IN partition_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(legitimization_requirement_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',h_payto BYTEA NOT NULL CHECK (LENGTH(h_payto)=32)'
      ',required_checks VARCHAR NOT NULL'
      ',UNIQUE (h_payto, required_checks)'
    ') %s ;'
    ,'legitimization_requirements'
    ,'PARTITION BY HASH (h_payto)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'List of required legitimizations by account'
    ,'legitimization_requirements'
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'unique ID for this legitimization requirement at the exchange'
    ,'legitimization_requirement_serial_id'
    ,'legitimization_requirements'
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'foreign key linking the entry to the wire_targets table, NOT a primary key (multiple legitimizations are possible per wire target)'
    ,'h_payto'
    ,'legitimization_requirements'
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'space-separated list of required checks'
    ,'required_checks'
    ,'legitimization_requirements'
    ,partition_suffix
  );
END
$$;

-- We need a separate function for this, as we call create_table only once but need to add
-- those constraints to each partition which gets created
CREATE FUNCTION constrain_table_legitimization_requirements(
  IN partition_suffix VARCHAR
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  partition_name VARCHAR;
BEGIN
  partition_name = concat_ws('_', 'legitimization_requirements', partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || partition_name || ' '
    'ADD CONSTRAINT ' || partition_name || '_serial_id_key '
    'UNIQUE (legitimization_requirement_serial_id)');
END
$$;


INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('legitimization_requirements'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('legitimization_requirements'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE);
